package org.lsst.ccs.subsystem.power;

import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;
import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.subsystem.power.config.Power;
import org.lsst.ccs.subsystem.power.data.PowerChanState;
import org.lsst.ccs.subsystem.power.data.PowerState;
import org.lsst.ccs.subsystem.power.constants.PowerAgentProperties;

/**
 *  Implements the power supply control subsystem.
 *
 *  *** Needs work if to be used again ***
 * 
 *  @author Owen Saxton
 */
public class MainPower extends Subsystem implements HasLifecycle {

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private final Map<String, PowerDeviceGroup> pwrGroups = new LinkedHashMap<>();
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService agentPropertiesService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private ConfigurationService agentConfigurationService;

    public static final String BROADCAST_TASK = "publish-data";
    private long broadcastMillis = 10000;

    private static final Logger LOG = Logger.getLogger(MainPower.class.getName());


    /**
     *  Constructor
     */
    public MainPower() {
        super("power-supply", AgentInfo.AgentType.WORKER);
    }


    /**
     *  Power subsystem initialization
     */
    @Override
    public void build() {

        AgentPeriodicTask periodicTask;

        //Create & schedule an AgentPeriodicTask to broadcast the power state
        periodicTask = new AgentPeriodicTask(BROADCAST_TASK, () -> periodicBroadcast()).withPeriod(Duration.ofMillis(broadcastMillis));
        periodicTaskService.scheduleAgentPeriodicTask(periodicTask);
    }


    /**
     *  Power subsystem post-initialization
     */
    @Override
    public void postInit()
    {
        //Set a property to define that this Agent is a Power Supply controller.
        agentPropertiesService.setAgentProperty(PowerAgentProperties.POWER_AGENT, getClass().getCanonicalName());
        
        // Sets up power device groups
        if (pwrGroups.isEmpty()) {
            LOG.severe("No power devices specified");
        }
    }


    /**
     *  Starts the subsystem.
     */
    @Override
    public void postStart()
    {
        LOG.info("Power subsystem started");
        publishState();        // For any GUIs
    }


    /**
     *  Performs periodic power data broadcast.
     */
    public void periodicBroadcast()
    {
        publishState();
    }

    /**
     *  Sets the tick period
     */
    private void setTickPeriod(long period)
    {
        periodicTaskService.setPeriodicTaskPeriod(BROADCAST_TASK, Duration.ofMillis(period));
    }
    
    private long getTickPeriod()
    {
        return periodicTaskService.getPeriodicTaskPeriod(BROADCAST_TASK).toMillis();
    }
    
    /**
     *  Sets the update (tick) period.
     *
     *  @param  period  The update period (milliseconds) to set.
     */
    @Command(type=CommandType.ACTION, description="Set the update period")
    public void setUpdatePeriod(int period)
    {
        setTickPeriod(period);
        publishState();    // Must do this for GUI
    }


    /**
     *  Saves the configuration data.
     *
     *  @param  name  The configuration name
     *  @throws IOException
     */
    @Command(type=CommandType.ACTION, description="Save the current configuration")
    public void saveNamedConfig(String name) throws IOException
    {
        agentConfigurationService.saveChangesForCategories(Power.POWER + ":" + name);
    }


    /**
     *  Gets the configuration data.
     *
     *  @return  The array of power configuration objects
     */
    @Command(type=CommandType.QUERY, description="Get the configuration data")
    public Power[] getPowerConfig()
    {
        Power[] power = new Power[pwrGroups.size()];
        int j = 0;
        for (PowerDeviceGroup pGroup : pwrGroups.values()) {
            power[j++] = pGroup.getPowerConfig();
        }

        return power;
    }


    /**
     *  Sets the configuration data.
     *
     *  @param  power  The array of power configuration objects
     *  @throws  Exception
     */
    @Command(type=CommandType.ACTION, description="Set the configuration data")
    public void setPowerConfig(Power[] power) throws Exception
    {
        for (Power pConf : power) {
            PowerDeviceGroup pGroup = pwrGroups.get(pConf.getName());
            pGroup.setPowerConfig(pConf);
        }
    }


    /**
     *  Gets the power state.
     *
     *  @return  The power state
     *  @throws  Exception
     */
    @Command(type=CommandType.QUERY, description="Get the power state")
    public List<PowerChanState> getPowerState() throws Exception
    {
        List<PowerChanState> pState = new ArrayList<>();
        for (PowerDeviceGroup pGroup : pwrGroups.values()) {
            pState.addAll(pGroup.getPowerState());
        }

        return pState;
    }


    /**
     *  Turns on the power.
     *
     *  @throws  Exception
     */
    @Command(type=CommandType.ACTION, description="Turn on the power")
    public void powerOn() throws Exception
    {
        for (PowerDeviceGroup pGroup : pwrGroups.values()) {
            pGroup.powerOn();
        }
    }


    /**
     *  Turns off the power.
     *
     *  @throws  Exception
     */
    @Command(type=CommandType.ACTION, description="Turn off the power")
    public void powerOff() throws Exception
    {
        for (PowerDeviceGroup pGroup : pwrGroups.values()) {
            pGroup.powerOff();
        }
    }


    /**
     *  Gets the full power state.
     *
     *  @return  The full power state
     */
    @Command(type=CommandType.QUERY, description="Get the full state")
    public PowerState getFullState()
    {
        List<PowerChanState> pState = new ArrayList<>();
        for (PowerDeviceGroup pGroup : pwrGroups.values()) {
            pState.addAll(pGroup.getPowerState());
        }
        return new PowerState((int)getTickPeriod(), pState);
    }


    /**
     *  Publishes the state of the power device.
     *
     *  This is intended to be called whenever any element of the state is
     *  changed.
     */
    private void publishState()
    {
        KeyValueData kvd = new KeyValueData(PowerState.KEY, getFullState());
        publishSubsystemDataOnStatusBus(kvd);
    }    

}
