package org.lsst.ccs.subsystem.power;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.TreeMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.utilities.conv.InputConversionEngine;

/**
 * A class to control the Reb Power Supplies wiring via build configuration properties.
 * 
 * @author The LSST CCS Team
 */
public class RebPowerSupplyWiring {
    
    private static final Logger LOG = Logger.getLogger(RebPowerSupplyWiring.class.getName());

    private final Map<String, PowerSupplyConfiguration> powerSuppliesMap = new TreeMap();
    private final Map<String, PowerSupplyConfiguration> rebsToPowerSupplyMap = new TreeMap();
    private final Map<String, String> rebsToPowerSupplyChannelMap = new TreeMap();
    private final Map<String, String> rebHeaterToPowerSupplyChannelMap = new TreeMap();
    
    public final static String PROPERTIES_PREFIX = "org.lsst.ccs.power.";
    
    public RebPowerSupplyWiring(Properties buildProperties) {
        
        List<String> powerSuppliesList = (List)InputConversionEngine.convertArgToType(buildProperties.getProperty(PROPERTIES_PREFIX+"supplies"), List.class);        
        if ( powerSuppliesList.isEmpty() ) {
            LOG.log(Level.WARNING, "No power supplies have been provided in build configuration "+PROPERTIES_PREFIX+"supplies");            
        }
        
        for ( String powerSupply : powerSuppliesList ) {
            PowerSupplyConfiguration powerSupplyConfiguration = new PowerSupplyConfiguration(powerSupply, buildProperties);
            powerSuppliesMap.put(powerSupply, powerSupplyConfiguration);
            
            int count = 0;
            for ( String reb: powerSupplyConfiguration.getRebs() ) {
                if (! reb.isEmpty() ) {                    
                    if ( reb.endsWith("Heater") ) {
                        String rebName = reb.replace("Heater", "");
                        rebHeaterToPowerSupplyChannelMap.put(rebName, String.valueOf(count));                        
                    } else {
                        if (rebsToPowerSupplyMap.containsKey(reb)) {
                            throw new RuntimeException("REB " + reb + " cannot be added to power supply " + powerSupply + " as it has already been assigned to power supply " + rebsToPowerSupplyMap.get(reb));
                        }
                        rebsToPowerSupplyMap.put(reb, powerSupplyConfiguration);
                        rebsToPowerSupplyChannelMap.put(reb, String.valueOf(count));
                    }
                }
                count++;
            }
        }
    }
    
    public Collection<PowerSupplyConfiguration> getRebPowerSuppyConfigurations() {
        return powerSuppliesMap.values();
    }
    
    public Set<String> getRebsList() {
        return rebsToPowerSupplyMap.keySet();
    }
    
    public PowerSupplyConfiguration getPowerSupplyConfigurationForReb(String reb) {
        return rebsToPowerSupplyMap.get(reb);
    }

    public int getPowerSupplyChannelForReb(String reb) {
        return Integer.valueOf(rebsToPowerSupplyChannelMap.get(reb));
    }
    public int getPowerSupplyChannelForRebHeater(String reb) {
        if ( ! rebHeaterToPowerSupplyChannelMap.containsKey(reb) ) {
            return -1;
        }
        return Integer.valueOf(rebHeaterToPowerSupplyChannelMap.get(reb));
    }
    
}
