package org.lsst.ccs.subsystem.power.ui;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.*;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.SwingConstants;
import javax.swing.border.BevelBorder;
import org.lsst.ccs.bus.data.AgentCategory;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.gconsole.base.Const;
import org.lsst.ccs.gconsole.base.filter.AbstractChannelsFilter;
import org.lsst.ccs.gconsole.plugins.monitor.AbstractMonitorView;
import org.lsst.ccs.gconsole.plugins.monitor.DisplayChannel;
import org.lsst.ccs.gconsole.plugins.monitor.FormattedValue;
import org.lsst.ccs.gconsole.plugins.monitor.MonitorField;
import org.lsst.ccs.gconsole.plugins.monitor.MonitorFormat;
import org.lsst.ccs.gconsole.plugins.monitor.Updatable;
import org.lsst.ccs.gconsole.services.aggregator.AgentChannel;
import org.lsst.ccs.gconsole.services.command.CommandService;
import org.lsst.ccs.gconsole.services.optpage.OptionalPage;
import org.lsst.ccs.subsystem.common.focalplane.data.HasFocalPlaneData;
import org.lsst.ccs.subsystem.common.ui.focalplane.Segment;

/**
 * Reb power control panel as described in LSSTCCSPOWER-61 (stub).
 *
 * @author onoprien
 */
public final class RebPowerControlPanel extends JPanel {

// -- Fields : -----------------------------------------------------------------
        
    static public final Color COLOR_UNKNOWN = Color.WHITE;
    static public final Color COLOR_ON = Color.GREEN;
    static public final Color COLOR_OFF = Color.RED;
    static public final Color COLOR_TRIPPED = Color.MAGENTA;
    static public final Color COLOR_OFFLINE = MonitorFormat.COLOR_OFF;
    
    static public final String PATH_MAIN_STATE = "RebPowerState";
    static public final String PATH_HV_STATE = "RebHvBiasState";
    static public final String PATH_DPHI_STATE = "RebDPhiState";
    static public final String PATH_HV_VALUE = "hvBias";
    static public final String PATH_DPHI_VALUE = "dphi";
    static public final String PATH_HV_V = "hvbias/VbefSwch";
    static public final String PATH_HV_I = "hvbias/IbefSwch";
    
    
    private final AgentInfo agent;
    
    private View view; // MonitorView used to receive status updates
    
    private Map<String,DisplayChannel> localData;
    private TreeSet<String> scienceRafts, cornerRafts;


// -- Life cycle : -------------------------------------------------------------
    
    public RebPowerControlPanel(AgentInfo agent) {
        this.agent = agent;
        setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
        setBorder(BorderFactory.createEmptyBorder(Const.VSPACE, Const.HSPACE, Const.VSPACE, Const.HSPACE));
    }
    
    private void opened() {
        view = new View();
        view.install();
    }
    
    private void closed() {
        if (view != null) {
            view.uninstall();
            view = null;
        }
    }
    
    private void rebuild() {
        
        // Destroy previous structure
        
        removeAll();
        
        // Fiind present rafts
        
        scienceRafts = new TreeSet<>();
        cornerRafts = new TreeSet<>();
        for (String path : localData.keySet()) {
            String raft = path.substring(0, Segment.RAFT.toString().length()+2);
            if (raft.contains("00") || raft.contains("04") || raft.contains("40") || raft.contains("44")) {
                cornerRafts.add(raft);
            } else {
                scienceRafts.add(raft);
            }
            if (cornerRafts.size() == 4 && scienceRafts.size() == (5*5-4)) break;
        }
        
        // Create new components
        
        if (!scienceRafts.isEmpty()) {
            JLabel title = new JLabel("Science RAFTs");
            Font f = title.getFont();
            f = new Font(f.getName(), Font.BOLD, f.getSize()+2);
            title.setFont(f);
            add(title);
            add(Box.createRigidArea(Const.VDIM));
            add(new ScienceTable());
            add(Box.createRigidArea(Const.VDIM));
        }
        if (!cornerRafts.isEmpty()) {
            add(Box.createRigidArea(Const.VDIM));
            JLabel title = new JLabel("Corner RAFTs");
            Font f = title.getFont();
            f = new Font(f.getName(), Font.BOLD, f.getSize()+2);
            title.setFont(f);
            add(title);
            add(Box.createRigidArea(Const.VDIM));
            add(new CornerTable());
            add(Box.createRigidArea(Const.VDIM));
        }
        add(Box.createVerticalGlue());
    }
    
    /**
     * It seems logical to have one plugin per subsystem. At the moment, there seem to be several
     * plugins in subsystem-power at various stages of abandonment, not sure which one I should use.
     * Once the package structure is streamlined, this code should go into the plugin.
     * For now, it's in jas/RebPowerControlPlugin.
     */
//    @Plugin(name = "Reb Power Control plugin", description = "Displays Reb power state including HV bias and DPhi")
//    static public class RebPowerControlPanelPlugin extends ConsolePlugin {
//        @Override
//        public void initialize() {
//            OptionalPage.getService().add(new RebPowerControlPanel.OptPage());
//        }
//    }
    
    static public class OptPage implements OptionalPage {

        private final HashMap<String,RebPowerControlPanel> panels = new HashMap<>(2, .75f);
        
        @Override
        public boolean isAutoOpen() {
            return false;
        }

        @Override
        public String getPage(AgentInfo agent) {
            if ( agent.getAgentProperty(AgentCategory.AGENT_CATEGORY_PROPERTY, "").equals(AgentCategory.POWER.name()) 
                    && !agent.getAgentProperty(HasFocalPlaneData.AGENT_PROPERTY, "").isEmpty() ) {
                return "CCS Subsystems/"+agent.getName() +"/Reb Power Control";
            }
            return null;
        }

        @Override
        public JComponent open(AgentInfo agent, JComponent existingComponent) {
            String path = getPage(agent);
            if (path == null) return null;
            RebPowerControlPanel panel = panels.get(path);
            if (panel == null) {
                panel = new RebPowerControlPanel(agent);
                panels.put(path, panel);
            }
            return new JScrollPane(panel);
        }

        @Override
        public void opened(String path) {
            RebPowerControlPanel panel = panels.get(path);
            if (panel != null) {
                panel.opened();
            }
        }

        @Override
        public void closed(String path) {
            RebPowerControlPanel panel = panels.remove(path);
            if (panel != null) {
                panel.closed();
            }
        }

    }
    
    
// -- Monitor View class : -----------------------------------------------------
    
    private final class View extends AbstractMonitorView {
        
        View() {
            setFilter(new Filter());
        }

        @Override
        protected void resetChannels() {
            if (localData != null) {
                localData.values().forEach(channel -> channel.setTarget(null));
            }
            localData = new HashMap(data);
            rebuild();
        }

        @Override
        public JComponent getPanel() {
            return null; // NEVER
        }

        @Override
        public Descriptor getDescriptor() {
            return null; // NEVER
        }
        
    }
    
    private class Filter extends AbstractChannelsFilter {
        
        private final String prefixState = agent.getName() +"/state/";
        private final String prefixConf = agent.getName() +"/configuration/";
        private final String prefixTrend = agent.getName() +"/";
        private final int prefixLengthState = prefixState.length();
        private final int prefixLengthConf = prefixConf.length();
        private final int prefixLengthTrend = prefixTrend.length();

        @Override
        public List<String> getAgents() {
            return Collections.singletonList(agent.getName());
        }

        @Override
        public List<String> getOriginChannels() {
            ArrayList<String> templates = new ArrayList<>(2);
            templates.add(prefixState);
            templates.add(prefixConf);
            templates.add(".+/"+ PATH_HV_V);
            templates.add(".+/"+ PATH_HV_I);
            return templates;
        }

        @Override
        public String getOriginPath(String displayPath) {
            String prefix;
            if ( displayPath.endsWith("State")) {
                prefix = prefixState;
            } else if (displayPath.contains("/")) {
                prefix = prefixTrend;
            } else {
                prefix = prefixConf;
            }
            return prefix + displayPath;
        }

        @Override
        public String getDisplayPath(String originPath) {
            if (originPath.endsWith(PATH_HV_V) || originPath.endsWith(PATH_HV_I)) {
                return originPath.substring(prefixLengthTrend);
            } else if (originPath.endsWith(PATH_MAIN_STATE) || originPath.endsWith(PATH_HV_STATE) || originPath.endsWith(PATH_DPHI_STATE)) {
                return originPath.substring(prefixLengthState);
            } else if (originPath.endsWith(PATH_HV_VALUE) || originPath.endsWith(PATH_DPHI_VALUE)) {
                return originPath.substring(prefixLengthConf);
            }
            return null;
        }
        
    }
    
    
// -- Graphical components : ---------------------------------------------------
    
    private class ScienceTable extends JPanel {
        
        ScienceTable() {
            setBorder(BorderFactory.createEtchedBorder());
            GridBagLayout gridbag = new GridBagLayout();
            setLayout(gridbag);

            GridBagConstraints c = new GridBagConstraints();
            c.insets = new Insets(Const.VSPACE, Const.HSPACE, Const.VSPACE, Const.HSPACE);
            c.anchor = GridBagConstraints.CENTER;
            c.gridx = 1;
            c.gridy = 0;
            JLabel label = new JLabel("REB 0");
            gridbag.setConstraints(label, c);
            add(label);
            c.gridx = 2;
            label = new JLabel("REB 1");
            gridbag.setConstraints(label, c);
            add(label);
            c.gridx = 3;
            label = new JLabel("REB 2");
            gridbag.setConstraints(label, c);
            add(label);

            GridBagConstraints cRaft = new GridBagConstraints();
            cRaft.insets = new Insets(Const.VSPACE, Const.HSPACE, Const.VSPACE, Const.HSPACE);
            cRaft.anchor = GridBagConstraints.CENTER;
            cRaft.gridx = 0;
            cRaft.gridy = 0;
            c = new GridBagConstraints();
            c.anchor = GridBagConstraints.CENTER;
            c.fill = GridBagConstraints.BOTH;
            c.weightx = 1.0;
            c.weighty = 1.0;
            for (String raft : scienceRafts) {
                cRaft.gridy++;
                label = new JLabel(raft);
                gridbag.setConstraints(label, cRaft);
                add(label);
                c.gridy = cRaft.gridy;
                for (int iReb=0; iReb<3; iReb++) {
                    c.gridx = iReb+1;
                    String reb = Segment.REB.toString() + iReb;
                    String prefix = raft +"/"+ reb +"/";
                    RebPanel p = new RebPanel(prefix);
                    gridbag.setConstraints(p, c);
                    add(p);
                }
            }
        }

        @Override
        public Dimension getMaximumSize() {
            return getPreferredSize();
        }
        
    }
    
    private class CornerTable extends JPanel {
        
        CornerTable() {
            setBorder(BorderFactory.createEtchedBorder());
            GridBagLayout gridbag = new GridBagLayout();
            setLayout(gridbag);

            GridBagConstraints c = new GridBagConstraints();
            c.insets = new Insets(Const.VSPACE, Const.HSPACE, Const.VSPACE, Const.HSPACE);
            c.anchor = GridBagConstraints.CENTER;
            c.gridx = 1;
            c.gridy = 0;
            JLabel label = new JLabel("Wavefront REB");
            gridbag.setConstraints(label, c);
            add(label);
            c.gridx = 2;
            label = new JLabel("Guider REB");
            gridbag.setConstraints(label, c);
            add(label);

            GridBagConstraints cRaft = new GridBagConstraints();
            cRaft.insets = new Insets(Const.VSPACE, Const.HSPACE, Const.VSPACE, Const.HSPACE);
            cRaft.anchor = GridBagConstraints.CENTER;
            cRaft.gridx = 0;
            cRaft.gridy = 0;
            c = new GridBagConstraints();
            c.anchor = GridBagConstraints.CENTER;
            c.fill = GridBagConstraints.BOTH;
            c.weightx = 1.0;
            c.weighty = 1.0;
            for (String raft : cornerRafts) {
                cRaft.gridy++;
                label = new JLabel(raft);
                gridbag.setConstraints(label, cRaft);
                add(label);
                c.gridy = cRaft.gridy;
                c.gridx = 1;
                String prefix = raft + "/" + Segment.REB.toString() + "W/";
                CornerPanel p = new CornerPanel(prefix);
                gridbag.setConstraints(p, c);
                add(p);
                c.gridx = 2;
                prefix = raft + "/" + Segment.REB.toString() + "G/";
                p = new CornerPanel(prefix);
                gridbag.setConstraints(p, c);
                add(p);
            }
        }

        @Override
        public Dimension getMaximumSize() {
            return getPreferredSize();
        }
        
    }
    
    private class RebPanel extends Box {
        
        RebPanel(String prefix) {
            super(BoxLayout.X_AXIS);
            setBorder(BorderFactory.createCompoundBorder(
                    BorderFactory.createLineBorder(Color.LIGHT_GRAY), 
                    BorderFactory.createEmptyBorder(Const.VSPACE, Const.HSPACE, Const.VSPACE, Const.HSPACE)));
            add(Box.createRigidArea(Const.HDIM));
            add(Box.createHorizontalGlue());
            
            String path = prefix + PATH_MAIN_STATE;
            DisplayChannel channel = localData.get(path);
            if (channel != null) {
                PowerButton b = new PowerButton(channel);
                channel.setTarget(b);
                add(b);
            }
            add(Box.createRigidArea(Const.HDIM));
            add(Box.createHorizontalGlue());
            
            path = prefix + PATH_HV_STATE;
            channel = localData.get(path);            
            if (channel != null) {
                PowerButton b = new PowerButton(channel);
                channel.setTarget(b);
                add(b);
            }
            add(Box.createRigidArea(Const.HDIM));
            add(Box.createHorizontalGlue());
            
            path = prefix + PATH_HV_VALUE;
            channel = localData.get(path);
            if (channel != null) {
                ConfigButton b = new ConfigButton(channel);
                channel.setTarget(b);
                add(b);
            }
            add(Box.createRigidArea(Const.HDIM));
            add(Box.createHorizontalGlue());
            
            path = prefix + PATH_HV_V;
            channel = localData.get(path);
            if (channel != null) {
                TrendButton b = new TrendButton(channel);
                channel.setTarget(b);
                add(b);
            }
            add(Box.createRigidArea(Const.HDIM));
            add(Box.createHorizontalGlue());
            
            path = prefix + PATH_HV_I;
            channel = localData.get(path);
            if (channel != null) {
                TrendButton b = new TrendButton(channel);
                channel.setTarget(b);
                add(b);
            }
            add(Box.createRigidArea(Const.HDIM));
            add(Box.createHorizontalGlue());
        }
    }
    
    private class CornerPanel extends Box {
        
        CornerPanel(String prefix) {
            super(BoxLayout.X_AXIS);
            setBorder(BorderFactory.createCompoundBorder(
                    BorderFactory.createLineBorder(Color.LIGHT_GRAY), 
                    BorderFactory.createEmptyBorder(Const.VSPACE, Const.HSPACE, Const.VSPACE, Const.HSPACE)));
            add(Box.createRigidArea(Const.HDIM));
            add(Box.createHorizontalGlue());
            
            String path = prefix + PATH_MAIN_STATE;
            DisplayChannel channel = localData.get(path);
            if (channel != null) {
                PowerButton b = new PowerButton(channel);
                channel.setTarget(b);
                add(b);
            }
            add(Box.createRigidArea(Const.HDIM));
            add(Box.createHorizontalGlue());
            
            path = prefix + PATH_HV_STATE;
            channel = localData.get(path);
            if (channel != null) {
                PowerButton b = new PowerButton(channel);
                channel.setTarget(b);
                add(b);
            }
            add(Box.createRigidArea(Const.HDIM));
            add(Box.createHorizontalGlue());
            
            path = prefix + PATH_HV_VALUE;
            channel = localData.get(path);
            if (channel != null) {
                ConfigButton b = new ConfigButton(channel);
                channel.setTarget(b);
                add(b);
            }
            add(Box.createRigidArea(Const.HDIM));
            add(Box.createHorizontalGlue());
            
            path = prefix + PATH_HV_V;
            channel = localData.get(path);
            if (channel != null) {
                TrendButton b = new TrendButton(channel);
                channel.setTarget(b);
                add(b);
            }
            add(Box.createRigidArea(Const.HDIM));
            add(Box.createHorizontalGlue());
            
            path = prefix + PATH_HV_I;
            channel = localData.get(path);
            if (channel != null) {
                TrendButton b = new TrendButton(channel);
                channel.setTarget(b);
                add(b);
            }
            add(Box.createRigidArea(Const.HDIM));
            add(Box.createHorizontalGlue());
            
            path = prefix + PATH_DPHI_STATE;
            channel = localData.get(path);
            if (channel != null) {
                PowerButton b = new PowerButton(channel);
                channel.setTarget(b);
                add(b);
            }
            add(Box.createRigidArea(Const.HDIM));
            add(Box.createHorizontalGlue());
            
            path = prefix + PATH_DPHI_VALUE;
            channel = localData.get(path);
            if (channel != null) {
                ConfigButton b = new ConfigButton(channel);
                channel.setTarget(b);
                add(b);
            }
            add(Box.createRigidArea(Const.HDIM));
            add(Box.createHorizontalGlue());
        }
        
    }
    
    static private class PowerButton extends JLabel implements Updatable {
        
        private final DisplayChannel channel;
        
        PowerButton(DisplayChannel channel){
            this.channel = channel;
            setBorder(BorderFactory.createBevelBorder(BevelBorder.RAISED));
            setOpaque(true);
            setBackground(COLOR_UNKNOWN);
            switch (getType()) {
                case 0: setText("   Main   "); break;
                case 1: setText(" HV Bias "); break;
                case 2: setText("   DPhi   "); break;
            }
            setToolTipText(Segment.getPathPrefix(channel.getPath()));
            update(channel);
            addMouseListener(new MouseAdapter() {
                @Override
                public void mouseClicked(MouseEvent e) {
                    if (e.getClickCount() == 1) onMouseClick();
                }
            
            });
        }

        @Override
        public void update(DisplayChannel channelHandle) {
            String state = channel.getChannel().get();
            if (state == null) {
                setBackground(COLOR_UNKNOWN);
            } else {
                switch (state) {
                    case "TRIPPED":
                        setBackground(COLOR_TRIPPED);
                        break;
                    case "OFF":
                        setBackground(COLOR_OFF);
                        break;
                    case "ON":
                        setBackground(COLOR_ON);
                        break;
                    default:
                        setBackground(COLOR_UNKNOWN);
                }
            }
        }
        
        private int getType() {
            String path = channel.getPath();
            if (path.endsWith(PATH_MAIN_STATE)) return 0;
            if (path.endsWith(PATH_HV_STATE)) return 1;
            if (path.endsWith(PATH_DPHI_STATE)) return 2;
            return -1;
        }
        
        private void onMouseClick() {
            Color bg = getBackground();
            String message = null;
            boolean switchOn = false;
            if (COLOR_TRIPPED.equals(bg)) {
                message = "Reset?";
            } else if (COLOR_OFF.equals(bg)) {
                message = "Switch ON?";
                switchOn = true;
            } else if (COLOR_ON.equals(bg)) {
                message = "Switch OFF?";
            }
            if (message != null) {
                String title = Segment.getPathPrefix(channel.getPath());
                int out = JOptionPane.showConfirmDialog(this, message, title, JOptionPane.OK_CANCEL_OPTION);
                if (out == JOptionPane.OK_OPTION) {
                    StringBuilder command = new StringBuilder(channel.getChannel().getAgentName()).append("/").append(title);
                    switch (getType()) {
                        case 0:
                            command.append("powerReb") ; break;
                        case 1:
                            command.append("hvBias") ; break;
                        case 2:
                            command.append("dphi") ; break;
                    }
                    command.append(switchOn ? "On" : "Off");
                    CommandService.getService().execute(command.toString());
                }
            }
        }
        
    }
    
    static private class ConfigButton extends JLabel implements Updatable {
        
        static private final MonitorField FIELD;
        static {
            FormattedValue fv = new FormattedValue(null, null, null, SwingConstants.CENTER, null, false);
            FIELD = new MonitorField(AgentChannel.Key.VALUE, "", fv, null);
        }
        
        DisplayChannel channel;
        
        ConfigButton(DisplayChannel channel) {
            this.channel = channel;
            setToolTipText(channel.getPath());
//            setBorder(BorderFactory.createBevelBorder(BevelBorder.RAISED));
            update(channel);
        }

        @Override
        public void update(DisplayChannel channelHandle) {
            FormattedValue fv = MonitorFormat.DEFAULT.format(FIELD, channel);
            setBackground(fv.getBgColor());
            setForeground(fv.getFgColor());
            setText(fv.getText());
        }
        
    }
    
    static private class TrendButton extends JLabel implements Updatable {
        
        DisplayChannel channel;
        
        TrendButton(DisplayChannel channel) {
            this.channel = channel;
            setOpaque(true);
            setToolTipText(channel.getPath());
            update(channel);
        }

        @Override
        public void update(DisplayChannel channelHandle) {
            FormattedValue fv = MonitorFormat.DEFAULT.format(MonitorField.VALUE, channel);
            setBackground(fv.getBgColor());
            setForeground(fv.getFgColor());
            setText(fv.getText() + channel.getChannel().get(AgentChannel.Key.UNITS));
        }
        
    }

}
