package org.lsst.ccs.subsystem.pathfinder;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.subsystem.common.devices.dataforth.Maq20Device;

/**
 *  Device for controlling the heat exchanger valves.
 * 
 *  @author saxton
 */
public class ValveMaq20Device extends Maq20Device {

    private static final Logger LOG = Logger.getLogger(ValveMaq20Device.class.getName());
    private int[] setMask, getMask, posnMask, outMask, inMask;
    private boolean[] offline;
    private final List<Integer> outModules = new ArrayList<>();
    private final List<Integer> inModules = new ArrayList<>();


    @Override
    protected void initDevice()
    {
        super.initDevice();
        outMask = new int[getModuleCount()];
        inMask = new int[getModuleCount()];
        setMask = new int[getModuleCount()];
        getMask = new int[getModuleCount()];
        posnMask = new int[getModuleCount()];
        offline = new boolean[getModuleCount()];
    }


    @Override
    protected void furtherInitialization() throws DriverException
    {
        for (int mod = 0; mod < getModuleCount(); mod++) {
            if (outMask[mod] != 0) {
                if (getModuleData(mod).maqDiscOut == null) {
                    LOG.log(Level.SEVERE, "{0} MAQ20 module with index {1} cannot perform discrete output", new Object[]{name, mod});
                    offline[mod] = true;
                }
                else {
                    outModules.add(mod);
                    offline[mod] = false;
                }
            }
            if (inMask[mod] != 0) {
                if (getModuleData(mod).maqDiscIn == null) {
                    LOG.log(Level.SEVERE, "{0} MAQ20 module with index {1} cannot perform discrete input", new Object[]{name, mod});
                    offline[mod] = true;
                }
                else {
                    inModules.add(mod);
                    offline[mod] = false;
                }
            }
        }
        getSetValves(true);
        System.arraycopy(getMask, 0, setMask, 0, getModuleCount());
        readPositions(true);
    }


    public void setValves()
    {
        for (int mod : outModules) {
            if (getModuleData(mod).maqDiscOut != null) {
                try {
                    getModuleData(mod).maqDiscOut.writeDiscAll(setMask[mod]);
                    offline[mod] = false;
                }
                catch (DriverException e) {
                    LOG.log(Level.SEVERE, "{0} MAQ20 write error: {1}", new Object[]{name, e});
                    offline[mod] = true;
                    setOnline(false);
                }
            }
        }
    }

    //There are two versions of the getSetValves and readPositions methods:
    //  - one to be invoked during the initialization process. This is meant to
    //    throw an exception and not invoke setOnline(false)
    //  - the other version of the method is invoked during runtime while the device is
    //    online. This is meant to catch the exception and invoke setOnline(false)
    //
    //TO-DO should the exception be thrown at the end of the loop?
    public void getSetValves() {
        try {
            getSetValves(false);
        } catch (DriverException ex) {
            //Nothing to catch since the main method should not throw an exception.
        }
    }

    private void getSetValves(boolean throwException) throws DriverException
    {
        for (int mod : outModules) {
            if (getModuleData(mod).maqDiscOut != null) {
                try {
                    getMask[mod] = getModuleData(mod).maqDiscOut.readDiscAll();
                    offline[mod] = false;
                }
                catch (DriverException e) {
                    LOG.log(Level.SEVERE, "{0} MAQ20 read error: {1}", new Object[]{name, e});
                    offline[mod] = true;
                    if ( throwException ) {
                        throw e;
                    }
                    setOnline(false);
                }
            }
        }
    }

    public void readPositions()
    {
        try {
            readPositions(false);
        } catch (DriverException ex) {
            //Nothing to catch since the main method should not throw an exception.
        }
    }

    private void readPositions(boolean throwException) throws DriverException
    {
        for (int mod : inModules) {
	    //	    LOG.log(Level.INFO, "module = " + mod); 
            if (getModuleData(mod).maqDiscIn != null) { // hn: out to in
		//		LOG.log(Level.INFO, "disc in module = " + mod); 
                try {
                    posnMask[mod] = getModuleData(mod).maqDiscIn.readDiscAll();
		    //		    LOG.log(Level.INFO, "posnMask: module= " + mod + " data= ", posnMask[mod]); 
                    offline[mod] = false;
                }
                catch (DriverException e) {
                    LOG.log(Level.SEVERE, "{0} MAQ20 read error: {1}", new Object[]{name, e});
                    offline[mod] = true;
                    if ( throwException ) {
                        throw e;
                    }
                    setOnline(false);
                }
            }
        }
    }


    public boolean registerOut(int index, int chan)
    {
        if ((outMask[index] & (1 << chan)) != 0 || inMask[index] != 0) return false;
        outMask[index] |= (1 << chan);
        return true;
    }


    public boolean registerIn(int index, int chan)
    {
        if ((inMask[index] & (1 << chan)) != 0 || outMask[index] != 0) return false;
        inMask[index] |= (1 << chan);
        return true;
    }


    public void setValve(int index, int chan, boolean on)
    {
        if (on) {
            setMask[index] |= (1 << chan);
        }
        else {
            setMask[index] &= ~(1 << chan);
        }
    }


    public Boolean isValveSet(int index, int chan)
    {
        return offline[index] ? null : (getMask[index] & (1 << chan)) != 0;
    }


    public Boolean isPositionSet(int index, int chan)
    {
        return offline[index] ? null : (posnMask[index] & (1 << chan)) != 0;
    }

}
