package org.lsst.ccs.subsystem.pathfinder;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.commons.DriverConstants;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.subsystem.common.ErrorUtils;

/**
 *  Defines a power supply interface.
 *
 *  @author Owen Saxton/Homer Neal
 */
public class PathfinderPowerDevice extends Device implements HasLifecycle {
    
    /**
     *  Inner class for saving the power state of a channel.
     */
    public static class ChannelState {
        
        private double
            power;         // Load power
        private boolean
            enabled,       // Output has been enabled
            currError,     // Error setting the voltage
            noLoad;        // Unable to determine the load resistance

    }


    protected static final int
        MON_TYPE_POWER = 0,
        MON_TYPE_VOLTAGE = 1,
        MON_TYPE_CURRENT = 2;

    private final static double
        TIMEOUT = 2.0;

    private static final String
        REFRIG = "Refrig";


    protected static final Map<String, Integer> mTypeMap = new HashMap<>();
    static {
        mTypeMap.put("POWER", MON_TYPE_POWER);
        mTypeMap.put("CURRENT", MON_TYPE_CURRENT);
    }

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;

    //    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    @LookupField(strategy = LookupField.Strategy.TREE)
    private Maq20DevicePF pfControl;

    protected int settlingTime = 100;           // Wait time after setting voltage (ms)

    private static final Logger LOG = Logger.getLogger(PathfinderPowerDevice.class.getName());
    private final String devcName;
    private static final int minChan = 0, maxChan = 1;

    private final Map<Integer, ChannelState> states = new HashMap<>();
    private boolean initError = false;


    /**
     *  Constructor.
     *
     *  @param  name      The device name
     *  @param  minChan   The minimum channel number
     *  @param  maxChan   The maximum channel number
     */
    public PathfinderPowerDevice(String name)
    {
        super();
        this.devcName = name;

        for (int chan = minChan; chan <= maxChan; chan++) {
            states.put(chan, new ChannelState());
        }
    }


    /**
     *  Build phase
     */
    @Override
    public void build() {
        // Create and schedule an AgentPeriodicTask to maintain the set power values
        AgentPeriodicTask pt;
        pt = new AgentPeriodicTask("maintain-power-" + name, () -> maintainPower()).withPeriod(Duration.ofMillis(6000));
        periodicTaskService.scheduleAgentPeriodicTask(pt);
    }


    /**
     *  Gets the minimum channel number.
     *
     *  @return  The minimum channel number
     */
    public int getMinChannel()
    {
        return minChan;
    }


    /**
     *  Gets the maximum channel number.
     *
     *  @return  The maximum channel number
     */
    public int getMaxChannel()
    {
        return maxChan;
    }

   

    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice() {

        for (int chan = minChan; chan <= maxChan; chan++) {
            states.put(chan, new ChannelState());
        }

        // more intelligence TBA                                                                     
        for (int chan : states.keySet()) {
            ChannelState state = states.get(chan);
            state.enabled = true;
            state.currError = false;
        }


	//	fullName = devcName + (devcId.isEmpty() ? "" : " (" + devcId + ")");
	fullName = devcName ;
    }


    @Override
    public void postInit() {

       if (pfControl == null) {
            ErrorUtils.reportConfigError(LOG, name, "Maq20DevicePF component", "has not been defined");
        }

    }

    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {

            LOG.log(Level.INFO, "Connected to {0}", fullName);
            initError = false;
            setOnline(true);

    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {

    }


    /**
     *  Checks a channel's parameters for validity.
     *
     *  @param  name     The channel name
     *  @param  hwChan   The hardware channel number
     *  @param  type     The channel type string
     *  @param  subtype  The channel subtype string
     *  @return  A two-element array containing the encoded type [0] and subtype [1] values.
     *  @throws  Exception if any errors found in the parameters.
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception
    {
        if (hwChan < minChan || hwChan > maxChan) {
            ErrorUtils.reportChannelError(LOG, name, "HW channel", hwChan);
        }
        Integer iType = mTypeMap.get(type.toUpperCase());
        if (iType == null) {
            ErrorUtils.reportChannelError(LOG, name, "type", type);
        }
        return new int[]{iType, 0};
    }





    /**
     *  Sets the online state.
     *
     *  If coming on line, set state data from reading the hardware.
     *
     *  @param  online  The online state to set: true or false
     */
    @Override
    protected void setOnline(boolean online)
    {
        super.setOnline(online);
    }


    /**
     *  Enables the output.
     *
     *  @param  chan    The hardware channel
     *  @param  enable  The enabled state to set
     */
    public void enableOutput(int chan, boolean enable)
    {

        ChannelState state = getChannelState(chan);
        state.enabled = enable;
        if (!enable) {
            pfControl.setHeaterPower(chan, 0.0);
            state.currError = false;
        }
    }


    /**
     *  Sets the power.
     *
     *  @param  chan   The hardware channel
     *  @param  fractionPower  The power to set, or NaN to use current value
     */
    public void setPower(int heaterId, double fractionPower)
    {
        ChannelState state = getChannelState(heaterId);
        synchronized(state) {
            if (!state.enabled) return;

	    
	    //            List<Float> values = new ArrayList<>();
            if (!Double.isNaN(fractionPower)) {
                state.power = fractionPower;
		LOG.log(Level.INFO, "The command would have been: pfControl.setHeaterPower("+heaterId+", "+fractionPower);
		pfControl.setHeaterPower(heaterId, fractionPower);
            }

        }
    }


    /**
     *  Maintains the power for all channels at their set value.
     */
    protected void maintainPower()
    {
        for (int chan : states.keySet()) {
            setPower(chan, Double.NaN);
        }
    }


    /**
     *  Gets the power for a channel.
     *
     *  @param  chan   The hardware channel
     *  @return  The power value
     */
    public double getPower(int chan)
    {
        return getChannelState(chan).power;
    }


    /**
     *  Gets the enabled state.
     *
     *  @param  chan   The hardware channel
     *  @return  Whether enabled
     */
    public boolean isEnabled(int chan)
    {
        return getChannelState(chan).enabled;
    }


    /**
     *  Gets whether a current setting error occurred.
     *
     *  @param  chan   The hardware channel
     *  @return  The error state
     */
    public boolean hasCurrError(int chan)
    {
        return getChannelState(chan).currError;
    }


    /**
     *  Gets whether a "no load" error occurred.
     *
     *  @param  chan   The hardware channel
     *  @return  The error state
     */
    public boolean hasNoLoad(int chan)
    {
        return getChannelState(chan).noLoad;
    }



    /**
     *  Gets the state object for a channel.
     *
     *  @param  chan   The hardware channel
     *  @return  The channel state
     */
    private ChannelState getChannelState(int chan)
    {
        return states.get(chan);
    }


    /**
     *  Log an error message for a channel.
     *
     *  @param  msg   The message text
     *  @param  chan  The hardware channel
     *  @param  excp  The exception, or null if none
     */
    private void logError(String msg, int chan, DriverException e)
    {
        LOG.log(Level.SEVERE, "{0} {1} channel {2}{3}", new Object[]{msg, name, chan, e == null ? "" : ": " + e});
    }

}
