package org.lsst.ccs.subsystem.pathfinder;

import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.common.devices.dataforth.Maq20AnalogControl;
import org.lsst.ccs.subsystem.common.devices.dataforth.Maq20DiscControl;
import org.lsst.ccs.subsystem.common.devices.dataforth.Maq20Device;
//import org.lsst.ccs.subsystem.refrig.constants.CompTypes;
import org.lsst.ccs.subsystem.pathfinder.data.VacuumException;

/**
 *  Handles compressor MAQ20 DIOL and VO modules.
 *
 *  @author Owen Saxton
 */
public class Maq20DeviceUT extends Maq20Device {
    //implements Compressor.SwitchDevice {

    /**
     *  Constants.
     */
    public static final int
        SW_OIL_SEP_HEATER  = 1,
        NUM_COLD_SWITCHES  = 1,   // Used in simulation
        SW_COOLANT_VALVE   = 1,
        SW_ORIFICE_VALVE   = 2,
        SW_BYPASS_VALVE    = 3,
        SW_SURGE_HEATER    = 4,
        NUM_CRYO_SWITCHES  = 4,   // Used in simulation
        CHAN_BYPASS_VALVE  = 0,
        CHAN_EEPR_VALVE    = 1,
        CHAN_COOLANT_VALVE = 2;

    /**
     *  Data fields.
     */
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private Maq20DiscControl discCtrl;
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private Maq20AnalogControl anlgCtrl;
    
    private static final Logger LOG = Logger.getLogger(Maq20DeviceUT.class.getName());
    private boolean gotValveError = false;


    /**
     *  Sets the compressor type.
     *
     *  @param  type  The type (cold or cryo)
     */
    public void setType(int type)
    {
        if (discCtrl == null) {
            ErrorUtils.reportConfigError(LOG, name, "MAQ20 discrete control", "not defined");
        }
	//        if (type == CompTypes.TYPE_COLD && anlgCtrl == null) {
	//           ErrorUtils.reportConfigError(LOG, name, "MAQ20 analog control", "not defined");
	//        }
    }


    /**
     *  Sets a switch on or off.
     *
     *  @param  sw  The switch number.
     *  @param  on  The on state to set: true or false
     *  @throws  VacuumException
     */
    //    @Override
    public void setSwitchOn(int sw, boolean on) throws VacuumException
    {
        try {
            discCtrl.setLineOn(sw, on);
        }
        catch (DriverException e) {
            throw new VacuumException(e.getMessage());
        }
    }


    /**
     *  Gets the on state of a switch.
     *
     *  @param  sw  The switch number.
     *  @return  Whether the switch is on
     */
    //    @Override
    public Boolean isSwitchOn(int sw)
    {
        return discCtrl.isLineOn(sw);
    }


    /**
     *  Sets a valve position.
     *
     *  @param  chan  The valve channel number
     *  @param  posn  Requested position (0 - 1)
     *  @throws  VacuumException
     */
    public void setValvePosition(int chan, double posn) throws VacuumException
    {
        double value = 10.0 * Math.min(Math.max(posn, 0.0), 1.0);
        try {
            anlgCtrl.writeValue(chan, value);
            gotValveError = false;
        }
        catch (DriverException e) {
            if (!gotValveError) {
                throw new VacuumException("Error setting valve position for " + getPath() + ": " + e);
            }
            gotValveError = true;
        }
    }


    /**
     *  Gets a valve position.
     *
     *  @param  chan  The valve channel number
     *  @return  Valve position (0 - 1), or NaN if error
     */
    public double getValvePosition(int chan)
    {
        try {
            double value = anlgCtrl.readValue(chan) / 10.0;
            gotValveError = false;
            return value;
        }
        catch (DriverException e) {
            if (!gotValveError) {
                LOG.log(Level.SEVERE, "Error getting valve position for {0}: {1}", new Object[]{getPath(), e});
            }
            gotValveError = true;
            return Double.NaN;
        }
    }

}
