package org.lsst.ccs.subsystem.pathfinder;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.config.ConfigurationParameterDescription;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.common.devices.dataforth.Maq20PWMControl;
import org.lsst.ccs.subsystem.common.PIController;
import org.lsst.ccs.subsystem.pathfinder.data.VacuumException;

/**
 *  Implements a fan speed loop controller for the utility trunk system.
 *
 *  Also allows open-loop speed control if no temperatures are specified.
 *
 *  @author Owen Saxton
 */
public class FanPIControl implements HasLifecycle {

    private static class TempData {

        Channel chan;    // Temperature channel
        double weight;   // Temperature weight

        TempData(Channel chan, double weight) {
            this.chan = chan;
            this.weight = weight;
        }

    }

    private static final String PIC = "Pic";

    @LookupField(strategy = LookupField.Strategy.TOP)
    Subsystem subsys;
    @LookupName
    private String name;
    @LookupPath
    private String path;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskServices;

    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile Double  gain = Double.NaN;              // loop gain
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile Double  timeConst = Double.NaN;         // integration time constant (secs)
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile Double  smoothTime = Double.NaN;        // input smoothing time (secs)
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile Double  awGain = Double.NaN;            // anti-windup gain
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile Double  baseDuty = Double.NaN;          // base duty cycle input
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile Double  tolerance = Double.NaN;         // maximum on-target error (%)
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile double  minInput = -10.0;               // minimum input
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile double  maxInput = 10.0;                // maximum input
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile double  minDutyCycle = 0.1;             // minimum duty cycle (and PID output)
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile double  maxDutyCycle = 0.9;             // maximum duty cycle (and PID output)
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile double  offDutyCycle = 0.05;            // turn-off duty cycle
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile int minRpm = 1000;                      // minimum fan RPM
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile int maxRpm = 7000;                      // maximum fan RPM

    private List<Channel> refTempChans;    // Reference temperature channels to use
    private List<Channel> ctrlTempChans;   // Reference temperature channels to use
    private double[] ctrlTempWeights;      // Control temperature weights
    private Maq20PWMControl fan;           // Maq20 fan controller to use

    private static final Logger LOG = Logger.getLogger(FanPIControl.class.getName());
    private final List<TempData> ctrlTempDataList = new ArrayList<>();  // Control temperature data to use
    private PIController pic;          // The PI controller
    private double dcSlope;            // The RPM/duty cycle slope
    private double lastDuty;           // The last duty cycle value set
    private boolean active = false;    // True if loop is active
    private double setTemp;            // Temperature set point
    private long speedSetTime;         // The time the speed was last set
    private boolean loopFailed = false;


    /**
     *  Sets up the fan speed control loop timer task.
     */
    @Override
    public void build() {
        ConfigurationParameterDescription desc = new ConfigurationParameterDescription().withCategory(PIC).withName("updateTime");
        AgentPeriodicTask apt;
        apt = new AgentPeriodicTask("fan-loop-" + name,
                                   () -> iterateLoop()).withPeriod(Duration.ofMillis(10000)).withPeriodParameterDescription(desc);
        periodicTaskServices.scheduleAgentPeriodicTask(apt);
    }


    /**
     *  Initializes the parameters.
     */
    @Override
    public void init()
    {
        dcSlope = (maxDutyCycle - minDutyCycle) / (maxRpm - minRpm);
        if (!Double.isFinite(dcSlope) || dcSlope < 0.0) {
            ErrorUtils.reportConfigError(LOG, path, "RPM conversion constants", "are invalid");
        }
        if (fan == null) {
            ErrorUtils.reportConfigError(LOG, path, "fan", "has not been specified");
        }
        if (refTempChans == null) {
            ErrorUtils.reportConfigError(LOG, path, "refTempChans", "has not been specified");
        }
        if (ctrlTempChans == null) {
            ErrorUtils.reportConfigError(LOG, path, "ctrlTempChans", "has not been specified");
        }
        if (ctrlTempWeights == null) {
            ErrorUtils.reportConfigError(LOG, path, "ctrlTempWeights", "has not been specified");
        }
        if (ctrlTempWeights.length != ctrlTempChans.size()) {
            ErrorUtils.reportConfigError(LOG, path, "ctrlTempWeights and ctrlTempChans", "have different numbers of elements");
        }
        double total = 0.0;
        for (int j = 0; j < ctrlTempChans.size(); j++) {
            double weight = ctrlTempWeights[j];
            if (weight < 0.0) {
                ErrorUtils.reportConfigError(LOG, path, "ctrlTempWeights", "contains a negative value");
            }
            ctrlTempDataList.add(new TempData(ctrlTempChans.get(j), weight));
            total += weight;
        }
        if (ctrlTempDataList.isEmpty() ^ refTempChans.isEmpty()) {
            ErrorUtils.reportConfigError(LOG, path, "ctrlTempChans and refTempChans", "are neither both non-empty nor both empty");
        }
        if (ctrlTempDataList.isEmpty()) return;  // No loop control requested

        if (total == 0.0) {
            ErrorUtils.reportConfigError(LOG, path, "ctrlTempWeights", "total is zero");
        }
        if (gain == null) {
            ErrorUtils.reportConfigError(LOG, path, "gain", "is missing");
        }
        if (timeConst.isNaN()) {
            ErrorUtils.reportConfigError(LOG, path, "timeConst", "is missing");
        }
        if (smoothTime.isNaN()) {
            ErrorUtils.reportConfigError(LOG, path, "smoothTime", "is missing");
        }
        if (awGain.isNaN()) {
            ErrorUtils.reportConfigError(LOG, path, "awGain", "is missing");
        }
        if (baseDuty.isNaN()) {
            ErrorUtils.reportConfigError(LOG, path, "baseDuty", "is missing");
        }
        if (tolerance.isNaN()) {
            ErrorUtils.reportConfigError(LOG, path, "tolerance", "is missing");
        }
        pic = new PIController(-gain, timeConst);
        pic.setSmoothTime(smoothTime);
        pic.setAwGain(awGain);
        pic.setBaseOutput(baseDuty);
        pic.setInputRange(minInput, maxInput);
        pic.setOutputRange(minDutyCycle, maxDutyCycle);
        pic.setTolerance(tolerance);
    }


    /**
     *  Sets the fan speed (duty cycle).
     *
     *  @param  speed  The speed to set
     */
    public void setSpeed(double speed)
    {

	LOG.log(Level.INFO, "setSpeed called with speed = "+speed);
        try {
            fan.setDutyCycle1(speed);
	    LOG.log(Level.INFO, "setDutyCycle1 called with speed = "+speed);
            speedSetTime = System.currentTimeMillis();
        }
        catch (Exception ex) {
            LOG.log(Level.SEVERE, "Error setting {0} fan speed: {1}", new Object[]{fan, ex});
        }
    }


    /**
     *  Gets the fan speed (duty cycle).
     *
     *  @return  The set speed
     */
    public Double getSpeed()
    {
        try {
            return fan.getDutyCycle1();
        }
        catch (Exception ex) {
            LOG.log(Level.SEVERE, "Error getting {0} fan speed: {1}", new Object[]{fan, ex});
            return null;
        }
    }


    /**
     *  Gets the expected RPM
     *
     *  @return  rpm  The RPM value
     */
    public Integer getRpm()
    {
        Double speed = getSpeed();
        return speed == null ? null : speed < offDutyCycle ? 0
                 : Math.max(minRpm, Math.min(maxRpm, (int)(minRpm + (speed - minDutyCycle) / dcSlope)));
    }


    /**
     *  Gets the speed set time.
     *
     *  Need to know this because it can take a couple of seconds to reach its set speed
     * 
     *  @return  The set time (ms)
     */
    public long getSpeedSetTime()
    {
        return speedSetTime;
    }


    /**
     *  Sets the target temperature.
     *
     *  @param  temp  The temperature to set
     *  @throws  VacuumException
     */
    public void setTemperature(double temp) throws VacuumException
    {
        checkPicPresent();
        setTemp = temp;
        pic.setSetpoint(setTemp);
    }


    /**
     *  Gets the target temperature.
     *
     *  @return  The set temperature
     */
    public double getTemperature()
    {
        return setTemp;
   }


    /**
     *  Sets the gain.
     *
     *  @param  gain  The gain to set
     *  @throws  VacuumException
     */
    public void setGain(double gain) throws VacuumException
    {
        checkPicPresent();
        this.gain = gain;
        if (pic != null) {  // Can get called during startup before pic exists
            pic.setPID(-gain, timeConst);
        }
    }


    /**
     *  Gets the gain.
     *
     *  @return  The gain
     */
    public double getGain()
    {
        return gain;
    }


    /**
     *  Sets the time constant.
     *
     *  @param  time  The time constant to set
     *  @throws  VacuumException
     */
    public void setTimeConstant(double time) throws VacuumException
    {
        checkPicPresent();
        timeConst = time;
        pic.setPID(-gain, timeConst);
    }


    /**
     *  Gets the time constant.
     *
     *  @return  The time constant
     */
    public double getTimeConstant()
    {
        return timeConst;
   }


    /**
     *  Starts the control loop.
     *
     *  @param  duty  The initial duty cycle value
     *  @throws  VacuumException
     */
    public void startLoop(double duty) throws VacuumException
    {
        checkPicPresent();
        if (active) return;
        lastDuty = duty;
        startLoop();
    }


    /**
     *  Starts the control loop.
     *
     *  @throws  VacuumException
     */
    public void startLoop() throws VacuumException
    {
        checkPicPresent();
        if (active) return;
        pic.reset();
        pic.setIntegral(lastDuty - baseDuty);
        active = true;
    }


    /**
     *  Stops the control loop.
     *
     *  @throws  VacuumException
     */
    public void stopLoop() throws VacuumException
    {
        checkPicPresent();
        if (!active) return;
        active = false;
    }


    /**
     *  Gets the control loop state.
     *
     *  @return  Whether the control loop is active
     */
    public boolean isLoopActive()
    {
        return active;
    }


    /**
     *  Resets the controller.
     * 
     *  @throws  VacuumException
     */
    public void reset() throws VacuumException
    {
        checkPicPresent();
        pic.reset();
    }


    /**
     *  Timer routine for control loop iteration.
     */
    private void iterateLoop()
    {
        if (!active) return;
        double refTemp = 0.0;
        int count = 0;
        for (Channel tempChan : refTempChans) {
            double value = tempChan.getValue();
            if (!Double.isNaN(value)) {
                refTemp += value;
                count++;
            }
        }
        if (count > 0) {
            refTemp /= count;
        }
        else {
            if (!loopFailed) {
                LOG.log(Level.SEVERE, "{0} control loop iteration failed: no valid reference temperature values available", path);
                loopFailed = true;
            }
            return;
        }
        double ctrlTemp = 0.0, totalWeight = 0.0;
        for (TempData tempData : ctrlTempDataList) {
            double value = tempData.chan.getValue();
            if (!Double.isNaN(value)) {
                ctrlTemp += value * tempData.weight;
                totalWeight += tempData.weight;
            }
        }
        if (totalWeight > 0.0) {
            ctrlTemp /= totalWeight;
        }
        else {
            if (!loopFailed) {
                LOG.log(Level.SEVERE, "{0} control loop iteration failed: no valid control temperature values available", path);
                loopFailed = true;
            }
            return;
        }
        lastDuty = pic.performPI(new double[]{ctrlTemp - refTemp}, (double)System.currentTimeMillis() / 1000);
        setSpeed(lastDuty);
        loopFailed = false;
    }


    /**
     *  Checks whether PI controller is present.
     */
    private void checkPicPresent() throws VacuumException
    {
        if (pic == null) {
            throw new VacuumException("Control loop not present");
        }
    }

}
