package org.lsst.ccs.subsystem.ocsbridge.util;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collector;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.sal.cccamera.event.FocalPlaneHardwareIdSettingsAppliedEvent;

/**
 * Generates a hardareIdSettigsApplied event from focal-plane config.
 * This should be converted to a generic converter in future.
 *
 * @author tonyj
 */
public class FocalPlaneHardwareIdConverter {

    private final static Pattern SENSOR_PATTERN = Pattern.compile("(R..)/(Reb.)/(S..)/(\\w+)");
    private final static Pattern REB_PATTERN = Pattern.compile("(R..)/(Reb.)(?:_hardware)?/(\\w+)");
    private final static Pattern RAFT_PATTERN = Pattern.compile("(R..)/(.+)");
    private final static DelimitedStringSplitJoin DELIMITED_STRING_SPLIT_JOIN = new DelimitedStringSplitJoin();

    public FocalPlaneHardwareIdSettingsAppliedEvent convert(ConfigurationInfo data) {
        Set<String> ccdLocations = new TreeSet<>();
        Set<String> rebLocations = new TreeSet<>();
        Set<String> raftLocations = new TreeSet<>();
        Map<String, String> ccdNames = new HashMap<>();
        Map<String, String> manSerNums = new HashMap<>();
        Map<String, String> rebNames = new HashMap<>();
        Map<String, String> raftNames = new HashMap<>();

        // First loop over hardware Ids
        Map<String, String> hardwareMap = data.getCurrentValuesForCategory("HardwareId");
        for (Map.Entry<String, String> entry : hardwareMap.entrySet()) {
            //System.out.println(entry.getKey() + "=" + entry.getValue());
            Matcher sensorMatcher = SENSOR_PATTERN.matcher(entry.getKey());
            Matcher rebMatcher = REB_PATTERN.matcher(entry.getKey());
            Matcher raftMatcher = RAFT_PATTERN.matcher(entry.getKey());

            if (sensorMatcher.matches()) {
                final String ccdName = sensorMatcher.group(1) + sensorMatcher.group(3);
                ccdLocations.add(ccdName);
                String type = sensorMatcher.group(4);
                if ("name".equals(type)) {
                    ccdNames.put(ccdName, entry.getValue());
                } else if ("manSerNum".equals(type)) {
                    manSerNums.put(ccdName, entry.getValue());
                }
            } else if (rebMatcher.matches()) {
                final String rebName = rebMatcher.group(1) + rebMatcher.group(2);
                rebLocations.add(rebName);
                String type = rebMatcher.group(3);
                if ("name".equals(type)) {
                    rebNames.put(rebName, entry.getValue());
                }
            } else if (raftMatcher.matches()) {
                final String raftName = raftMatcher.group(1);
                raftLocations.add(raftName);
                String type = raftMatcher.group(2);
                if ("name".equals(type)) {
                    raftNames.put(raftName, entry.getValue());
                }
            }
        }

        return FocalPlaneHardwareIdSettingsAppliedEvent.builder()
                .version(data.getConfigVersion("HardwareId"))
                .ccdLocation(ccdLocations.stream().collect(join()))
                .ccdLSSTName(ccdLocations.stream().map(k -> ccdNames.get(k)).collect(join()))
                .ccdManSerNum(ccdLocations.stream().map(k -> get(manSerNums, k)).collect(join()))
                .rebLocation(rebLocations.stream().collect(join()))
                .rebLSSTName(rebLocations.stream().map(name -> get(rebNames, name)).collect(join()))
                .raftLocation(raftLocations.stream().collect(join()))
                .raftLSSTName(raftLocations.stream().map(name -> get(raftNames, name)).collect(join()))
                .build();
    }
    
    private static <K> String get(Map<K, String> map, K name) {
        return get(map, name, "Unknown");
    }
    
    private static <K, V> V get(Map<K, V> map, K key, V defaultValue) {
        return map.getOrDefault(key, defaultValue);
    }
    
    private Collector<CharSequence, ?, String> join() {
        return DELIMITED_STRING_SPLIT_JOIN.joining();
    }
}
