package org.lsst.ccs.subsystem.ocsbridge;

import java.time.Duration;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.subsystem.ocsbridge.events.EventListener;
import org.lsst.ccs.subsystem.ocsbridge.states.CameraMotionState;
import org.lsst.sal.SAL;
import org.lsst.sal.SALCommand;
import org.lsst.sal.SALCommandResponse;
import org.lsst.sal.SALEvent;
import org.lsst.sal.SALException;
import org.lsst.sal.SALTelemetry;

/**
 * This class instantiated from groovy
 * @author tonyj
 */
public class MotionLockUnlockEventHandler implements EventListener<SALEvent>, HasLifecycle {

    private static final Logger LOG = Logger.getLogger(MotionLockUnlockEventHandler.class.getName());
    private SAL<SALCommand, SALEvent, SALTelemetry> rotatorManager;
    private SAL<SALCommand, SALEvent, SALTelemetry> mountManager;
    private org.lsst.sal.mtmount.enumeration.MotionLockState mountMotionLockState;
    private org.lsst.sal.mtrotator.enumeration.MotionLockState rotatorMotionLockState;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService agentStateService;
 
    @Override
    public void eventFired(SALEvent event) {
        if (event instanceof org.lsst.sal.mtmount.states.SummaryStateEvent sse) {
            LOG.log(Level.INFO, "MTMount service state changed {0}", sse);
        } else if (event instanceof org.lsst.sal.mtrotator.states.SummaryStateEvent sse) {
            LOG.log(Level.INFO, "MTRotator service state changed {0}", sse);
        } else if (event instanceof org.lsst.sal.mtmount.event.HeartbeatEvent heartbeat) {
            LOG.log(Level.FINE, "MTMount heartbeat {0}", heartbeat);
        } else if (event instanceof org.lsst.sal.mtrotator.event.HeartbeatEvent heartbeat) {
            LOG.log(Level.FINE, "MTRotator heartbeat {0}", heartbeat);
        } else if (event instanceof org.lsst.sal.mtmount.states.MotionLockStateEvent mlse) {
            LOG.log(Level.INFO,"MTMount MotionLockStateEvent {0}", mlse);
            mountMotionLockState = org.lsst.sal.mtmount.enumeration.MotionLockState.forSALValue(mlse.getLockState());
            setMotionLockState();
        } else if (event instanceof org.lsst.sal.mtrotator.states.MotionLockStateEvent mlse) {
            LOG.log(Level.INFO,"MTRotator MotionLockStateEvent {0}", mlse);
            rotatorMotionLockState = org.lsst.sal.mtrotator.enumeration.MotionLockState.forSALValue(mlse.getLockState());
            setMotionLockState();
        }
    }

    private synchronized void setMotionLockState() {
        boolean locked = (mountMotionLockState == org.lsst.sal.mtmount.enumeration.MotionLockState.LOCKED) &&
            (rotatorMotionLockState == org.lsst.sal.mtrotator.enumeration.MotionLockState.LOCKED);
         agentStateService.updateAgentState(locked ? CameraMotionState.LOCKED : CameraMotionState.UNLOCKED);       
    }
    
    void lock() throws TimeoutException, SALException {
        execute("MTRotator", rotatorManager, new org.lsst.sal.mtrotator.command.LockMotionCommand());
        execute("MTMount", mountManager, new org.lsst.sal.mtmount.command.LockMotionCommand());
    }

    void unlock() throws TimeoutException, SALException {
        execute("MTRotator", rotatorManager, new org.lsst.sal.mtrotator.command.UnlockMotionCommand());
        execute("MTMount", mountManager, new org.lsst.sal.mtmount.command.UnlockMotionCommand());
    }
    
    boolean isLocked() {
        return agentStateService.getState(CameraMotionState.class) == CameraMotionState.LOCKED;
    }
    
    public void execute(String csc, SAL<SALCommand, SALEvent, SALTelemetry> mgr, SALCommand cmd) throws TimeoutException, SALException {
        try {
            LOG.log(Level.INFO, "Sending {0}: {1}", new Object[]{csc, cmd});
            SALCommandResponse response = mgr.issueCommand(cmd);
            Duration estimatedTime = response.waitForAck(Duration.ofSeconds(10));
            if (estimatedTime != Duration.ZERO) {
                LOG.log(Level.INFO, "Expected command duration {0}", estimatedTime);
            }
            int rc = response.waitForCompletion(estimatedTime.plus(Duration.ofSeconds(10)));
            LOG.log(Level.INFO, "Command complete ack={0}", rc);
        } catch (SALCommandResponse.CommandFailedException ex) {
            LOG.log(Level.SEVERE, "Command failed: errorCode={1}, ack={2}, message={0}", new Object[]{ex.getMessage(), ex.getErrorCode(), ex.getAck()});
            throw ex;
        } catch (TimeoutException ex) {
            LOG.log(Level.WARNING, "Timeout while waiting for ack from: {0}", cmd);
            throw ex;
        } catch (SALException ex) {
            LOG.log(Level.SEVERE, "Exception executing SAL command: {0}", ex.getMessage());
            throw ex;
        }
    }

    void setManagers(SAL<SALCommand, SALEvent, SALTelemetry> mtRotatorManager, SAL<SALCommand, SALEvent, SALTelemetry> mtMountManager) {
        this.rotatorManager = mtRotatorManager;
        this.mountManager = mtMountManager;
    }
    
}
