package org.lsst.ccs.subsystem.ocsbridge;

import java.io.IOException;
import org.lsst.ccs.subsystem.ocsbridge.events.CCSEvent;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import java.io.Serializable;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentLock;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.CommandAck;
import org.lsst.ccs.bus.messages.CommandNack;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.CommandResult;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusStateChangeNotification;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.command.RawCommand;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.messaging.CommandOriginator;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.services.AgentLockService;
import org.lsst.ccs.services.AgentLoginService;
import org.lsst.ccs.services.UnauthorizedLevelException;
import org.lsst.ccs.services.UnauthorizedLockException;
import org.lsst.ccs.subsystem.imagehandling.data.AdditionalFile;
import org.lsst.ccs.subsystem.ocsbridge.events.EventListener;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * An implementation of MCMLayer which uses the CCS buses to communicate with an
 * MCM running in a separate process.
 *
 * @author tonyj
 */
class MCMCCSLayer implements MCMLayer {

    private static final Logger LOG = Logger.getLogger(MCMCCSLayer.class.getName());

    private final Subsystem subsystem;
    private final CCS ccs;
    private final OCSBridgeConfig config;
    private final AgentLockService agentLockService;
    private final AgentLoginService loginService;
    private final LinkedBlockingQueue<StatusMessage> statusMessagesQueue = new LinkedBlockingQueue<>();
    private final Future processQueueFuture;
    private final StatusMessageListener mcmStatusChangeListener; 
    private final StatusMessageListener mcmStatusMessageListener; 
    private final List<StatusMessageListener> listenSubsystemStatusMessageListenerList = new ArrayList<>(); 
    private final StatusMessageListener listenSubsystemAdditionalFileListener;
   

    MCMCCSLayer(Subsystem subsystem, CCS ccs, OCSBridgeConfig config, AgentLoginService loginService, AgentLockService agentLockService) {
        this.subsystem = subsystem;
        this.ccs = ccs;
        this.config = config;
        this.agentLockService = agentLockService;
        this.loginService = loginService;

        ExecutorService queueExecutor = Executors.newSingleThreadExecutor(r -> {
            Thread t = new Thread(r, "StatusMessage processing queue");
            t.setDaemon(true);
            return t;
        });	      
        
        processQueueFuture = queueExecutor.submit(()->processBusMessage());		
        
        Predicate<BusMessage<? extends Serializable, ?>> stateChangeFilter
                = BusMessageFilterFactory.messageOrigin(config.getMCMName()).and(BusMessageFilterFactory.messageClass(StatusStateChangeNotification.class));

        mcmStatusChangeListener = (msg) -> handleMCMStatusChangeMessage(msg);
        subsystem.getMessagingAccess().addStatusMessageListener(mcmStatusChangeListener, stateChangeFilter);

        
        Predicate<BusMessage<? extends Serializable, ?>> mcmOrigin = BusMessageFilterFactory.messageOrigin(config.getMCMName());
        Predicate<BusMessage<? extends Serializable, ?>> keyValueDataClass = BusMessageFilterFactory.embeddedObjectClass(KeyValueData.class);

        //Listen to all KeyValue data from the MCM. This includes AdditionalFiles
        mcmStatusMessageListener = (msg) -> handleMCMEvent(msg);
        subsystem.getMessagingAccess().addStatusMessageListener(mcmStatusMessageListener, mcmOrigin.and(keyValueDataClass));

        //Predicate for additionalFiles content
        Predicate<BusMessage<? extends Serializable, ?>> additionalFileFilter = (BusMessage<? extends Serializable, ?> msg) -> {
            if ( msg instanceof StatusSubsystemData ) {
                return ((StatusSubsystemData)msg).getSubsystemData().getValue() instanceof AdditionalFile;
            }
            return false;           
        };

        //Listen for status messages coming from listenSubsystems, except for AdditionaFiles
        for (String l : config.getListenSubsystems()) {
            Predicate<BusMessage<? extends Serializable, ?>> subsystemFilter = BusMessageFilterFactory.messageOrigin(l);
            StatusMessageListener subsystemStatusMessageListener = (msg) -> handleSubsystemEvent(msg);
            listenSubsystemStatusMessageListenerList.add(subsystemStatusMessageListener);
            subsystem.getMessagingAccess().addStatusMessageListener(subsystemStatusMessageListener, subsystemFilter.and(additionalFileFilter.negate()));
        }
        
        //Listen for AdditionalFiles published by anybody else, except the MCM, since we are already listening to all
        //KeyValue data published by the MCM
        listenSubsystemAdditionalFileListener = (msg) -> handleSubsystemEvent(msg);
        subsystem.getMessagingAccess().addStatusMessageListener(listenSubsystemAdditionalFileListener, additionalFileFilter.and(mcmOrigin.negate()));                
    }

    @Override
    public CCSCommand.CCSCommandResponse execute(CCSCommand ccsCommand) {
        CCSExecutor executor = new CCSBusesExecutor(ccsCommand);
        return new CCSCommand.CCSCommandResponse(executor);
    }

    @Override
    public void addStateChangeListener(State.StateChangeListener<Enum> stateChangeListener) {
        ccs.addStateChangeListener(stateChangeListener);
    }

    @Override
    public void removeStateChangeListener(State.StateChangeListener<Enum> stateChangeListener) {
        ccs.removeStateChangeListener(stateChangeListener);
    }

    @Override
    public void addEventListener(CCSEvent.CCSEventListener eventListener) {
        ccs.addEventListener(eventListener);
    }

    @Override
    public void removeEventListener(CCSEvent.CCSEventListener eventListener) {
        ccs.removeEventListener(eventListener);
    }

    @Override
    public void addStatusMessageListener(EventListener<StatusMessage> eventListener) {
        ccs.addStatusMessageListener(eventListener);
    }

    @Override
    public void removeStatusMessageListener(EventListener<StatusMessage> eventListener) {
        ccs.removeStatusMessageListener(eventListener);
    }
    
    @Override
    public void shutdown() {
        //Remove all the status message listeners.
        subsystem.getMessagingAccess().removeStatusMessageListener(mcmStatusChangeListener);
        subsystem.getMessagingAccess().removeStatusMessageListener(mcmStatusMessageListener);
        for ( StatusMessageListener l : listenSubsystemStatusMessageListenerList ) {
            subsystem.getMessagingAccess().removeStatusMessageListener(l);
        }
        subsystem.getMessagingAccess().removeStatusMessageListener(listenSubsystemAdditionalFileListener);
        
        //Clear the queue of status messages to be processed.
        statusMessagesQueue.clear();        
        //Stop the status message processing thread.
        processQueueFuture.cancel(true);
    }
    
    private void processBusMessage() {
        try {
            while (true) {
                ccs.fireEvent(statusMessagesQueue.take());
            }
        } catch (InterruptedException e) {
            throw new RuntimeException("StatusMessage processing thread has been interrupted.", e);
        }
    }

    private void handleMCMStatusChangeMessage(StatusMessage msg) {
        StatusStateChangeNotification statusChange = (StatusStateChangeNotification) msg;
        StateBundle newStates = statusChange.getNewState();
        StateBundle oldStates = statusChange.getOldState();
        StateBundle changedStates = newStates.diffState(oldStates);
        CCSTimeStamp when = statusChange.getStateTransitionTimestamp();
        String cause = statusChange.getCause();
        changedStates.getDecodedStates().entrySet().stream().map((changedState) -> changedState.getValue()).forEachOrdered((value) -> {
            ccs.getAggregateStatus().add(when, new State(value, cause));
        });
    }

    private void handleMCMEvent(StatusMessage msg) {
        Object object = msg.getObject();
        if (object instanceof KeyValueData) {
            KeyValueData data = (KeyValueData) object;
            if ("CCSEvent".equals(data.getKey())) {
                CCSEvent evt = (CCSEvent) data.getValue();
                ccs.fireEvent(evt);
                return;
            }
        }
        statusMessagesQueue.offer(msg);
    }

    private void handleSubsystemEvent(StatusMessage msg) {
        statusMessagesQueue.offer(msg);
    }

    @Override
    public void lock() throws ExecutionException {
        String targetSubsystem = config.getMCMName();
        try {
            AgentLock lock = agentLockService.getLockForAgent(targetSubsystem);
            if (lock == null) {
                LOG.log(Level.INFO, "Locking agent {0}", targetSubsystem);
                agentLockService.setLevelForAgent(targetSubsystem, 0);
            } 
            try {
                CCSCommand.CCSCommandResponse cr = execute(new CCSCommand.NoArgCCSCommand("switchToNormalMode"));
                CCSCommand.CCSAckOrNack waitForAckOrNack = cr.waitForAckOrNack();
                if (waitForAckOrNack.isNack()) {
                    throw new Exception(waitForAckOrNack.getReason());
                }
                cr.waitForCompletion();
            } catch (Exception x) {
                // No point leaving it locked in this case
                agentLockService.unlockAgent(targetSubsystem);
                throw x instanceof ExecutionException ? (ExecutionException) x : new ExecutionException(x);
            }
        } catch (RuntimeException | UnauthorizedLevelException | UnauthorizedLockException x) {
            throw new ExecutionException("Failed to lock subsystem " + targetSubsystem, x);
        }
    }

    @Override
    public void unlock() throws ExecutionException {
        String agent = config.getMCMName();
        try {
            agentLockService.unlockAgent(agent);
        } catch (UnauthorizedLockException | RuntimeException ex) {
            throw new ExecutionException("Failed to unlock " + agent, ex);
        }
    }

    /**
     * This class is used by the OCSBridge to send a command to the MCM.
     */
    private class CCSBusesExecutor extends CCSExecutor {

        private final ArrayBlockingQueue queue = new ArrayBlockingQueue(2);
        private final CCSCommand ccsCommand;

        public CCSBusesExecutor(CCSCommand ccsCommand) {
            this.ccsCommand = ccsCommand;
            CommandOriginator originator = new CommandOriginator() {
                @Override
                public void processAck(CommandAck ack) {
                    queue.offer(ack);
                }

                @Override
                public void processResult(CommandResult result) {
                    queue.offer(result);
                }

                @Override
                public void processNack(CommandNack nack) {
                    queue.offer(nack);
                }
            };
            CommandRequest request = new CommandRequest(config.getMCMName(), new RawCommand(ccsCommand.getCommand(), ccsCommand.getArguments()));
            subsystem.getMessagingAccess().sendCommandRequest(request, originator);
        }

        @Override
        protected Duration testPreconditions() throws CCSCommand.CCSPreconditionsNotMet {
            try {
                Object result = queue.take();
                if (result instanceof CommandAck) {
                    return ((CommandAck) result).getTimeout();
                } else if (result instanceof CommandNack) {
                    throw new CCSCommand.CCSPreconditionsNotMet(((CommandNack) result).getReason());
                } else {
                    throw new RuntimeException("Unexpected object received while waiting for ack/nack: " + ccsCommand.toString());
                }
            } catch (InterruptedException ex) {
                throw new RuntimeException("Unexpected interrupt while waiting for command ack/nack: " + ccsCommand.toString(), ex);
            }
        }

        @Override
        protected void execute() throws ExecutionException {
            try {
                Object result = queue.take();
                if (result instanceof CommandResult) {
                    CommandResult cr = (CommandResult) result;
                    if (!cr.wasSuccessful()) {
                        Throwable t = (Throwable) cr.getObject();
                        if (t instanceof ExecutionException) {
                            throw (ExecutionException) t;
                        } else {
                            throw new ExecutionException("MCM command failed: " + ccsCommand.toString(), t);
                        }
                    }
                } else {
                    throw new RuntimeException("Unexpected object received while waiting for command result: " + ccsCommand + " " + result);
                }
            } catch (InterruptedException x) {
                throw new RuntimeException("Interrupt while waiting for command result: " + ccsCommand.toString(), x);
            }
        }
    }

}
