package org.lsst.ccs.subsystem.mmm;

import java.util.ArrayList;
import java.util.logging.Level;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.RaisedAlertHistory;
import org.lsst.ccs.bus.data.RaisedAlertSummary;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.messaging.AgentPresenceManager;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.services.alert.AlertEvent;
import org.lsst.ccs.services.alert.AlertEvent.AlertEventType;
import org.lsst.ccs.services.alert.AlertService;

public class AlertNotifier {

    static Logger log = Logger.getLogger("lsst.ccs.bus");

    // Map containing the most recend RaisedAlertSummary for all agents on the buses
    private final Object updateInProgress = new Object();

    private final MMMUtilities mmmUtils;
    private final AlertService alertService;
    private final AgentPresenceManager apm;
    private final AgentInfo thisAgentInfo;

    public AlertNotifier(MMMUtilities mmmUtils, Agent agent) {
        this.mmmUtils = mmmUtils;
        alertService = agent.getAgentService(AlertService.class);
        alertService.addListener((event) -> processEvent(event));
        apm = agent.getMessagingAccess().getAgentPresenceManager();
        thisAgentInfo = agent.getAgentInfo();
    }

    private void processEvent(AlertEvent event) {
        if (event.getType()==AlertEventType.ALERT_RAISED) {
            String origin = event.getSource();
            AgentInfo ai = null;
            if ( thisAgentInfo.getName().equals(origin) ) {
                ai = thisAgentInfo;
            } else {
                for (AgentInfo info : apm.listConnectedAgents()) {
                    if (info.getName().equals(origin)) {
                        ai = info;
                        break;
                    }
                }
            }
            if ( ai == null ) {
                log.log(Level.SEVERE,"Could not find connected agent for origin: {0}",origin);
            }
            
            // is this me?
            // FIXME find a better way
            if (origin.contains("MMM")) {
                log.info("Got a raised alert message from "+origin+" -- ignored, source = MMM");
                return;
            }
            MinionGroup g = mmmUtils.getSubsystemGroup(origin);
            try {
                Minion m = mmmUtils.getSubsystemType(origin);
                if ( m != null ) {
                Alert alert = event.getAlert();
                log.info("Got a raised alert message from "+origin);

                RaisedAlertHistory hist = event.getSummary().getRaisedAlert(alert.getAlertId());
                AlertNotification an = ai == null? new AlertNotification(origin, g, m, hist.getLatestAlertCause(), hist.getLatestAlert(),
                        hist.getLatestAlertState(), hist.getLatestAlertCCSTimeStamp().getUTCInstant().toEpochMilli()) :
                        new AlertNotification(ai, g, m, hist.getLatestAlertCause(), hist.getLatestAlert(),
                        hist.getLatestAlertState(), hist.getLatestAlertCCSTimeStamp().getUTCInstant().toEpochMilli());
                notifyObservers(an);
                } else {
                    log.log(Level.WARNING,"Cannot process alert from origin {0} ({1})",new Object[]{origin, event.getAlert()});
                }
            } catch (MMMUtilities.UnknownMinionException e) {
                log.warning("Could not process event "+event, e);
            }
        }
    }

    public static interface Observer {
        public void alertRaised(AlertNotification notif);
    }

    ArrayList<Observer> observers = new ArrayList<>();

    public void addObserver(Observer o) {
        synchronized (updateInProgress) {
            observers.add(o);
        }
    }

    public void deleteObserver(Observer o) {
        synchronized (updateInProgress) {
            observers.remove(o);
        }
    }

    private void notifyObservers(AlertNotification notif) {
        ArrayList<Observer> obs;
        synchronized (updateInProgress) {
            obs = new ArrayList<>(observers);
            obs.forEach((o) -> {
                notifyObserver(o, notif);
            });
        }
    }

    private void notifyObservers(String origin, RaisedAlertSummary summary) {
        ArrayList<Observer> obs;
        synchronized (updateInProgress) {
            obs = new ArrayList<>(observers);
            obs.forEach((o) -> {
                notifyObserver(o, origin, summary);
            });
        }
    }

    private void notifyObserver(Observer o, AlertNotification notif) {
        o.alertRaised(notif);
    }

    private void notifyObserver(Observer o, String origin, RaisedAlertSummary summary) {
        synchronized (updateInProgress) {
            for (RaisedAlertHistory hist : summary.getAllRaisedAlertHistories()) {
                if (hist.getLatestAlertState()!=AlertState.NOMINAL) {
                    try {
                        MinionGroup g = mmmUtils.getSubsystemGroup(origin);
                        Minion m = mmmUtils.getSubsystemType(origin);
                        notifyObserver(o, new AlertNotification(origin, g, m, hist.getLatestAlertCause(),
                                hist.getLatestAlert(), hist.getLatestAlertState(), hist.getLatestAlertCCSTimeStamp().getUTCInstant().toEpochMilli()));
                    } catch (Exception e) {
                        log.warning("Could not process alert notification from "+origin, e);
                    }
                }
            }
        }
    }
}
