package org.lsst.ccs.subsystem.mmm;

import java.io.Serializable;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Predicate;

import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.StatusEnum;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusRaisedAlert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.bus.states.OperationalState;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.AgentPresenceManager;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.messaging.ConcurrentMessagingUtils;
import org.lsst.ccs.messaging.StateBundleAggregator;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.mmm.AlertNotifier.Observer;
import org.lsst.ccs.subsystem.mmm.data.CCSOperationalState;
import org.lsst.ccs.subsystem.mmm.data.CCSSalState;
import org.lsst.ccs.subsystem.mmm.data.CameraOperationalState;
import org.lsst.ccs.subsystem.mmm.data.InvalidStateException;
import org.lsst.ccs.subsystem.mmm.data.OperationTimeoutException;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.subsystem.mmm.alerts.MmmAlerts;
import org.lsst.ccs.subsystem.mmm.data.MMMIR2Event;

public final class MMMUtilities
        implements AgentPresenceListener {

    private ConcurrentMessagingUtils cmu;
    private AgentPresenceManager apm;
    private StateBundleAggregator sba;

    private final AgentStateService stateService;
    private final AlertService alertService;

    // As a subsystem, the MMM has its own set of states, OperationalState for
    // instance.
    // It also publishes an aggregate OperationalState for the whole camera.

    // todo map to states expected by OCS, to be translated by the OCS bridge

    private long defaultTimeout = 1000L;

    private Logger log = Logger.getLogger("org.lsst.ccs.subsystem.mmm");

    private Agent mmm;
    private AgentMessagingLayer agentMessagingLayer;

    private Minion minionInstance = null; // to instantiate EnumMaps
    
    private AlertNotifier an;

    public MMMUtilities(Agent agent) {
        mmm = agent;
        // sa.setAggregate("raftsim/temperature", 10000, 60000);
        stateService = mmm.getAgentService(AgentStateService.class);
        alertService = mmm.getAgentService(AlertService.class);
    }

    
    void init() {
        for (MinionGroup group: groupNameToGroupEnumMap.values()) {
            Alert alert = MmmAlerts.MissingSubsystem.getAlert(group);
            alertService.registerAlert(alert);
        }
    }

    
    void start() {
        agentMessagingLayer = mmm.getMessagingAccess();
        cmu = new ConcurrentMessagingUtils(agentMessagingLayer);
        apm = agentMessagingLayer.getAgentPresenceManager();
        // apm.addAgentPresenceListener(this);
        sba = new StateBundleAggregator(agentMessagingLayer);
        agentMessagingLayer.addStatusMessageListener(sba);
        //The AlertNotifier is added with a filter on null origin to get messages
        //from the MMM itself.
        an = new AlertNotifier(this,mmm);        
    }
    
    void addMinion(MinionGroup g, Minion m, String name) {
        if (minionInstance == null) {
            minionInstance = m;
            groupTypeNameMap = new EnumMap<MinionGroup, EnumMap<Minion, String>>(MinionGroup.class);
        }
        
        String groupName = g.name().toLowerCase();
        if ( ! groupNameToGroupEnumMap.containsKey(groupName) ) {
            groupNameToGroupEnumMap.put(groupName, g);
        }
        
        EnumMap<Minion, String> minionMap = groupTypeNameMap.get(g);
        if (minionMap == null) {
            minionMap = new EnumMap<Minion,String>(m.getDeclaringClass());
            groupTypeNameMap.put(g,minionMap);
        }
        
        minionNameToMinionEnumMap.put(name, m);
        addSubsystemToGroup(g,name);
    }

    void addSubsystemToGroup(MinionGroup group, String subsystemName) {
        EnumMap<Minion,String> minionsInGroup = groupTypeNameMap.get(group);
        try {
            Minion minion = getSubsystemType(subsystemName);
            if ( minion == null ) {
                throw new IllegalArgumentException("Agent "+subsystemName+" cannot be added to group "+group);
            }
            if (!minionsInGroup.containsKey(minion)) {
                //Add this subsystem to the new group.
                minionsInGroup.put(minion, subsystemName);
            } else {
                throw new RuntimeException("Subsystem with type " + minion + " (" + minionsInGroup.get(minion) + ") is already part of this group. It first has to be removed.");
            }

            //Remove the minion from any other group it belongs to
            MinionGroup originalGroup = subsystemNameGroupMap.get(subsystemName);
            if (originalGroup != null) {
                groupTypeNameMap.get(originalGroup).remove(minion);
            }
            subsystemNameGroupMap.put(subsystemName, group);
        } catch (UnknownMinionException e) {
            throw new RuntimeException(e);
        }
    }
    
    public MinionGroup getSubsystemGroup(String subsystemName) {
        return subsystemNameGroupMap.get(subsystemName);
    }
    
    private MinionGroup getGroup(String groupName) {
        MinionGroup gt = groupNameToGroupEnumMap.get(groupName);
        if (gt == null) {
            throw new RuntimeException("Group " + groupName + " is not a valid value.");
        }
        return gt;
    }
    public Minion getSubsystemType(String subsystemName) throws UnknownMinionException {
        Minion mt = minionNameToMinionEnumMap.get(subsystemName);
        if (mt == null) {
            log.warning("Could not find subsystem type for "+subsystemName);
        }
        return mt;
    }
    
    public static class UnknownMinionException extends Exception {

        public UnknownMinionException(String cause) {
            super(cause);
        }
        
    }
    
    @Command(type = Command.CommandType.QUERY)
    public List<String> getAvailableGroups() {
        List<String> result = new ArrayList<>();
        for ( String k : groupNameToGroupEnumMap.keySet() ) {
            result.add(k);
        }
        return result;
    }
    
    @Command(type = Command.CommandType.QUERY)
    public List<String> getSubsystemsInGroup(String group) {
        List<String> result = new ArrayList<>();
        Map<Minion,String> set = groupTypeNameMap.get(getGroup(group));
        for ( Entry<Minion,String> e : set.entrySet() ) {
            result.add(e.getValue());
        }
        return result;
    }
    
    @Command
    public void addSubsystemToGroup(String subsystemName, String group) {
        addSubsystemToGroup(getGroup(group), subsystemName);
        checkPresence();
    }

    @Command
    public void setOptionalSubsystemGroup(String groupName) {
        MinionGroup g = getGroup(groupName);
        EnumMap<Minion,String> map = groupTypeNameMap.get(g);
        if (map != null) {
            for (String subsystemName : map.values()) {
                setOptionalSubsystem(subsystemName,false);
            }
            checkPresence();
        }
    }

    @Command
    public void setRequiredSubsystemGroup(String groupName) {
        MinionGroup g = getGroup(groupName);
        EnumMap<Minion,String> map = groupTypeNameMap.get(g);
        if (map != null) {
            for (String subsystemName : map.values()) {
                setRequiredSubsystem(subsystemName,false);
            }
            checkPresence();
        }
    }

    private void setOptionalSubsystem(String subsystemName, boolean check) {
        optionalMinions.add(subsystemName);
        if ( check ) {
            checkPresence();
        }
    }
    
    private void setRequiredSubsystem(String subsystemName, boolean check) {
        optionalMinions.remove(subsystemName);
        if ( check ) {
            checkPresence();
        }
    }
    
    @Command
    public void setOptionalSubsystem(String subsystemName) {
        setOptionalSubsystem(subsystemName, true);
    }

    @Command
    public void setRequiredSubsystem(String subsystemName) {
        setRequiredSubsystem(subsystemName, true);
    }

    private void updatePresentMinions() {
        for (AgentInfo i : apm.listConnectedAgents()) {
            if ( subsystemNameGroupMap.containsKey(i.getName()) ) {
                presentMinions.add(i.getName());
            }
        }
    }

    public void activate() {
        for (String s : subsystemNameGroupMap.keySet()) {
            sba.addOrigin(s);
        }


        // todo/tocheck we should not too quickly report an error when we miss
        // information
        execute(() -> {
            waitMillis(2000);
            updatePresentMinions();
            apm.addAgentPresenceListener(this);
            checkCamOpState();
            checkPresence();
        });

        sba.addObserver((source, old, change) -> minionStateChange(source, old, change));

        // initEventRules();

    }

    // todo keep a list of the subsystems we are dealing with, along with their
    // name

    private EnumMap<MinionGroup,EnumMap<Minion, String>> groupTypeNameMap = null;
    private final Map<String, Minion> minionNameToMinionEnumMap = new HashMap<String, Minion>();
    private final Map<String, MinionGroup> groupNameToGroupEnumMap = new HashMap<String, MinionGroup>();
    private final Map<String, MinionGroup> subsystemNameGroupMap = new HashMap<>();

//These don't work since Minion is not unique!!!!    
    private final Set<String> optionalMinions = new HashSet<>();
    private final Set<String> presentMinions = new HashSet<>();

    // todo if a required subsystem is not present it should be reflected in our
    // state

    // todo more specific exceptions?

    public Object send(MinionGroup group, Minion dst, String command, Object... parms) throws Exception {
        CommandRequest cmd = new CommandRequest(getDestination(group,dst), command, parms);
        return cmu.sendSynchronousCommand(cmd, Duration.ofMillis(defaultTimeout));
    }

    public Object send(String dst, String command, Object... parms) throws Exception {
        CommandRequest cmd = new CommandRequest(dst, command, parms);
        return cmu.sendSynchronousCommand(cmd, Duration.ofMillis(defaultTimeout));
    }

    public String getDestination(MinionGroup group, Minion minion) {
        return groupTypeNameMap.get(group).get(minion);
    }
    
    /**
     * To send a command which lasts more than defaultTimeout. Used to send
     * command to fcs subsystem (FILTER).
     * 
     * @param dst
     * @param command
     * @param timeout
     * @param parms
     * @return
     * @throws Exception
     */
    public Object sendLongCommand(MinionGroup group, Minion dst, long timeout, String command, Object... parms) throws Exception {
        CommandRequest cmd = new CommandRequest(getDestination(group,dst), command, parms);
        return cmu.sendSynchronousCommand(cmd, Duration.ofMillis(timeout));
    }

    public Object sendLongCommand(String dst, long timeout, String command, Object... parms) throws Exception {
        CommandRequest cmd = new CommandRequest(dst, command, parms);
        return cmu.sendSynchronousCommand(cmd, Duration.ofMillis(timeout));
    }

    public Future<Object> sendAsync(MinionGroup group, Minion dst, String command, Object... parms) {
        CommandRequest cmd = new CommandRequest(getDestination(group,dst), command, parms);
        return cmu.sendAsynchronousCommand(cmd);
    }

    public Future<Object> sendAsync(String dst, String command, Object... parms) {
        CommandRequest cmd = new CommandRequest(dst, command, parms);
        return cmu.sendAsynchronousCommand(cmd);
    }

    private final Map<MinionGroup,Set<Minion>> abortingOnAlarmMinions = new HashMap<>();

    @SafeVarargs
    public final void setAbortingOnAlarmMinions(MinionGroup group, Minion... m) {
        Set<Minion> set = abortingOnAlarmMinions.get(group);
        if ( set == null ) {
            set = new HashSet<>();
            abortingOnAlarmMinions.put(group,set);
        }
        set.clear();
        set.addAll(Arrays.asList(m));
        createAlarmFilter(group);
    }

    private Predicate<BusMessage<? extends Serializable, ?>> alarmFilter;

    private void createAlarmFilter(MinionGroup group) {
        alarmFilter = abortingOnAlarmMinions.get(group).stream()
                .map(mm -> BusMessageFilterFactory.messageOrigin(getDestination(group,mm))).reduce(Predicate::or)
                .orElse(x -> false).and(BusMessageFilterFactory.embeddedObjectClass(Alert.class));
    }


    public ScheduledFuture<?> schedule(Runnable r, Duration delay) {
        return mmm.getScheduler().schedule(r, delay.toMillis(), TimeUnit.MILLISECONDS);
    }

    public Future<?> execute(Runnable r) {
        return mmm.getScheduler().schedule(r,0,TimeUnit.MILLISECONDS);
    }

    @SuppressWarnings({ "rawtypes", "unchecked" })
    public Future<StatusMessage> watchForState(MinionGroup group, Minion sys, Enum state) {
        Predicate<BusMessage<? extends Serializable, ?>> f = BusMessageFilterFactory.messageOrigin(getDestination(group,sys))
                .and(BusMessageFilterFactory.messageClass(StatusMessage.class));

        Predicate<BusMessage<? extends Serializable, ?>> p = (m) -> ((StatusMessage) m).getState().isInState(state);
        f = f.and(p);

        for (Minion m : abortingOnAlarmMinions.get(group)) {
            Predicate<BusMessage<? extends Serializable, ?>> x = BusMessageFilterFactory
                    .embeddedObjectClass(Alert.class).and(BusMessageFilterFactory.messageOrigin(getDestination(group,m)));
            f = f.or(x);
        }

        return cmu.startListeningForStatusBusMessage(f);
    }

    public <MMMIR2State extends Enum<MMMIR2State>> void checkState(MinionGroup g, Minion sys, MMMIR2State state) {
        StateBundle current = sba.getState(getDestination(g,sys));
        if (current == null || !current.isInState(state)) {
            throw new InvalidStateException(current == null ? "null" : current.toString(), " expecting " + state);
        }
    }

    @SafeVarargs
    public final <MMMIR2State extends Enum<MMMIR2State>> void checkState(MinionGroup g, Minion sys, MMMIR2State... state) {
        StateBundle current = sba.getState(getDestination(g,sys));
        if (current != null) {
            for (MMMIR2State s : state) {
                if (current.isInState(s))
                    return;
            }
        }
        StringBuilder expected = new StringBuilder("current state ");
        expected.append(current == null ? "null" : current.toString());
        expected.append(" expecting one of [");
        for (int i = 0; i < state.length; i++) {
            expected.append(state[i].toString());
            if (i < state.length - 1)
                expected.append(", ");
        }
        expected.append("]");
        throw new InvalidStateException(expected.toString());
    }

    public StateBundle getState(MinionGroup g, Minion sys) {
        return sba.getState(getDestination(g,sys));
    }

    public StateBundle getState(String destination) {
        return sba.getState(destination);
    }
    
    public void addCustomStateOrigin(String destination) {
        sba.addOrigin(destination);
    }
    
    @SuppressWarnings("rawtypes")
    public <MMMIR2State extends Enum<MMMIR2State>> void waitForState(MinionGroup g, Minion sys, MMMIR2State state, long timeout) {
        Future<StatusMessage> f = watchForState(g, sys, state);

        StateBundle current = sba.getState(getDestination(g,sys));
        if (current != null && current.isInState(state)) {
            log.info("state for " + sys + " already ok for " + state);
            return;
        }
        log.debug("current state for " + sys + " : " + current);
        try {
            StatusMessage m = f.get(timeout, TimeUnit.MILLISECONDS);
            if (m != null)
                log.debug("received " + m.getState());
            if (m == null) {
                if (sba.getState(getDestination(g,sys)).isInState(state)) {
                    log.info("null future, but in state");
                } else {
                    log.info("null future, not in state");
                    log.info(sba.getState(getDestination(g,sys)));
                    throw new OperationTimeoutException("waiting for state " + state + " on " + sys);
                }
            }
            if (m instanceof StatusRaisedAlert) {
                throw new AlertException("interrupted waitForState", ((StatusRaisedAlert) m).getObject());
            }
        } catch (TimeoutException e) {
            throw new OperationTimeoutException("waiting for state " + state + " on " + sys, e);
        } catch (InterruptedException | ExecutionException e) {
            throw new RuntimeException(e);
        }
    }

    public <MMMIR2State extends Enum<MMMIR2State>> ExpectedStateCombination<MMMIR2State> expectingState(MinionGroup g, Minion m, MMMIR2State state) {
        return new ExpectedStateCombination<MMMIR2State>(g, m, state);
    }

    public final class ExpectedStateCombination<T extends Enum<T>> {
        private Map<Minion, T> states = null;// new EnumMap<Minion,
                                              // Enum>(Minion.class);
        private final MinionGroup group;

        private Map<Minion, T> getStates() {
            if (states == null)
                states = new EnumMap<Minion, T>(minionInstance.getDeclaringClass());
            return states;
        }

        private ExpectedStateCombination(MinionGroup g, Minion m, T state) {
            getStates().put(m, state);
            this.group = g;
        }

        private ExpectedStateCombination(MinionGroup g, ExpectedStateCombination<T> src, Minion m, T state) {
            getStates().putAll(src.getStates());
            getStates().put(m, state);
            this.group = g;
        }

        public ExpectedStateCombination<T> expectingState(MinionGroup g, Minion m, T state) {
            return new ExpectedStateCombination<T>(g, this, m, state);
        }

        private Predicate<BusMessage<? extends Serializable, ?>> filter(Minion sys, T state) {
            return BusMessageFilterFactory.messageOrigin(getDestination(group,sys))
                    .and(m -> ((StatusMessage<?, ?>) m).getState().isInState(state));
        }

        private Predicate<BusMessage<? extends Serializable, ?>> createFilter(Minion sys) {
            return filter(sys, getStates().get(sys)).or(alarmFilter);
        }

        public void waitForAllStatesHappening(long timeout) {

            Set<Minion> waitingFor = new HashSet<Minion>(states.keySet());

            for (Map.Entry<Minion, T> e : states.entrySet()) {
                if (sba.getState(getDestination(group,e.getKey())).isInState(e.getValue())) {
                    waitingFor.remove(e.getKey());
                }
            }

            log.debug("waiting for subsystems " + waitingFor + " timeout " + timeout);

            if (waitingFor.isEmpty()) {
                log.debug("waitForAllStatesHappening: already all in state");
                return;
            }

            int timeoutseconds = timeout > 1000 ? (int) timeout / 1000 : 1;
            // TODO the cmu should have timeouts in milliseconds, or use
            // Duration

            Set<Future<StatusMessage>> futures = new HashSet<Future<StatusMessage>>();
            for (Minion m : waitingFor) {
                Predicate<BusMessage<? extends Serializable, ?>> f = createFilter(m);
                futures.add(cmu.startListeningForStatusBusMessage(f, Duration.ofMillis(timeoutseconds)));
            }

            // futures.stream().parallel().forEach(f -> f.get()); // Damn
            // checked exceptions...

            for (Future<StatusMessage> f : futures) {
                try {
                    /* StatusMessage m = */f.get();
                } catch (InterruptedException | ExecutionException e) {
                    log.warn("interrupted");
                    throw new RuntimeException(e);
                }
            }
        }

        public void waitForAllStates(long timeout) {
            long endTimeout = System.currentTimeMillis() + timeout;
            outer: while (System.currentTimeMillis() < endTimeout) {
                waitForAllStatesHappening(endTimeout - System.currentTimeMillis());

                for (Map.Entry<Minion, T> e : states.entrySet()) {
                    if (!sba.getState(getDestination(group,e.getKey())).isInState(e.getValue()))
                        continue outer;
                }

                return;
            }
            throw new OperationTimeoutException("waitForAllStates");
        }

    }

    public void waitMillis(long millis) {
        try {
            Thread.sleep(millis);
        } catch (InterruptedException e) {
            log.error("wait interrupted ", e);
            throw new RuntimeException(e);
        }
    }

    @Override
    public void connecting(AgentInfo... agents) {
        for (AgentInfo agent : agents) {
            Minion m = minionNameToMinionEnumMap.get(agent.getName());
            if (m != null) {
                presentMinions.add(agent.getName());
                checkPresence();
            }
        }
    }

    @Override
    public void disconnected(AgentInfo... agents) {
        for ( AgentInfo agent : agents ) {
            Minion m = minionNameToMinionEnumMap.get(agent.getName());
            if (m != null) {
                presentMinions.remove(agent.getName());

                // should we clear from the SBA because the missing subsystem state
                // is
                // at best unknown?
                checkPresence();
            }
        }
    }

    private abstract class StateChangeToEventRule {
        protected MMMIR2Event event;

        public StateChangeToEventRule(MMMIR2Event e) {
            event = e;
        }

        public MMMIR2Event getEvent() {
            return event;
        }

        public abstract int matches(StateBundle out, StateBundle changes); // returns
                                                                           // level,
                                                                           // 0
        // for no match
    }

    private class DefaultStateChangeToEventRule<T extends Enum<T>> extends StateChangeToEventRule

    {
        Minion minion;
        Class<T> stateClass;

        public DefaultStateChangeToEventRule(Minion m, Class<T> s, MMMIR2Event e) {
            super(e);
            stateClass = s;
            minion = m;
        }

        @Override
        public int matches(StateBundle out, StateBundle changes) {
            return changes.getState(stateClass) == null ? 0 : 1;
        }
    }

    private class SingleStateChangeToEventRule<T extends Enum<T>> extends StateChangeToEventRule {
        Minion minion;
        T state;

        public SingleStateChangeToEventRule(Minion m, T s, MMMIR2Event e) {
            super(e);
            minion = m;
            state = s;
        }

        @Override
        public int matches(StateBundle out, StateBundle changes) {
            return changes.isInState(state) ? 2 : 0;
        }
    }

    private class SingleStateChangeOutToEventRule<T extends Enum<T>> extends StateChangeToEventRule {
        Minion minion;
        T outState;

        public SingleStateChangeOutToEventRule(Minion m, T out, MMMIR2Event e) {
            super(e);
            minion = m;
            outState = out;
        }

        @Override
        public int matches(StateBundle out, StateBundle changes) {
            return (out.isInState(outState)) ? 2 : 0;
        }
    }

    private class CombinedStatesChangeToEventRule<T extends Enum<T>> extends StateChangeToEventRule {
        private Map<Minion, T> states = null;// new EnumMap<Minion,
        // Enum>(Minion.class);

        private Map<Minion, T> getStates() {
            if (states == null)
                states = new EnumMap<Minion, T>(minionInstance.getDeclaringClass());
            return states;
        }

        public CombinedStatesChangeToEventRule(Minion m, T state, MMMIR2Event e) {
            super(e);
            getStates().put(m, state);
        }

        public CombinedStatesChangeToEventRule<T> addState(Minion m, T state) {
            getStates().put(m, state);
            return this;
        }

        public int matches(StateBundle out, StateBundle changes) {
            throw new RuntimeException("not implemented");
            // need to check the other states in the sba
        }

    }

    List<StateChangeToEventRule> eventRules = new ArrayList<StateChangeToEventRule>();

    public <MMMIR2State extends Enum<MMMIR2State>> void addSingleStateChangeToEventRule(Minion minion, MMMIR2State state,
            MMMIR2Event event) {
        eventRules.add(new SingleStateChangeToEventRule<MMMIR2State>(minion, state, event));
    }

    public <MMMIR2State extends Enum<MMMIR2State>> void addSingleStateChangeOutToEventRule(Minion minion, MMMIR2State state,
            MMMIR2Event event) {
        eventRules.add(new SingleStateChangeOutToEventRule<MMMIR2State>(minion, state, event));
    }

    public <MMMIR2State extends Enum<MMMIR2State>> void addDefaultStateChangeToEventRule(Minion minion, Class<MMMIR2State> stateClass,
            MMMIR2Event event) {
        eventRules.add(new DefaultStateChangeToEventRule<MMMIR2State>(minion, stateClass, event));
    }

    public void checkEventRules(StateBundle out, StateBundle change) {
        MMMIR2Event bestEvent = null;
        int level = 0;
        for (StateChangeToEventRule rule : eventRules) {
            int l = rule.matches(out, change);
            if (l > level) {
                level = l;
                bestEvent = rule.getEvent();
            }
        }
        if (bestEvent != null) {
            log.info("sending MMM event " + bestEvent);
            StatusEnum<MMMIR2Event> message = new StatusEnum<MMMIR2Event>(bestEvent, stateService.getState());
            mmm.getMessagingAccess().sendStatusMessage(message);
        }
    }

    public void minionStateChange(String source, StateBundle out, StateBundle change) {

        log.info("from " + source + " state change " + change);

        if (change.getState(OperationalState.class) != null) {
            checkCamOpState();
        }
        checkEventRules(out, change);

    }

    private CCSOperationalState ccsState = CCSOperationalState.NORMAL;
    private CameraOperationalState camState = CameraOperationalState.NORMAL;
    private CCSSalState salState = CCSSalState.PUBLISH_ONLY;

    private Map<Enum,List<String>> currentlyMissing = new HashMap();
    
    synchronized void checkPresence() {
        CCSOperationalState newS = CCSOperationalState.NORMAL;

        Map<Enum,List<String>> missing = new HashMap<>();

        for (String subsystemName : subsystemNameGroupMap.keySet()) {
            if (optionalMinions.contains(subsystemName))
                continue;
            if (presentMinions.contains(subsystemName))
                continue;
            if ( subsystemName.equals(mmm.getName())) {
                continue;
            }
            newS = CCSOperationalState.MISSING_SUBSYSTEM;
            Enum group = subsystemNameGroupMap.get(subsystemName);
            log.warn("Missing Subsystem " + subsystemName + "  "+group);
            List<String> missingForGroup = missing.get(group);
            if ( missingForGroup == null ) {
                missingForGroup = new ArrayList<>();
                missing.put(group,missingForGroup);
            }
            missingForGroup.add(subsystemName);
            // break;
        }

        if (ccsState != newS || !currentlyMissing.equals(missing)) {
            ccsState = newS;
            stateService.updateAgentState(ccsState);
            log.info("MMM state: " + ccsState);
            
            //Loop over the missing subsystem information to figure out
            //what type of alert to raise
            Map<Enum,List<String>> currentlyMissingCopy = new HashMap<>(currentlyMissing);
            for ( Entry<Enum,List<String>> e : missing.entrySet()) {
                Enum group = e.getKey();                
                List<String> missingSubsystems = e.getValue();
                boolean raiseAlert = false;
                List<String> alreadyMissing = new ArrayList();
                if ( currentlyMissingCopy.containsKey(group) ) {
                    //This group already had missing subsystems
                    alreadyMissing = currentlyMissingCopy.remove(group);
                    if ( ! alreadyMissing.equals( missingSubsystems ) ) {
                        //The list of missing subsystems for this group has changed
                        raiseAlert = true;
                    }
                } else {
                    //This is a new group with missing subsystems
                    raiseAlert = true;
                }
                if ( raiseAlert ) {
                    List<String> connected = new ArrayList(alreadyMissing);
                    connected.removeAll(missingSubsystems);
                    List<String> disconnected = new ArrayList(missingSubsystems);
                    disconnected.removeAll(alreadyMissing);
                    
                    log.warn("MMM raising alert for missing : " + group+ " "+missingSubsystems+" connected: "+connected+" disconnected: "+disconnected);
                    Alert alert = MmmAlerts.MissingSubsystem.getAlert(group, (Serializable)missingSubsystems);
                    StringBuilder cause = new StringBuilder("MMM: Missing subsystem for group ").append(group).append("\n");
                    if ( !disconnected.isEmpty() ) {
                        cause.append("Newly disconnected: ").append(disconnected).append("\n");
                    }
                    if ( !connected.isEmpty() ) {
                        cause.append("Just reconnected: ").append(connected).append("\n");
                    }
                    cause.append("All missing: ").append(missingSubsystems).append("\n");
                    mmm.getAgentService(AlertService.class).raiseAlert(alert, AlertState.ALARM, cause.toString());
                }
            }
            
            //If there are any groups left in the copy of the currently missing
            //subsystems it means that all the subsystems are now back online
            for ( Entry<Enum,List<String>> e : currentlyMissingCopy.entrySet()) {
                Enum group = e.getKey();
                Alert alert = MmmAlerts.MissingSubsystem.getAlert(group,new ArrayList<String>());
                mmm.getAgentService(AlertService.class).raiseAlert(alert, AlertState.NOMINAL, "MMM: All required subsystems are present for group: "+group);
            }
            
            currentlyMissing = missing;
        }
    }

    void checkCamOpState() {
        CameraOperationalState newS = camState;
        for (String mn : subsystemNameGroupMap.keySet()) {
            StateBundle sb = sba.getState(mn);
            if (sb != null) {
                OperationalState s = (OperationalState) sb.getState(OperationalState.class);
                if (s == null) {
                    newS = CameraOperationalState.ENGINEERING_FAULT;
                    break;
                }
                if (s.equals(OperationalState.ENGINEERING_FAULT)) {
                    newS = CameraOperationalState.ENGINEERING_FAULT;
                    break;
                }
                if (s.equals(OperationalState.ENGINEERING_OK)) {
                    newS = CameraOperationalState.ENGINEERING_OK;
                }

            } else {
                newS = CameraOperationalState.ENGINEERING_FAULT;
                break;
            }
        }
        if (!newS.equals(camState)) {
            log.info("camera operational state " + camState + " -> " + newS);
            camState = newS;
            stateService.updateAgentState(camState);
        }
    }

    void checkSalState() {
        CCSSalState newS = salState;

        // should we transition based on commands from OCS only, except for
        // FAULT ?

        if (camState == CameraOperationalState.ENGINEERING_FAULT)
            salState = CCSSalState.FAULT;

        if (!newS.equals(salState)) {
            log.info("camera SAL state " + salState + " -> " + newS);
            salState = newS;
            stateService.updateAgentState(salState);
        }

    }

    // TODO:
    // monitor alarms
    // callback method of MMM on alarm
    // send email. Text message?

    public void addAlertObserver(Observer o) {
        an.addObserver(o);
    }

    public void deleteAlertObserver(Observer o) {
        an.deleteObserver(o);
    }
    
}
