package org.lsst.ccs.subsystem.mmm;

import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledFuture;

import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.messages.StatusEnum;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentCommandDictionaryService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.subsystem.mmm.data.MMMIR2Event;
import org.lsst.ccs.utilities.logging.Logger;

public class GenericMMM
        extends Subsystem implements HasLifecycle {

    protected MMMUtilities mu;

    @ConfigurationParameter(isFinal = true, maxLength = 20)
    protected final Map<MinionGroup,Map<String, Minion>> minions = new HashMap<>();

    @LookupField(strategy = Strategy.TOP)
    Subsystem subsystem;

    @LookupField(strategy = Strategy.CHILDREN)
    protected Map<String, AlertDispatcher> dispatchers = new HashMap<>();

    @LookupField(strategy = Strategy.TREE)
    AgentStateService agentStateService;

    @Override
    public void init() {
        mu = new MMMUtilities(subsystem);

        for ( Entry<MinionGroup,Map<String,Minion>> e : minions.entrySet() ) {
            MinionGroup group = e.getKey();
            for ( Entry<String,Minion> e1 : e.getValue().entrySet() ) {
                mu.addMinion(group, e1.getValue(), e1.getKey());
            }
        }
        subsystem.getAgentService(AgentCommandDictionaryService.class).addCommandSetToObject(mu, subsystem);
        
        mu.init();
    }
        
    @Override
    public void start() {
        mu.start();
        mu.addAlertObserver(this::onAlert);
        mu.activate();
    }

    protected MMMUtilities getMmmUtilities() {
        return mu;
    }

    protected Random random = new Random();
    protected static final Logger log = Logger.getLogger("org.lsst.ccs.subsystem.mmm");

    public GenericMMM() {
        this("genericMMM");
    }
    
    public GenericMMM(String name) {
        super(name,AgentInfo.AgentType.MMM);
    }


    public void onAlert(AlertNotification notif) {
        log.warn(notif+" "+notif.getOrigin()+" "+notif.getSubsystemType()+" "+notif.getSubsystemGroup());
        for ( Entry<String,AlertDispatcher> e : dispatchers.entrySet() ) {
            AlertDispatcher ad = e.getValue();
            boolean affectsDispatcher = false;

            //Check if the message is from the MMM and decide if its group applies to this dispatcher
            if ( notif.getOrigin().equals(subsystem.getName()) ) {
                String group = (String)notif.getAlert().getAlertData("group");
                if ( ad.getGroupName().equals(group) ) {
                    affectsDispatcher = true;
                }
            }
            
            if ( affectsDispatcher || ad.getGroup().equals(notif.getSubsystemGroup())) {
                e.getValue().onAlert(notif);
            }
        }
    }

    @Command
    public String[] getAlertDispatchers() {
        return dispatchers.keySet().toArray(new String[0]);
    }
    
    @Command
    public String status() {
        StringBuilder sb = new StringBuilder();
        sb.append("Dispatchers: \n");
        for ( Entry<String,AlertDispatcher> e : dispatchers.entrySet() ) {
            sb.append("\n");
            sb.append(e.getValue().status("\t"));
        }
        return sb.toString();
    }
    

    public void publishEvent(MMMIR2Event e) {
        StatusEnum<MMMIR2Event> message = new StatusEnum<MMMIR2Event>(e, stateService.getState());
        subsystem.getMessagingAccess().sendStatusMessage(message);
    }

    public Object send(MinionGroup g, Minion dst, String command, Object... parms) throws Exception {
        return mu.send(g, dst, command, parms);
    }

    public Object sendLongCommand(MinionGroup g, Minion dst, long duration, String command, Object... parms) throws Exception {
        return mu.sendLongCommand(g, dst, duration, command, parms);
    }

    public Future<Object> sendAsync(MinionGroup g, Minion dst, String command, Object... parms) {
        return mu.sendAsync(g, dst, command, parms);
    }

    public <MinionMMMIR2State extends Enum<MinionMMMIR2State>> Future<StatusMessage> watchForState(MinionGroup g, Minion sys,
            MinionMMMIR2State state) {
        return mu.watchForState(g, sys, state);
    }

    public <MinionMMMIR2State extends Enum<MinionMMMIR2State>> void waitForState(MinionGroup g, Minion sys, MinionMMMIR2State state, long timeout) {
        mu.waitForState(g, sys, state, timeout);
    }

    public void waitMillis(long millis) {
        mu.waitMillis(millis);
    }

    public <MinionMMMIR2State extends Enum<MinionMMMIR2State>> void checkState(MinionGroup g, Minion sys, MinionMMMIR2State state) {
        mu.checkState(g, sys, state);
    }

    @SafeVarargs
    public final <MinionMMMIR2State extends Enum<MinionMMMIR2State>> void checkState(MinionGroup g, Minion sys, MinionMMMIR2State... state) {
        mu.checkState(g, sys, state);
    }

    public <MMMIR2State extends Enum<MMMIR2State>> MMMUtilities.ExpectedStateCombination<MMMIR2State> expectingState(
            MinionGroup g, Minion m, MMMIR2State state) {
        return mu.expectingState(g, m, state);
    }

    @SafeVarargs
    public final void setAbortingOnAlarmMinions(MinionGroup g, Minion... m) {
        mu.setAbortingOnAlarmMinions(g, m);
    }

    public ScheduledFuture<?> schedule(Runnable r, Duration delay) {
        return mu.schedule(r, delay);
    }

    public Future<?> execute(Runnable r) {
        return mu.execute(r);
    }

}
