package org.lsst.ccs.subsystem.metrology;

import java.io.IOException;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Date;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.TimeUnit;
import javax.usb.UsbException;
import jline.console.ConsoleReader;
import org.lsst.ccs.drivers.ad.Ad7747Eval;
import org.lsst.ccs.utilities.sa.CmndProcess;

/**
 ***************************************************************************
 **
 **  Program to record capacitance data using Analog Devices 7747
 **  evaluation boards
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class CapTest implements CmndProcess.Dispatch {

   /**
    ***************************************************************************
    **
    **  Inner class for running the DAQ on a board.
    **
    ***************************************************************************
    */
    class DaqThread extends Thread {

        private Ad7747Eval ad;
        private int index;

        public DaqThread(Ad7747Eval ad, int index)
        {
            this.ad = ad;
            this.index = index;
        }

        @Override
        public void run()
        {
            while (true) {
                while (true) {
                    try {
                        Thread.sleep(1000000);
                    }
                    catch (InterruptedException e) {
                        break;
                    }
                }
                double vsum = 0.0, vsqsum = 0.0, tsum = 0.0, tsqsum = 0.0,
                       vtsum = 0;
                int count = 0, nexcpt = 0;
                long millis0 = 0;
                for (int j = -nToss; j < daqNsamp; j++) {
                    try {
                        double value, time;
                        long millis;
                        if (daqChan == CHAN_CAP)
                            value = ad.readCapacitance(0);
                        else if (daqChan == CHAN_TEMP)
                            value = ad.readTemperature(0);
                        else if (daqChan == CHAN_VOLT)
                            value = ad.readVoltage(Ad7747Eval.OPTN_EXTERN);
                        else
                            value = ad.readVoltage(0);
                        if (j < 0) continue;   // Ignore first (stale) value
                        millis = System.currentTimeMillis();
                        if (j == 0) millis0 = millis;
                        time = (millis - millis0) / 1000.0;
                        data[index] = value;
                        vsum += value;
                        vsqsum += value * value;
                        tsum += time;
                        tsqsum += time * time;
                        vtsum += value * time;
                        count++;
                    }
                    catch (UsbException e) {
                        if (nexcpt++ == 0) excptn[index] = e;
                    }
                }
                if (count > 0) {
                    double avg = vsum / count;
                    mean[index] = avg;
                    sigma[index] = Math.sqrt(vsqsum / count - avg * avg);
                    error[index] = sigma[index] / count;
                }
                else {
                    mean[index] = 0.0;
                    sigma[index] = 0.0;
                    error[index] = 0.0;
                }
                if (count > 1)
                    drift[index] = (count * vtsum - vsum * tsum)
                                      / (count * tsqsum - tsum * tsum);
                else
                    drift[index] = 0.0;
                nvalue[index] = count;
                nexcptn[index] = nexcpt;
                doneQueue.offer(index);
            }
        }

    }

   /**
    ***************************************************************************
    **
    **  Inner class running a timer task to monitor current DAQ data.
    **
    ***************************************************************************
    */
    private class MonitorData extends TimerTask
    {
        @Override
        public void run()
        {
            if (!monActive) return;
            out.print("\r");
            for (int j = 0; j < boards.size(); j++)
                if ((actvMask & (1 << j)) != 0)
                    out.format("%s %.5g  ", id[j], data[j]);
        }

    }

    /**
    ***************************************************************************
    **
    **  Inner class for detecting console input.
    **
    ***************************************************************************
    */
    class ConsThread extends Thread {

        @Override
        public void run()
        {
            while (true) {
                while (true) {
                    try {
                        Thread.sleep(1000000);
                    }
                    catch (InterruptedException e) {
                        break;
                    }
                }
                try {
                    String line = reader.readLine("");
                }
                catch (IOException e) {
                }
                stopMon = true;
            }
        }

    }

   /**
    ***************************************************************************
    **
    **  Fields.
    **
    ***************************************************************************
    */
    /*
    **  Constants
    */
    private final static int
        CHAN_CAP  = 0,
        CHAN_TEMP = 1,
        CHAN_VOLT = 2,
        CHAN_VDD  = 3,
        MONITOR_PERIOD = 500,
        SEL_NONE = 0x100,
        SEL_ALL  = 0x101,
        SEP_ON   = 1,
        SEP_OFF  = 0;

    private final static String DEFAULT_NAME = "Captest";

    /*
    **  Command codes
    */
    private final static int
        CMD_SAMPLE   = 0,
        CMD_RATE     = 1,
        CMD_GO       = 2,
        CMD_LOG      = 3,
        CMD_DAC      = 4,
        CMD_MONITOR  = 5,
        CMD_LOAD     = 6,
        CMD_SELECT   = 7,
        CMD_SEPARATE = 8,
        NUM_CMDS     = 9;

    /*
    **  Command help text
    */
    private final static String[] helpSample = {
        "Show or set the number of samples to be read each DAQ cycle",
        "sample [<count>]",
        "count  The number of samples to set",
    };

    private final static String[] helpRate = {
        "Show or set the nominal rate at which samples are acquired",
        "rate [<value>]",
        "value  The rate to set (45.5, 41.9, 25.0, 13.2, 8.1, 6.5, 5.5 or 4.6)",
    };

    private final static String[] helpSelect = {
        "Show or select the boards to be used for DAQ",
        "select [<id1> | all | none] [<id2>]... [<id10>]",
        "idn  The id of a board to be used"
    };

    private final static String[] helpSeparate = {
        "Show or set separate DAQ mode",
        "separate [<onoff>]",
        "onoff  If 'on', DAQ is performed separately for each board",
        "       If 'off', DAQ is performed on all selected boards simultaneously",
    };

    private final static String[] helpGo = {
        "Perform a data acquisition cycle",
        "go",
    };

    private final static String[] helpLog = {
        "Close the log file, and optionally set the name for the next one",
        "log  [<name>]",
        "name  The new stem name of the log file: a time stamp is appended",
    };

    private final static String[] helpDac = {
        "Show or set the state of the capacitance offset DACs",
        "dac <id> [<dac>] [<enab>] [<value>]",
        "id     The ID of the board to use, or -1 for all",   
        "dac    The DAC to use, a or b (default is both when showing)",
        "enab   Enable the DAC if non-zero, disable it if zero",
        "value  The value to set, 0 - 63: 1 unit ~= 0.28 pF",
    };

    private final static String[] helpMonitor = {
        "Continuously monitor a sensor on each selected board",
        "monitor [<sensor>]",
        "sensor  The sensor to read: capacitance, temperature, voltage or vdd",
    };

    private final static String[] helpLoad = {
        "Load a board with its firmware",
        "load <board>",
        "board  The index of the board to be loaded",
    };

    /*
    **  Lookup tables
    */
    private final static CmndProcess.Lookup rateNames;
    static {
        rateNames = new CmndProcess.Lookup(8);
        rateNames.add("45.5", Ad7747Eval.CAPFS_RATE_45);
        rateNames.add("41.9", Ad7747Eval.CAPFS_RATE_42);
        rateNames.add("25.0", Ad7747Eval.CAPFS_RATE_25);
        rateNames.add("13.2", Ad7747Eval.CAPFS_RATE_13);
        rateNames.add("8.1",  Ad7747Eval.CAPFS_RATE_8);
        rateNames.add("6.5",  Ad7747Eval.CAPFS_RATE_7);
        rateNames.add("5.5",  Ad7747Eval.CAPFS_RATE_6);
        rateNames.add("4.6",  Ad7747Eval.CAPFS_RATE_5);
    }

    private final static CmndProcess.Lookup chanNames;
    static {
        chanNames = new CmndProcess.Lookup(4);
        chanNames.add("capacitance", CHAN_CAP);
        chanNames.add("temperature", CHAN_TEMP);
        chanNames.add("voltage",     CHAN_VOLT);
        chanNames.add("vdd",         CHAN_VDD);
    }

    private final static CmndProcess.Lookup selNames;
    static {
        selNames = new CmndProcess.Lookup(2);
        selNames.add("all",  SEL_ALL);
        selNames.add("none", SEL_NONE);
    }

    private final static CmndProcess.Lookup sepNames;
    static {
        sepNames = new CmndProcess.Lookup(2);
        sepNames.add("on",  SEP_ON);
        sepNames.add("off", SEP_OFF);
    }

    /*
    **  Command setup
    */
    private final static CmndProcess.Command cmnd;
    static {
        cmnd = new CmndProcess.Command(NUM_CMDS);
        cmnd.add("sample",   CMD_SAMPLE,   helpSample,   "i");
        cmnd.add("rate",     CMD_RATE,     helpRate,     "k",        rateNames);
        cmnd.add("select",   CMD_SELECT,   helpSelect,   "jiiiiiii", selNames);
        cmnd.add("separate", CMD_SEPARATE, helpSeparate, "k",        sepNames);
        cmnd.add("go",       CMD_GO,       helpGo,       "e");
        cmnd.add("log",      CMD_LOG,      helpLog,      "s");
        cmnd.add("dac",      CMD_DAC,      helpDac,      "Isii");
        cmnd.add("monitor",  CMD_MONITOR,  helpMonitor,  "k",        chanNames);
        cmnd.add("load",     CMD_LOAD,     helpLoad,     "Is");
    }

    /*
    **  Private fields
    */
    private final static PrintStream out = System.out;
    private final CmndProcess proc = new CmndProcess();
    private final MonitorData monData = new MonitorData();
    private final Timer monTimer = new Timer();
    private final ConsThread consThread = new ConsThread();
    private ConsoleReader reader;
    private int idMask, nsamp, rate, actvMask, daqChan, daqNsamp, nEvent, nToss;
    private DaqThread[] daqThreads;
    private ArrayList<Ad7747Eval> boards = new ArrayList<Ad7747Eval>();
    private ArrayBlockingQueue<Integer> doneQueue;
    private Exception[] excptn;
    private double[] data, mean, sigma, error, drift;
    private int[] id, nvalue, nexcptn;
    private boolean debug, monActive, sepDaq = true, stopMon;
    private String logName = DEFAULT_NAME;
    private PrintStream logFile;


   /**
    ***************************************************************************
    **
    **  Main constructor.
    **
    ***************************************************************************
    */
    public CapTest(int nsamp, int rate, boolean debug)
    {
        this.nsamp = nsamp;
        this.rate = rate;
        this.debug = debug;
    }


   /**
    ***************************************************************************
    **
    **  Main program.
    **
    ***************************************************************************
    */
    public static void main(String[] args)
    {
        int nsamp = (args.length >= 1) ? Integer.decode(args[0]) : 1;
        if (nsamp <= 0) nsamp = 1;
        int rate = (args.length >= 2) ? rateNames.encode(args[1], true) : 0;
        if (rate < 0) rate = 0;
        boolean debug = (args.length >= 3);

        try {
            (new CapTest(nsamp, rate, debug)).run();
        }
        catch (Exception e) {
            out.println(e);
        }

        System.exit(0);
    }


   /**
    ***************************************************************************
    **
    **  Runs the DAQ program.
    **
    ***************************************************************************
    */
    public void run() throws Exception
    {
        /*
        **  Create console
        */
        reader = new ConsoleReader();

        /*
        **  Find attached boards
        */
        for (int j = 0; true; j++) {
            try {
                boards.add(new Ad7747Eval(j));
            }
            catch (UsbException e) {
                break;
            }
        }
        int nboard = boards.size();
        if (nboard == 0) {
            out.println("No AD7747 boards found");
            return;
        }
        out.println(nboard + " AD7747 board" + (nboard == 1 ? "" : "s")
                      + " found");
        out.format("Sample count = %s, rate = %s Hz\n", nsamp,
                   rateNames.decode(rate));

        /*
        **  Initialize boards
        */
        id = new int[nboard];
        data = new double[nboard];
        mean = new double[nboard];
        sigma = new double[nboard];
        error = new double[nboard];
        drift = new double[nboard];
        nvalue = new int[nboard];
        nexcptn = new int[nboard];
        excptn = new Exception[nboard];
        actvMask = (1 << nboard) - 1;
        doneQueue = new ArrayBlockingQueue<Integer>(nboard);
        daqThreads = new DaqThread[nboard];

        for (int j = 0; j < nboard; j++) {
            Ad7747Eval ad = boards.get(j);
            ad.load(false);
            ad.setupStandard();
            ad.setExcEnabled(false);
            ad.setVtEnabled(true);
            id[j] = readBoardId(ad);
            ad.setConvMode(Ad7747Eval.MODE_CONT);
            daqThreads[j] = new DaqThread(ad, j);
            daqThreads[j].start();
        }
        out.print("Board IDs:");
        for (int j = 0; j < nboard; j++) {
            out.format(" %s", id[j]);
            idMask |= 1 << id[j];
        }
        out.println();

        /*
        **  Start console and monitor threads
        */
        consThread.start();
        monTimer.schedule(monData, 0, MONITOR_PERIOD);

        /*
        **  Command processing
        */
        proc.add(this, cmnd);

        while (true) {
            String line = reader.readLine(">> ");
            if (line == null) break;
            if (!proc.process(line)) break;
        }
    }


   /**
    ***************************************************************************
    **
    **  Dispatches command for processing.
    **
    ***************************************************************************
    */
    @Override
    public boolean dispatch(int code, int found, Object[] args)
    {
        try {
            switch (code) {
            case CMD_GO:
                procGo(found, args); break;
            case CMD_MONITOR:
                procMonitor(found, args); break;
            case CMD_SAMPLE:
                procSample(found, args); break;
            case CMD_RATE:
                procRate(found, args); break;
            case CMD_SELECT:
                procSelect(found, args); break;
            case CMD_SEPARATE:
                procSeparate(found, args); break;
            case CMD_DAC:
                procDac(found, args); break;
            case CMD_LOG:
                procLog(found, args); break;
            case CMD_LOAD:
                procLoad(found, args); break;
            default:
                out.println("Command not fully implemented");
            }
        }
        catch (UsbException e) {
            out.println(e);
        }

        return true;
    }


   /**
    ***************************************************************************
    **
    **  Processes the GO command
    **
    ***************************************************************************
    */
    private void procGo(int found, Object[] args) throws UsbException
    {
        for (int j = 0; j < boards.size(); j++)
            data[j] = 0.0;
        monActive = true;
        nToss = 2;
        for (int j = 0; j < boards.size(); j++) {
            if ((actvMask & (1 << j)) == 0) continue;
            Ad7747Eval ad = boards.get(j);
            ad.setExcEnabled(true);
            ad.setCapEnabled(true);
            ad.setVtEnabled(false);
            ad.setCapConvRate(rate);
            if (sepDaq) {
                doDaq(CHAN_CAP, nsamp, j);
                ad.setExcEnabled(false);
            }
        }
        if (!sepDaq)
            doDaq(CHAN_CAP, nsamp);
        monActive = false;
        out.println();
        logEvent((found & 0x01) != 0 ? (String)args[0] : "");
        for (int j = 0; j < boards.size(); j++) {
            if ((actvMask & (1 << j)) == 0) continue;
            if (!sepDaq)
                boards.get(j).setExcEnabled(false);
            if (nvalue[j] != 0)
                out.format("   %s %.6g %.6g", id[j], mean[j], sigma[j]);
            else
                out.format("   %s --------- -----------", id[j]);
        }
        out.println();
        for (int j = 0; j < boards.size(); j++) {
            if ((actvMask & (1 << j)) == 0) continue;
            if (nexcptn[j] == 0) continue;
            out.format("Board %s had %s USB exceptions.  First was: %s\n",
                       id[j], nexcptn[j], excptn[j]);
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the MONITOR command
    **
    ***************************************************************************
    */
    private void procMonitor(int found, Object[] args) throws UsbException
    {
        int chan = (found & 0x01) != 0 ? (Integer)args[0] : CHAN_CAP;
        for (int j = 0; j < boards.size(); j++) {
            if ((actvMask & (1 << j)) == 0) continue;
            Ad7747Eval ad = boards.get(j);
            if (chan == CHAN_CAP) {
                if (!sepDaq) {
                    ad.setExcEnabled(true);
                    ad.setCapConvRate(7);
                }
                else
                    ad.setCapConvRate(rate);
                ad.setCapEnabled(true);
                ad.setVtEnabled(false);
            }
            else {
                ad.setCapEnabled(false);
                ad.setVtEnabled(true);
                ad.setVtConvRate(3);
                int mode;
                if (chan == CHAN_TEMP)
                    mode = Ad7747Eval.VTMD_INT_TEMP;
                else if (chan == CHAN_VOLT)
                    mode = Ad7747Eval.VTMD_EXT_VOLT;
                else
                    mode = Ad7747Eval.VTMD_VDD_MON;
                ad.setVtMode(mode);
            }
        }
        out.println("Monitoring " + chanNames.decode(chan)
                    + ".  Press <return> to stop.");
        stopMon = false;
        consThread.interrupt();
        int[] nexcpt = new int[boards.size()];
        while (!stopMon) {
            if (chan != CHAN_CAP || !sepDaq) {
                nToss = 0;
                doDaq(chan, 1);
            }
            else {
                nToss = 2;
                for (int j = 0; j < boards.size(); j++) {
                    Ad7747Eval ad = boards.get(j);
                    ad.setExcEnabled(true);
                    doDaq(chan, 1, j);
                    ad.setExcEnabled(false);
                }
            }
            if (stopMon) break;
            out.print("\r");
            for (int j = 0; j < boards.size(); j++) {
                if ((actvMask & (1 << j)) == 0) continue;
                out.format("%s %.5g  ", id[j], mean[j]);
                nexcpt[j] += nexcptn[j];
            }
            if (stopMon) out.println();
        }
        for (int j = 0; j < boards.size(); j++) {
            if ((actvMask & (1 << j)) == 0) continue;
            if (chan == CHAN_CAP && !sepDaq)
                boards.get(j).setExcEnabled(false);
            if (nexcpt[j] != 0)
                out.format("Board %s had %s USB exceptions.  Last was: %s\n",
                           id[j], nexcpt[j], excptn[j]);
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the LOAD command
    **
    ***************************************************************************
    */
    private void procLoad(int found, Object[] args) throws UsbException
    {
        int index = (Integer)args[0];
        Ad7747Eval ad = boards.get(index);
        if ((found & 0x02) == 0)
            ad.load(false);
        else
            ad.load((String)args[1], false);
        ad.setupStandard();
        ad.setVtEnabled(true);
        id[index] = readBoardId(ad);
    }


   /**
    ***************************************************************************
    **
    **  Processes the SAMPLE command
    **
    ***************************************************************************
    */
    private void procSample(int found, Object[] args)
    {
        if ((found & 0x01) == 0) {
            out.println("Sample count = " + nsamp);
        }
        else {
            int value = (Integer)args[0];
            if (value > 0)
                nsamp = value;
            else
                out.println("Invalid sample count");
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the RATE command
    **
    ***************************************************************************
    */
    private void procRate(int found, Object[] args) throws UsbException
    {
        if ((found & 0x01) == 0) {
            out.println("Sampling rate = " + rateNames.decode(rate)
                          + " Hz");
        }
        else {
            rate = (Integer)args[0];
            for (int j = 0; j < boards.size(); j++)
                boards.get(j).setCapConvRate(rate);
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the SELECT command
    **
    ***************************************************************************
    */
    private void procSelect(int found, Object[] args)
    {
        if (found == 0) {
            out.print("Selected boards:");
            if (actvMask == 0)
                out.format(" %s\n", selNames.decode(SEL_NONE));
            else {
                for (int j = 0; j < boards.size(); j++) {
                    if ((actvMask & (1 << j)) != 0)
                        out.format(" %s", id[j]);
                }
                if (actvMask == (1 << boards.size()) - 1)
                    out.format(" (%s)", selNames.decode(SEL_ALL));
                out.println();
            }
        }
        else {
            int mask = 0;
            for (int j = 0; j < 16; j++) {
                if ((found & (1 << j)) == 0) continue;
                int value = (Integer)args[j];
                if (j == 0 && (value == SEL_ALL || value == SEL_NONE)) {
                    if ((found & ~0x01) != 0) {
                        out.println("Too many arguments");
                        mask = -1;
                    }
                    else if (value == SEL_ALL)
                        mask = (1 << boards.size()) - 1;
                    else
                        mask = 0;
                    break;
                }
                if ((idMask & (1 << value)) == 0) {
                    out.println("Invalid board ID: " + value);
                    mask = -1;
                    break;
                }
                mask |= (1 << getBoardIndex(value));
            }
            if (mask >= 0) actvMask = mask;
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the SEPARATE command
    **
    ***************************************************************************
    */
    private void procSeparate(int found, Object[] args)
    {
        if (found == 0)
            out.println("Separate DAQ = "
                          + sepNames.decode(sepDaq ? SEP_ON : SEP_OFF));
        else
            sepDaq = (Integer)args[0] == SEP_ON;
    }


   /**
    ***************************************************************************
    **
    **  Processes the DAC command
    **
    ***************************************************************************
    */
    private void procDac(int found, Object[] args) throws UsbException
    {
        int index = getBoardIndex((Integer)args[0]);
        if (index == -2) return;
        int first = (index == -1) ? 0 : index;
        int last = (index == -1) ? id.length : index + 1;
        int dacs = 3;
        if ((found & 0x02) != 0) {
            String dac = (String)args[1];
            if (dac.equalsIgnoreCase("a"))
                dacs = 1;
            else if (dac.equalsIgnoreCase("b"))
                dacs = 2;
            else {
                out.println("Invalid DAC id");
                return;
            }
        }
        if ((found & 0x0c) == 0) {
            for (int j = first; j < last; j++) {
                Ad7747Eval ad = boards.get(j);
                out.print("ID " + id[j] + ":");
                if ((dacs & 1) != 0) {
                    out.print("  DAC A: "
                              + (ad.isDacAEnabled() ? "En" : "Dis")
                              + "abled, value = " + ad.getDacAValue());
                }
                if ((dacs & 2) != 0) {
                    out.print("  DAC B: "
                              + (ad.isDacBEnabled() ? "En" : "Dis")
                              + "abled, value = " + ad.getDacBValue());
                }
                out.println();
            }
        }
        else {
            if (dacs == 3) {
                out.println("DAC id not specified");
                return;
            }
            for (int j = first; j < last; j++) {
                Ad7747Eval ad = boards.get(j);
                if ((found & 0x04) != 0) {
                    boolean enable = (Integer)args[2] != 0;
                    if (dacs == 1)
                        ad.setDacAEnabled(enable);
                    else
                        ad.setDacBEnabled(enable);
                }
                if ((found & 0x08) != 0) {
                    int value = (Integer)args[3];
                    if (dacs == 1)
                        ad.setDacAValue(value);
                    else
                        ad.setDacBValue(value);
                }
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the LOG command
    **
    ***************************************************************************
    */
    private void procLog(int found, Object[] args)
    {
        if ((found & 0x01) != 0)
            logName = (String)args[0];
        if (nEvent != 0) {
            logFile.close();
            nEvent = 0;
        }
    }


   /**
    ***************************************************************************
    **
    **  Performs a DAQ cycle on a single board.
    **
    ***************************************************************************
    */
    private void doDaq(int chan, int nsamp, int index)
    {
        if ((actvMask & (1 << index)) == 0) return;
        daqChan = chan;
        daqNsamp = nsamp;
        daqThreads[index].interrupt();
        for (int done = 0; done != (1 << index);) {
            try {
                Integer value = doneQueue.poll(10L, TimeUnit.SECONDS);
                if (value != null)
                    done |= (1 << value);
            }
            catch (InterruptedException e) {
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Performs a DAQ cycle on all boards simultaneously.
    **
    ***************************************************************************
    */
    private void doDaq(int chan, int nsamp)
    {
        daqChan = chan;
        daqNsamp = nsamp;
        for (int j = 0; j < boards.size(); j++)
            if ((actvMask & (1 << j)) != 0)
                daqThreads[j].interrupt();
        for (int done = 0; done != actvMask;) {
            try {
                Integer value = doneQueue.poll(10L, TimeUnit.SECONDS);
                if (value != null)
                    done |= (1 << value);
            }
            catch (InterruptedException e) {
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Logs event data.
    **
    ***************************************************************************
    */
    private void logEvent(String comment)
    {
        Date ts = new Date();
        if (nEvent == 0) {
            String
              name = logName + "_"
                       + String.format("%tY%<tm%<td_%<tH%<tM%<tS", ts) + ".dat";
            try {
                logFile = new PrintStream(name);
            }
            catch (IOException e) {
                out.println("Cannot open log file: " + e);
                return;
            }
        }

        logFile.format("%tY-%<tm-%<td\t%<tH:%<tM:%<tS\t%s\t%s",
                       ts, comment, rateNames.decode(rate));
        for (int j = 0; j < id.length; j++)
            logFile.format("\t%s\t%s\t%.6g\t%.6g\t%.6g\t%.6g", id[j],
                           nvalue[j], mean[j], sigma[j], error[j], drift[j]);
        logFile.println("\r");    // Need <cr> for Windows
        nEvent++;
    }


   /**
    ***************************************************************************
    **
    **  Reads a board's ID
    **
    ***************************************************************************
    */
    private int readBoardId(Ad7747Eval ad) throws UsbException
    {
        float value = ad.readVoltage(Ad7747Eval.OPTN_EXTCFG
                                     | Ad7747Eval.OPTN_EXTERN
                                     | Ad7747Eval.OPTN_SINGLE);
        return (int)(10F * value + 0.5F);
    }


   /**
    ***************************************************************************
    **
    **  Gets a board's index given its ID
    **
    ***************************************************************************
    */
    private int getBoardIndex(int bid)
    {
        if (bid == -1) return -1;
        for (int j = 0; j < id.length; j++)
            if (id[j] == bid) return j;
        out.println("Unknown board ID");

        return -2;
    }

}
