package org.lsst.ccs.subsystem.metrology;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.keyence.KeyenceG5001;
import org.lsst.ccs.subsystem.metrology.data.MetrologyState;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.MonitorLogUtils;

/**
 * KeyenceG5001 device class for the metrology subsystem
 *
 * @author: Homer Neal
 */
public class KeyenceG5001Device extends Device implements KeyenceDevice {

    private String devcName = null;
    private int baudRate = 0;

    protected MetrologyState.pwrstates kstate = MetrologyState.pwrstates.NOTCONFIGURED;
    protected boolean failedToInitialize = false;
    protected boolean isconnected = true;
    protected double[] lastRead = new double[]{0, 0, 0, 0, 0, 0, 0, 0};
    private final KeyenceG5001 keydev = new KeyenceG5001();

    /**
     * Initializes the Device instance
     */
    @Override
    protected void initDevice() {
        if (devcName == null) {
            MonitorLogUtils.reportConfigError(log, name, "devcName", "not specified");
        }
        fullName = "KeyenceG5001 module (" + devcName +")";
    }

    /**
     * Initializes the device
     */
    @Override
    protected void initialize() {
        try {
            keydev.openSerial(devcName, baudRate);
            setOnline(true);
            initSensors();
            kstate = MetrologyState.pwrstates.OK;
            log.info("Connected to " + fullName);
        } catch (DriverException e) {
            if (!inited) {
                log.error("Error connecting to " + fullName + ": " + e);
            }
        }
        inited = true;
    }

    /**
     * Closes the connection.
     */
    @Override
    protected void close() {
        try {
            keydev.close();
            kstate = MetrologyState.pwrstates.NOTCONFIGURED;
        } catch (DriverException e) {
            log.error("Error disconnecting from " + fullName + ": " + e);
        }
    }

    /**
     * Checks a channel's parameters for validity.
     * 
     * @param name
     * @param hwChan
     * @param type
     * @param subtype
     * @return Encoded type & subtype
     * @throws Exception
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception {
        if (hwChan < 0 || hwChan >= 2) {
            MonitorLogUtils.reportError(log, name, "hwChan", hwChan);
        }
        return new int[]{0, 0};
    }

    /**
     * Reads a channel.
     *
     * @param chan
     * @param type
     * @return The channel value
     */
    @Override
    protected double readChannel(int chan, int type) {
        double value = 0;
//        log.info("KeyenceG5001Device readChannel called! chan=" + chan + " type=" + type);
//        System.out.println("KeyenceG5001Device readChannel called! chan=" + chan + " type=" + type);
        if (chan == 0) {
            value = lastRead[0];
        }
        if (chan == 1) {
            value = lastRead[1];
        }
        return value;
    }

    /**
     * Reads the distance from the Keyence head
     * 
     * @param ikey
     * @return
     * @throws DriverException
     */
    @Override
    @Command(name = "read", description = "get keyence head measurement")
    public double read(@Argument(name = "ikey", description = "keyence head number") int ikey)
        throws DriverException {
        double[] response = this.readAll();
        return (response[ikey - 1]);
    }

    /**
     * Gets the sampling cycle setting being used for the displacement measurements
     * 
     * @return
     * @throws DriverException
     */
    @Override
    @Command(name = "getcycles", description = "get sampling cycle setting")
    public int getcycles() throws DriverException {
        return keydev.getnsamps();
    }

    /**
     * Sets the sampling cycle mode to be used for the displacement measurements
     * 
     * @param cycles
     * @throws DriverException
     */
    @Override
    @Command(name = "setcycles", description = "set number of cycles")
    public void setcycles(@Argument(name = "cycles", description = "sampling cycle mode") int cycles)
        throws DriverException {
        keydev.setnsamps(cycles);
    }

    /**
     * Gets the sensing mode.
     * 
     * @param ikey
     * @return
     * @throws DriverException
     */
    @Override
    @Command(name = "getmeasmode", description = "get mode")
    public int getmeasmode(@Argument(name = "ikey", description = "keyence head number") int ikey)
        throws DriverException {
        return keydev.getmeasmode(ikey);
    }

    /**
     * Sets the sensing mode.
     * 
     * @param ikey
     * @param mode
     * 0: Normal, 
     * 1: Translucent object,
     * 2: Transparent object, 
     * 3: Transparent object 2, 
     * 4: Semi opaque
     * 
     * @throws DriverException
     */
    @Override
    @Command(name = "setmeasmode", description = "set mode")
    public void setmeasmode(@Argument(name = "ikey", description = "keyence head number") int ikey,
                            @Argument(name = "mode", description = "mode number") int mode
    ) throws DriverException {
        keydev.setmeasmode(ikey, mode);
    }

    /**
     * Sets the minimum display unit.
     * 
     * @param dspu
     * @throws DriverException
     */
    @Override
    @Command(name = "setmindispunit", description = "set mode")
    public void setmindispunit(@Argument(name = "dspu", description = "(0: 0.01 mm, 1: 0.001 mm, 2: 0.0001 mm, 3: 0.00001 mm, 4: 0.1 um, 5: 0.01 um, 6: 0.001 um)") int dspu
    ) throws DriverException {
        keydev.setmindispunit(dspu);
    }

    /**
     * Reads the distances from the Keyence heads
     * 
     * @return
     * @throws DriverException
     */
    @Override
    @Command(name = "readAll", description = "get keyence head measurement")
    public double[] readAll() throws DriverException {
        double[] response = keydev.readDistance();
        System.arraycopy(response, 0, lastRead, 0, response.length);
        return response;
    }

    /**
     * Sets communications mode
     * 
     * @throws DriverException
     */
    @Override
    @Command(description = "communications mode")
    public void commmode() throws DriverException {
        keydev.commmode();
    }

    /**
     * Sets general mode
     * 
     * @throws DriverException
     */
    @Override
    @Command(description = "general mode")
    public void genmode() throws DriverException {
        keydev.genmode();
    }

    /**
     * Gets errors
     * 
     * @return
     * @throws DriverException
     */
    @Override
    @Command(description = "returns any axis fault messages")
    public String getError() throws DriverException {
//        return(keydev.getError());
        return ("not implemented yet");
    }

    /**
     * Turns laser on/off
     * 
     * @param laserstate
     * @throws DriverException
     */
    @Override
    @Command(description = "sets lasers on/off")
    public void laser(@Argument(name = "laserstate", description = "false for laser off / true for on") boolean laserstate) throws DriverException {
        keydev.laser(laserstate);
    }

    /**
     * Generic write operation followed by a read.
     * 
     * @param command
     * @return
     * @throws DriverException
     */
    @Command(description = "generic write", level=10)
    public String keyenceChat(@Argument(name = "command", description = "The raw Keyence command") String command) throws DriverException {
        return keydev.read(command);
    }

    @Override
    public double[] getLastRead() {
        return lastRead;
    }

    @Command(name = "setstate", description = "set KeyenceG5001 device status")
    public void setState(int istate) {
        kstate = MetrologyState.pwrstates.values()[istate];
    }

    @Command(name = "getstate", description = "get KeyenceG5001 device status")
    public int getState() {
        return (kstate.ordinal());
    }

}
