package org.lsst.ccs.subsystem.metrology;

import java.util.Map;
import java.util.HashMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.aerotech.AerotechPro165;
import org.lsst.ccs.drivers.commons.DriverLockedException;
import org.lsst.ccs.subsystem.metrology.data.MetrologyState;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.subsystem.monitor.Device;
import org.lsst.ccs.subsystem.monitor.Monitor;
import org.lsst.ccs.subsystem.monitor.Channel;

/**
 ***************************************************************************
 **
 * AerotechP165 device class for the metrology subsystem
 *
 * * @author: Homer Neal
 * **************************************************************************
 */
public class AerotechP165Device extends Device implements AerotechDevice {

    private final static Map<String, Integer> typeMap = new HashMap<>();
    static {
        typeMap.put("POSITION", Channel.TYPE_UNKNOWN);
    }

    private int last_state = -1; // last channel state seen
    private int nRead = 0, nByte = 0, nTimeout = 0;
    private boolean open = false;                // are we logged in and able to talk
    private boolean isconnected = false; // is the device defined

    AerotechPro165 devaero = null;

    String aerotech_host = null;

    int aerotech_port = 8000;

    MetrologyState.pwrstates kstate = MetrologyState.pwrstates.NOTCONFIGURED;

    boolean failedToInitialize = false;
    private double lastPosx, lastPosy, lastPosz;

    /**
     ***************************************************************************
     **
     ** AerotechP165Device constructor - put AerotechP165 in initial subsystem
     * state
     *
     ** @param host device address
     ** @param port port or baud
     *
     * @throws org.lsst.ccs.drivers.commons.DriverException
     * **************************************************************************
     */
    public AerotechP165Device(String host, int port)
            throws DriverException {
        System.out.println("In Aerotech initialization class");
        isconnected = true;
        aerotech_host = host;
        aerotech_port = port;

        try {
            devaero = new AerotechPro165();
            devaero.opennet(host, port);
            System.out.println("Openned connection to Aerotech!");
        } catch (DriverException f) {
            System.out.println("Failed to open connection to AerotechP165 device!");
            isconnected = false;
        }

    }

    /**
     ***************************************************************************
     **
     ** Constructor *
     * **************************************************************************
     */
    public AerotechP165Device() {
//        isconnected = true;
        try {
            devaero = new AerotechPro165();
        } catch (DriverException f) {
            System.out.println("Failed to create to AerotechP165 device instance!");
        }

    }

    /**
     ***************************************************************************
     **
     ** Performs configuration.
     *
     *
     * @param mon
     * **************************************************************************
     */
    @Override
    protected void configure(Monitor mon) {
        super.configure(mon);
        fullName = "AerotechP165 module";
    }

    /**
     ***************************************************************************
     **
     ** Closes the connection. *
     * **************************************************************************
     */
    @Override
    protected void close() {
        try {
            if (devaero != null) {
                devaero.close();
            }
            kstate = MetrologyState.pwrstates.NOTCONFIGURED;
        } catch (DriverException e) {
            log.error("AerotechP165 device failed to close!!!");
        }
    }

    /**
     ***************************************************************************
     **
     ** Initializes the device
     * **************************************************************************
     */
    @Override
    protected void initialize() {
        System.out.println("aerotechP165Device: initialize called");
        if (isconnected) {
            log.debug("AerotechP165 device is already initialized");
            /*
             try {
             log.debug(devaero.getVersion());
             } catch (Exception e) {
             if (!failedToInitialize) {
             log.error("AerotechP165 device failed to respond to ident request! The initialization has FAILED.");
             failedToInitialize = true;
             }
             isconnected = false;
             }
             */
            log.info("Setting device online.");
            System.out.println("aerotech: setting device online");
            setOnline(true);
            kstate = MetrologyState.pwrstates.OK;
        } else {
            if (!failedToInitialize) {
                log.error("Tried to initialize unconnected PDU device!");
                failedToInitialize = true;
            }

        }
    }

    /**
     ***************************************************************************
     **
     ** gets version of the Ensemble controller *
     * **************************************************************************
     */
    @Command(description = "return controller version")
    public String getVersion() throws DriverException {
        return devaero.getVersion();
    }

    /**
     ***************************************************************************
     **
     ** abort any motion
     * **************************************************************************
     */
    @Command(description = "abort motion")
    public void abort() throws DriverException {
        devaero.abort();
    }

    /**
     ***************************************************************************
     **
     ** enable axes 
     * **************************************************************************
     */
    @Command(description = "enable axes")
    public void enableAxes() throws DriverException {
        log.info("Enabling X axis");
        devaero.enableAxis('X');
        Sleep(1.0);
       log.info("Enabling Y axis");
        devaero.enableAxis('Y');
        Sleep(1.0);
        log.info("Enabling Z axis");
        devaero.enableAxis('Z');
    }

    /**
     ***************************************************************************
     **
     ** disable axes
     * **************************************************************************
     */
    @Command(description = "disable axes")
    public void disableAxes() throws DriverException {
        log.info("Disabling X axis");
        devaero.disableAxis('X');
        Sleep(1.0);
        log.info("Disabling Y axis");
        devaero.disableAxis('Y');
        Sleep(1.0);
        log.info("Disabling Z axis");
        devaero.disableAxis('Z');
    }

    /**
     ***************************************************************************
     **
     ** Sleep - what a waste
     * **************************************************************************
     */
    public void Sleep(double secs) {
        try {
            long millis = (long) (secs * 1000);
            log.info("Sleeping for " + millis + " millis");
            Thread.sleep(millis);
        } catch (InterruptedException ex) {
            log.error("Rude awakening!" + ex);
        }
    }

    /**
     ***************************************************************************
     **
     ** setSpeed: step/s
     *
     * **************************************************************************
     * @param velocity
     * @throws org.lsst.ccs.drivers.commons.DriverException
     */
    @Command(description = "setSpeed")
    public void setSpeed(@Argument(name = "velocity", description = "step / s") double velocity) throws DriverException {
 //       devaero.rampMode("RATE");
        devaero.setSpeed(velocity);
    }

    /**
     ***************************************************************************
     **
     ** rampRate: step/s^2
     *
     * **************************************************************************
     * @param rate
     * @throws org.lsst.ccs.drivers.commons.DriverException
     */
    @Command(description = "rampRate")
    public void rampRate(@Argument(name = "rate", description = "step / s2") double rate) throws DriverException {
        devaero.rampMode("RATE");
        devaero.rampRate(rate);
    }

    /**
     ***************************************************************************
     **
     ** moveInc_x:
     ** perform incremental move
     *
     * **************************************************************************
     * @param delta
     * @throws org.lsst.ccs.drivers.commons.DriverException
     */
    @Command(description = "move incrementally in x")
    public void moveInc_x(@Argument(name = "dx", description = "amount to move in user units") double delta
    ) throws DriverException {
        devaero.moveInc_x(delta);
    }

    /**
     ***************************************************************************
     **
     ** *incremental move
     * **************************************************************************
     */
    @Command(description = "move incrementally in y")
    public void moveInc_y(@Argument(name = "dy", description = "amount to move in user units") double delta
    ) throws DriverException {
        devaero.moveInc_y(delta);
    }

    /**
     ***************************************************************************
     **
     ** *incremental move
     * **************************************************************************
     */
    @Command(description = "move incrementally in z")
    public void moveInc_z(@Argument(name = "dz", description = "amount to move in user units") double delta
    ) throws DriverException {
        devaero.moveInc_z(delta);
    }

    /**
     ***************************************************************************
     **
     ** *incremental move in x and y
     * **************************************************************************
     */
    @Command(description = "move incrementally in x and y")
    public void moveInc_xy(@Argument(name = "x", description = "amount to move in x in user units") double dx,
            @Argument(name = "y", description = "amount to move in y in user units") double dy) throws DriverException {
        devaero.moveInc(dx, dy);
    }

    /**
     ***************************************************************************
     ** moveInx_xyz:
     ** incremental move in , y and z
     * **************************************************************************
     * @param dx
     * @param dy
     * @param dz
     * @throws org.lsst.ccs.drivers.commons.DriverException
     */
    @Command(description = "move incrementally in x, y and z")
    public void moveInc_xyz(@Argument(name = "x", description = "amount to move in x in user units") double dx,
            @Argument(name = "y", description = "amount to move in y in user units") double dy,
            @Argument(name = "z", description = "amount to move in z in user units") double dz
    ) throws DriverException {
        devaero.moveInc(dx, dy, dz);
    }

    /**
     ***************************************************************************
     **
     ** *absolute move
     * **************************************************************************
     */
    @Command(description = "move absolutely in x")
    public void moveAbs_x(@Argument(name = "dx", description = "amount to move in user units") double delta
    ) throws DriverException {
        devaero.moveAbs_x(delta);
    }

    /**
     ***************************************************************************
     **
     ** *absolute move
     * **************************************************************************
     */
    @Command(description = "move absolutely in y")
    public void moveAbs_y(@Argument(name = "dy", description = "amount to move in user units") double delta
    ) throws DriverException {
        devaero.moveAbs_y(delta);
    }

    /**
     ***************************************************************************
     **
     ** *absolute move
     * **************************************************************************
     */
    @Command(description = "move absolutely in z")
    public void moveAbs_z(@Argument(name = "dz", description = "amount to move in user units") double delta
    ) throws DriverException {
        devaero.moveAbs_z(delta);
    }

    /**
     ***************************************************************************
     **
     ** *absolute move in x and y
     * **************************************************************************
     */
    @Command(description = "move absolutely in x and y")
    public void moveAbs_xy(@Argument(name = "x", description = "amount to move in x in user units") double dx,
            @Argument(name = "y", description = "amount to move in y in user units") double dy) throws DriverException {
        devaero.moveAbs(dx, dy);
    }

    /**
     ***************************************************************************
     **
     ** *absolute move in , y and z
     * **************************************************************************
     */
    @Command(description = "move absolutely in x, y and z")
    public void moveAbs_xyz(@Argument(name = "x", description = "amount to move in x in user units") double dx,
            @Argument(name = "y", description = "amount to move in y in user units") double dy,
            @Argument(name = "z", description = "amount to move in z in user units") double dz
    ) throws DriverException {
        devaero.moveAbs(dx, dy, dz);
    }

    /**
     ***************************************************************************
     **
     ** get xyz position
     * **************************************************************************
     */
    @Command(description = "returns position in x, y and z")
    public double[] getPos_xyz() throws DriverException {
/*
        double p[] = devaero.getPos_xyz();
*/
        double p[] = new double[3];
        p[0] = this.getPos_x();
        p[1] = this.getPos_y();
        p[2] = this.getPos_z();

        lastPosx = p[0];
        lastPosy = p[1];
        lastPosz = p[2];

        System.out.println("lastPosx = "+lastPosx);
        System.out.println("lastPosy = "+lastPosy);
        System.out.println("lastPosz = "+lastPosz);
        return (p);
    }

    /**
     ***************************************************************************
     **
     ** get x position
     * **************************************************************************
     */
    @Command(description = "returns position in x")
    public double getPos_x() throws DriverException {
        double xmeas = Double.NaN;
        while (Double.isNaN(xmeas)) {
            xmeas = devaero.getPos_x();
            if (Double.isNaN(xmeas)) log.info("Waiting for x position value to become available");
        }
        lastPosx = xmeas;
        System.out.println("lastPosx = "+lastPosx);
        return (lastPosx);
    }

    /**
     ***************************************************************************
     **
     ** get y position
     * **************************************************************************
     */
    @Command(description = "returns position in y")
    public double getPos_y() throws DriverException {
        double ymeas = Double.NaN;
        while (Double.isNaN(ymeas)) {
            ymeas = devaero.getPos_y();
            if (Double.isNaN(ymeas)) log.info("Waiting for y position value to become available");
        }
        lastPosy = ymeas;
        System.out.println("lastPosy = "+lastPosy);
        return (lastPosy);
    }

    /**
     ***************************************************************************
     **
     ** get z position
     * **************************************************************************
     */
    @Command(description = "returns position in z")
    public double getPos_z() throws DriverException {
        double zmeas = Double.NaN;
        while (Double.isNaN(zmeas)) {
            zmeas = devaero.getPos_z();
            if (Double.isNaN(zmeas)) log.info("Waiting for z position value to become available");
        }
        lastPosz = zmeas;
        System.out.println("lastPosz = "+lastPosz);
        return (lastPosz);
    }

    /**
     ***************************************************************************
     **
     ** return last x position
     * **************************************************************************
     */
    public double getLastPos_x() {
        return lastPosx;
    }

    /**
     ***************************************************************************
     **
     ** return last y position
     * **************************************************************************
     */
    public double getLastPos_y() {
        return lastPosy;
    }

    /**
     ***************************************************************************
     **
     ** return last z position
     * **************************************************************************
     */
    public double getLastPos_z() {
        return lastPosz;
    }


    /**
     ***************************************************************************
     **
     ** get errors
     * **************************************************************************
     */
    @Command(description = "returns any axis fault messages")
    public String getError() throws DriverException {
        return (devaero.getError());
    }

    /**
     ***************************************************************************
     **
     ** go to home position
     * **************************************************************************
     */
    @Command(description = "goes to home position")
    public void goHome() throws DriverException {
        devaero.goHome();
        return;
    }

    /**
     ***************************************************************************
     **
     ** reset
     * **************************************************************************
     */
    @Command(description = "clears all fault mesdsages")
    public void reset() throws DriverException {
        log.error("trying to reopen the connection");
        try {
            devaero.opennet(aerotech_host, aerotech_port);
        } catch (DriverException ex) {
            log.error("failed attempt to reopen connection");
        }
        devaero.reset();
        return;
    }

    /**
     ***************************************************************************
     **
     ** Checks a channel's parameters for validity. *
     * **************************************************************************
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type,
            String subtype)
            throws Exception {
        Integer iType = typeMap.get(type.toUpperCase());
        /*
         if (iType == null) {
         mon.reportError(name, "type", type);
         } else if (iType != Channel.TYPE_SWITCH) {
         mon.reportError(name, "Wrong channel type specified! type = ", type);
         Exception e;
         }
         */
        return new int[]{iType, 0};
    }

    /**
     ***************************************************************************
     **
     ** Initializes a channel. *
     * **************************************************************************
     */
    @Override
    protected void initChannel(int chan, int type, int subtype) {
        try {
//            if (type == Channel.TYPE_SWITCH) {
            log.info("setting channel online - chan=" + chan);
            System.out.println("setting channel online - chan=" + chan);
            setOnline(true);
//            }
        } catch (Exception e) {
            log.error("Error configuring channel type " + type + ": " + e);
        }
    }

    /**
     ***************************************************************************
     **
     ** Reads a channel. *
     * **************************************************************************
     */
    @Override
    protected double readChannel(int chan, int type) {
        double value = 0;
//        log.info("AerotechP165Device readChannel called! chan=" + chan + " type=" + type);
//        System.out.println("AerotechP165Device readChannel called! chan=" + chan + " type=" + type);

//        try {
            switch (chan) {
                case 0: // pos_x
                    value = lastPosx;
                    break;
                case 1: // pos_y
                    value = lastPosy;
                    break;
                case 2: // pos_z
                    value = lastPosz;
                    break;
            }
//                oldval[chan+2] = value;
            /*
        } catch (DriverLockedException f) {
            System.out.print("A"); // just to indicate that we are waiting
        } catch (DriverException e) {
            log.error("Error reading channel type " + chan + ": " + e);
            log.error("trying to reopen the connection");
            try {
                devaero.opennet(aerotech_host, aerotech_port);
            } catch (DriverException ex) {
                log.error("failed attempt to reopen connection");
            }

        }
*/
        return (value);
    }

    @Command(name = "setstate", description = "set AerotechP165 device status")
    public void setState(int istate) {
        kstate = MetrologyState.pwrstates.values()[istate];
    }

    @Command(name = "getstate", description = "get AerotechP165 device status")
    public int getState() {
        return (kstate.ordinal());
    }

}
