package org.lsst.ccs.subsystem.mcm;

import java.time.Duration;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.subsystem.mcm.data.InvalidStateException;
import org.lsst.ccs.subsystem.mcm.data.MCMEvent;
import org.lsst.ccs.subsystem.mcm.data.MCMState;
import org.lsst.ccs.subsystem.mcm.data.OperationTimeoutException;
import org.lsst.ccs.subsystem.mcm.data.RaftState;
import org.lsst.ccs.subsystem.mcm.data.ShutterState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterReadinessState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterState;

// TODO handle alarm while we are waiting for a task
// => abort? 

// TODO easy way to configure the aggregated status sent to OCS bridge
// (and trending)

// TODO camera state = logic on individual states

// TODO update for new calibration commands. Accept them while integrating.

public class MCM extends GenericMCM<Minion, MinionGroup, MCMEvent, MCMState, MCM.MCMCommand> {

    @Override
    protected void initMCM() {

        setState(MCMState.IDLE);

        // init : installer un listener qui va transformer de manière continue
        // les événements en MCMEvent, plutôt que de le faire dans les méthodes.
        // ensemble de règles, quand bundle simultané => état publié.

        mu.addSingleStateChangeToEventRule(Minion.FILTER, FilterState.ONLINE_U, MCMEvent.filterULoaded);
        mu.addSingleStateChangeToEventRule(Minion.FILTER, FilterState.ONLINE_G, MCMEvent.filterGLoaded);
        mu.addSingleStateChangeToEventRule(Minion.FILTER, FilterState.ONLINE_R, MCMEvent.filterRLoaded);
        mu.addSingleStateChangeToEventRule(Minion.FILTER, FilterState.ONLINE_I, MCMEvent.filterILoaded);
        mu.addSingleStateChangeToEventRule(Minion.FILTER, FilterState.ONLINE_Z, MCMEvent.filterZLoaded);
        mu.addSingleStateChangeToEventRule(Minion.FILTER, FilterState.ONLINE_Y, MCMEvent.filterYLoaded);
        mu.addSingleStateChangeToEventRule(Minion.FILTER, FilterState.ONLINE_NONE, MCMEvent.filterNoneLoaded);
        mu.addDefaultStateChangeToEventRule(Minion.FILTER, FilterState.class, MCMEvent.filterSystemMoving);
        mu.addSingleStateChangeToEventRule(Minion.RAFTS, RaftState.QUIESCENT, MCMEvent.ccdCleared);
        mu.addSingleStateChangeToEventRule(Minion.RAFTS, RaftState.INTEGRATING, MCMEvent.startIntegration);
        mu.addSingleStateChangeToEventRule(Minion.RAFTS, RaftState.READING_OUT, MCMEvent.startReadout);
        mu.addSingleStateChangeOutToEventRule(Minion.RAFTS, RaftState.READING_OUT, MCMEvent.endReadout);
        mu.addSingleStateChangeToEventRule(Minion.RAFTS, RaftState.NEEDS_CLEAR, MCMEvent.ccdNotReady);

        mu.addSingleStateChangeToEventRule(Minion.SHUTTER, ShutterState.OPENING, MCMEvent.startShutterOpen);
        mu.addSingleStateChangeToEventRule(Minion.SHUTTER, ShutterState.OPEN, MCMEvent.endShutterOpen);
        mu.addSingleStateChangeToEventRule(Minion.SHUTTER, ShutterState.CLOSING, MCMEvent.startShutterClose);
        mu.addSingleStateChangeToEventRule(Minion.SHUTTER, ShutterState.CLOSED, MCMEvent.endShutterClose);

    }

    // todo abort, stop

    // allowed commands

    public enum MCMCommand {
        INITIMAGES, CONFIGURECAMERA, SETFILTER, TAKEIMAGES, INITGUIDER, CLEAR, STARTIMAGE, ENDIMAGE, DISCARDROWS
    };

    // IDLE, TAKINGIMAGES, SETTINGFILTER, PREPARING, IMAGESTARTED, CONFIGURING

    protected void initAllowedTransitions() {
        Set<MCMCommand> s = new HashSet<MCMCommand>();
        s.addAll(Arrays.asList(MCMCommand.values()));
        s.remove(MCMCommand.DISCARDROWS);
        s.remove(MCMCommand.ENDIMAGE);
        allowedTransition.put(MCMState.IDLE, s);

        s = new HashSet<MCMCommand>();
        s.add(MCMCommand.INITGUIDER);
        allowedTransition.put(MCMState.TAKINGIMAGES, s);

        s = new HashSet<MCMCommand>();
        s.add(MCMCommand.INITIMAGES);
        allowedTransition.put(MCMState.SETTINGFILTER, s);

        s = new HashSet<MCMCommand>();
        // nothing?
        allowedTransition.put(MCMState.PREPARING, s);

        s = new HashSet<MCMCommand>();
        s.add(MCMCommand.INITGUIDER);
        s.add(MCMCommand.ENDIMAGE);
        s.add(MCMCommand.DISCARDROWS);
        allowedTransition.put(MCMState.IMAGESTARTED, s);

        s = new HashSet<MCMCommand>();
        // nothing?
        allowedTransition.put(MCMState.CONFIGURING, s);
    }

    @Override
    protected Class<MCMState> getStateClass() {
        return MCMState.class;
    }

    @Command
    public void initImages(long delay) {
        checkCommandValidity(MCMCommand.INITIMAGES);
        setState(MCMState.PREPARING);

        doInitImages(delay);
        setState(MCMState.IDLE);
    }

    protected void doInitImages(long delay) {
        setAbortingOnAlarmMinions(MinionGroup.CAMERA);

        if (delay < 70) {
            log.info("sending clear to rafts");
            sendAsync(MinionGroup.CAMERA, Minion.RAFTS, "clear");
            log.info("sending prepare to shutter");
            sendAsync(MinionGroup.CAMERA, Minion.SHUTTER, "prepare");

            // not waiting any more, we will publish events when this happens.

            // log.info("waiting for RAFTS quiescent and shutter READY");
            // try {
            // int sel = random.nextInt() % 4;
            // sel = 3;
            // switch (sel) {
            // case 0:
            // waitForState(Minion.SHUTTER, ShutterReadinessState.READY, 200);
            // waitForState(Minion.RAFTS, RaftState.QUIESCENT, 200);
            // break;
            // case 1:
            // waitForState(Minion.RAFTS, RaftState.QUIESCENT, 200);
            // waitForState(Minion.SHUTTER, ShutterReadinessState.READY, 200);
            // break;
            // case 2:
            // expectingState(Minion.SHUTTER, ShutterReadinessState.READY)
            // .expectingState(Minion.RAFTS,
            // RaftState.QUIESCENT).waitForAllStatesHappening(200);
            // break;
            // case 3:
            // expectingState(Minion.SHUTTER, ShutterReadinessState.READY)
            // .expectingState(Minion.RAFTS,
            // RaftState.QUIESCENT).waitForAllStates(200);
            // break;
            //
            // }
            //
            // } catch (OperationTimeoutException e) {
            // log.error("timeout while waiting", e);
            // throw new RuntimeException(e);
            // }
            // log.info("Raft and shutter ready");
        } else {
            schedule(() -> sendAsync(MinionGroup.CAMERA, Minion.RAFTS, "clear"), Duration.ofMillis(delay - 70));
            if (delay < 150) {
                sendAsync(MinionGroup.CAMERA, Minion.SHUTTER, "prepare");
            } else {
                schedule(() -> sendAsync(MinionGroup.CAMERA, Minion.SHUTTER, "prepare"), Duration.ofMillis(delay - 150));
            }

            // should we wait?? NO
            // try {
            // waitForState(Minion.RAFTS, RaftState.QUIESCENT, delay + 200);
            // waitForState(Minion.SHUTTER, ShutterReadinessState.READY, delay +
            // 200);
            // } catch (OperationTimeoutException e) {
            // log.error("timeout while waiting", e);
            // throw new RuntimeException(e);
            // }
            // log.info("Raft and shutter ready");

        }
    }

    @Command
    public void configureCamera(String configId) {
        checkCommandValidity(MCMCommand.CONFIGURECAMERA);
        setState(MCMState.CONFIGURING);

        Future<?> e1 = execute(() -> {
            log.info("e1 - start");
            waitMillis(1000);
            log.info("e1 - end");
        });
        Future<?> e2 = execute(() -> {
            log.info("e2 - start");
            waitMillis(1500);
            log.info("e2 - end");
        });

        try {
            e2.get();
            e1.get();
        } catch (InterruptedException | ExecutionException e) {
            log.error(e);
        }

        log.info("done");
        setState(MCMState.IDLE);
    }

    @Command
    public void setFilter(String filterName) {
        setFilterSync(filterName);
    }

    // WORK IN PROGRESS
    @Command
    public void setFilterAsync(String filterName) {
        checkCommandValidity(MCMCommand.SETFILTER);
        setAbortingOnAlarmMinions(MinionGroup.CAMERA, Minion.FILTER);
        filterName = filterName.toUpperCase();
        FilterState goal = "NONE".equals(filterName) ? FilterState.ONLINE_NONE
                : FilterState.valueOf("ONLINE_" + filterName);

        if (isInState(MinionGroup.CAMERA, Minion.FILTER, goal)) {
            log.info("MCM : filter already ok for " + filterName);
            return;
        }

        setState(MCMState.SETTINGFILTER);

        log.info("MCM : sending command to fcs : setFilter filter" + filterName);
        sendAsync(MinionGroup.CAMERA, Minion.FILTER, "setFilter", filterName);

        execute(() -> {
            try {
                waitForState(MinionGroup.CAMERA, Minion.FILTER, goal, 60000);
            } catch (OperationTimeoutException e) {
                log.error("timeout while waiting", e);
                throw e;
            } catch (AlertException e) {
                log.error("alert occurred - cause= " + e.alert.getDescription());
                // do not wrap AlarmException, local
                throw new RuntimeException("alert occurred " + e.getMessage() + " while setFilter"
                        + " Alert description=" + e.alert.getDescription());
            } catch (Exception e) {
                log.error("error", e);
                throw new RuntimeException("error ", e);
            } finally {
                setState(MCMState.IDLE);
            }
        });

    }

    @Command
    public void setFilterSync(String filterName) {
        checkCommandValidity(MCMCommand.SETFILTER);

        // check other conditions?

        setAbortingOnAlarmMinions(MinionGroup.CAMERA, Minion.FILTER);

        filterName = filterName.toUpperCase();
        FilterState goal = "NONE".equals(filterName) ? FilterState.ONLINE_NONE
                : FilterState.valueOf("ONLINE_" + filterName);

        if (isInState(MinionGroup.CAMERA, Minion.FILTER, goal)) {
            log.info("MCM : filter already ok for " + filterName);
            return;
        }

        setState(MCMState.SETTINGFILTER);

        log.info("MCM : sending command to fcs : setFilter filter" + filterName);

        try {
            sendLongCommand(MinionGroup.CAMERA, Minion.FILTER, 60000L, "setFilter", filterName);
            waitForState(MinionGroup.CAMERA, Minion.FILTER, goal, 60000);
        } catch (OperationTimeoutException e) {
            log.error("timeout while waiting", e);
            throw e;
        } catch (AlertException e) {
            log.error("alert occurred - cause= " + e.alert.getDescription());
            // do not wrap AlarmException, local
            throw new RuntimeException("alert occurred " + e.getMessage() + " while setFilter" + " Alert description="
                    + e.alert.getDescription());
        } catch (Exception e) {
            log.error("error", e);
            throw new RuntimeException("error ", e);
        } finally {
            setState(MCMState.IDLE);
        }
        log.info("filter " + filterName + " is ONLINE.");
    }

    @Command
    public void takeImages(int n, int exposureMillis, boolean openShutter, boolean scienceActive, boolean guidingActive,
            boolean WFSActive, String seqName) {
        takeImagesSync(n, exposureMillis, openShutter, scienceActive, guidingActive, WFSActive, seqName);
    }

    @Command(level = 1)
    public void takeImagesSync(int n, int exposureMillis, boolean openShutter, boolean scienceActive,
            boolean guidingActive, boolean WFSActive, String seqName) {
        checkCommandValidity(MCMCommand.TAKEIMAGES);

        // NB raft integrate is sent sync to have invalid state exception
        // immediately
        // and not through the Future.

        if (openShutter && exposureMillis < 1000) {
            throw new RuntimeException("cannot handle opening the shutter for less that 1 second");
        }

        checkState(MinionGroup.CAMERA, Minion.FILTER, FilterReadinessState.READY);

        setState(MCMState.TAKINGIMAGES);

        // we should not fail, but initImages again if not in the proper state
        // TODO add safety margin? Possible race condition.
        // unless rafts agree to start integrating even if state quiescent was
        // just terminated.

        if (!isInState(MinionGroup.CAMERA, Minion.RAFTS, RaftState.QUIESCENT)) {
            doInitImages(0);
            waitForState(MinionGroup.CAMERA, Minion.RAFTS, RaftState.QUIESCENT, 200);
        }
        // checkState(Minion.RAFTS, RaftState.QUIESCENT);

        try {

            for (int i = 0; i < n; i++) {
                try {
                    log.info("MCM: send raft integrate");
                    send(MinionGroup.CAMERA, Minion.RAFTS, "integrate");
                    waitForState(MinionGroup.CAMERA, Minion.RAFTS, RaftState.INTEGRATING, 200);
                    log.info("MCM: raft is integrating");

                    if (openShutter) {
                        log.info("MCM: send shutter expose");
                        sendAsync(MinionGroup.CAMERA, Minion.SHUTTER, "expose", exposureMillis);
                        waitForState(MinionGroup.CAMERA, Minion.SHUTTER, ShutterState.OPEN, 1200);
                        log.info("MCM: shutter is open");
                        // schedule a stop integrating about .5 second before
                        // shutter is closed.
                        waitForState(MinionGroup.CAMERA, Minion.SHUTTER, ShutterState.CLOSED, exposureMillis + 3000L);
                        log.info("MCM: shutter is closed");
                    } else {
                        waitMillis(exposureMillis);
                        log.info("MCM: exposure time elapsed, with closed shutter");
                    }
                    log.info("MCM: send raft readout");
                    send(MinionGroup.CAMERA, Minion.RAFTS, "readout");
                    // if (i == n - 1)
                    // break;
                    waitForState(MinionGroup.CAMERA, Minion.RAFTS, RaftState.READING_OUT, 200);
                    log.info("MCM: raft is reading out");
                    waitForState(MinionGroup.CAMERA, Minion.RAFTS, RaftState.QUIESCENT, 3000);
                    log.info("MCM: raft is quiescent");
                } catch (OperationTimeoutException e) {
                    log.error("timeout while waiting", e);
                    throw e;
                } catch (Exception e) {
                    log.error("error", e);
                    throw new RuntimeException("error ", e);
                }

            }

        } finally {
            setState(MCMState.IDLE);
        }

    }

    @Command(level = 1)
    public void takeImagesAsync(int n, int exposureMillis, boolean openShutter, boolean scienceActive,
            boolean guidingActive, boolean WFSActive, String seqName) {

        checkCommandValidity(MCMCommand.TAKEIMAGES);

        // NB raft integrate is sent sync to have invalid state exception
        // immediately
        // and not through the Future.

        if (openShutter && exposureMillis < 1000) {
            throw new RuntimeException("cannot handle opening the shutter for less that 1 second");
        }

        checkState(MinionGroup.CAMERA, Minion.FILTER, FilterReadinessState.READY);

        setState(MCMState.TAKINGIMAGES);

        // we should not fail, but initImages again if not in the proper state
        // TODO add safety margin? Possible race condition.
        // unless rafts agree to start integrating even if state quiescent was
        // just terminated.

        if (!isInState(MinionGroup.CAMERA, Minion.RAFTS, RaftState.QUIESCENT)) {
            doInitImages(0);
            waitForState(MinionGroup.CAMERA, Minion.RAFTS, RaftState.QUIESCENT, 200);
        }
        // checkState(Minion.RAFTS, RaftState.QUIESCENT);

        execute(() -> {
            try {
                for (int i = 0; i < n; i++) {
                    try {
                        log.info("MCM: send raft integrate");
                        send(MinionGroup.CAMERA, Minion.RAFTS, "integrate");
                        waitForState(MinionGroup.CAMERA, Minion.RAFTS, RaftState.INTEGRATING, 200);
                        log.info("MCM: raft is integrating");

                        if (openShutter) {
                            log.info("MCM: send shutter expose");
                            sendAsync(MinionGroup.CAMERA, Minion.SHUTTER, "expose", exposureMillis);
                            waitForState(MinionGroup.CAMERA, Minion.SHUTTER, ShutterState.OPEN, 1200);
                            log.info("MCM: shutter is open");
                            // schedule a stop integrating about .5 second
                            // before
                            // shutter is closed.
                            waitForState(MinionGroup.CAMERA, Minion.SHUTTER, ShutterState.CLOSED, exposureMillis + 3000L);
                            log.info("MCM: shutter is closed");
                        } else {
                            waitMillis(exposureMillis);
                            log.info("MCM: exposure time elapsed, with closed shutter");
                        }
                        log.info("MCM: send raft readout");
                        send(MinionGroup.CAMERA, Minion.RAFTS, "readout");
                        // We could return during the readout since we do not
                        // have anything else to do.
                        // but then at the beginning of takeimage, we need to
                        // wait for readoutTime in case we find the rafts
                        // still in readout.
                        // if (i == n - 1)
                        // break;
                        waitForState(MinionGroup.CAMERA, Minion.RAFTS, RaftState.READING_OUT, 200);
                        log.info("MCM: raft is reading out");
                        waitForState(MinionGroup.CAMERA, Minion.RAFTS, RaftState.QUIESCENT, 3000);
                        log.info("MCM: raft is quiescent");
                    } catch (OperationTimeoutException e) {
                        log.error("timeout while waiting", e);
                        throw e;
                    } catch (Exception e) {
                        log.error("error", e);
                        throw new RuntimeException("error ", e);
                    }

                }
            } finally {
                setState(MCMState.IDLE);
            }
        });
    }

    @Command
    public void initGuiders() {
        checkCommandValidity(MCMCommand.INITGUIDER);
        // To be done
    }

    @Command
    public void clear(int n) {
        checkCommandValidity(MCMCommand.CLEAR);
        try {
            send(MinionGroup.CAMERA, Minion.RAFTS, "clear", n);
        } catch (OperationTimeoutException e) {
            log.error("timeout while waiting", e);
            throw e;
        } catch (Exception e) {
            log.error("error", e);
            throw new RuntimeException("error ", e);
        }
    }

    private final ScheduledThreadPoolExecutor scheduler = new ScheduledThreadPoolExecutor(4);

    volatile ScheduledFuture<?> imageTimeoutFuture = null;

    @Command
    public void startImage(String seqName, boolean openShutter, boolean scienceActive, boolean WFSActive,
            boolean guidingActive, int timeoutMillis) {
        checkCommandValidity(MCMCommand.STARTIMAGE);
        checkState(MinionGroup.CAMERA, Minion.FILTER, FilterReadinessState.READY);

        setState(MCMState.IMAGESTARTED);

        if (!isInState(MinionGroup.CAMERA, Minion.RAFTS, RaftState.QUIESCENT)) {
            doInitImages(0);
            waitForState(MinionGroup.CAMERA, Minion.RAFTS, RaftState.QUIESCENT, 200);
        }

        try {
            log.info("MCM: send raft integrate");
            send(MinionGroup.CAMERA, Minion.RAFTS, "integrate");
            waitForState(MinionGroup.CAMERA, Minion.RAFTS, RaftState.INTEGRATING, 200);
            log.info("MCM: raft is integrating");

            if (openShutter) {
                log.info("MCM: send shutter expose");
                sendAsync(MinionGroup.CAMERA, Minion.SHUTTER, "expose", timeoutMillis);
                waitForState(MinionGroup.CAMERA, Minion.SHUTTER, ShutterState.OPEN, 1200);
                log.info("MCM: shutter is open");
            }

            // handle the timeout: close shutter, clear ccd.
            imageTimeoutFuture = scheduler.schedule(() -> startImageTimeout(), timeoutMillis, TimeUnit.MILLISECONDS);

        } catch (OperationTimeoutException e) {
            log.error("timeout while waiting", e);
            throw e;
        } catch (Exception e) {
            log.error("error", e);
            throw new RuntimeException("error ", e);
        }

    }

    public void startImageTimeout() {
        synchronized (imageTimeoutFuture) {
            if (imageTimeoutFuture == null)
                return;// we lost
            imageTimeoutFuture = null;
            log.info("MCM: timeout on startImage");
            sendAsync(MinionGroup.CAMERA, Minion.SHUTTER, "close"); // for security. Could trigger
                                                // error, ignored, if not open.
            sendAsync(MinionGroup.CAMERA, Minion.RAFTS, "clear");
            waitForState(MinionGroup.CAMERA, Minion.SHUTTER, ShutterState.CLOSED, 2000L);
            log.info("MCM: shutter is closed");
            setState(MCMState.IDLE);
        }
    }

    @Command
    public void endImage() {
        synchronized (imageTimeoutFuture) {
            if (imageTimeoutFuture == null)
                return;// we lost TODO this should lead to rejection of endImage
                       // with error
            imageTimeoutFuture.cancel(false);
            imageTimeoutFuture = null;
            checkCommandValidity(MCMCommand.ENDIMAGE);
            log.info("MCM: endImage, closing shutter");
            sendAsync(MinionGroup.CAMERA, Minion.SHUTTER, "close");
            waitForState(MinionGroup.CAMERA, Minion.SHUTTER, ShutterState.CLOSED, 2000L);
            log.info("MCM: shutter is closed, send raft readout");
            sendAsync(MinionGroup.CAMERA, Minion.RAFTS, "readout");
            waitForState(MinionGroup.CAMERA, Minion.RAFTS, RaftState.READING_OUT, 200);
            log.info("MCM: raft is reading out");
            setState(MCMState.IDLE);
        }
    }

    @Command
    public void discardRows(int n) {
        checkCommandValidity(MCMCommand.DISCARDROWS);
        // we should be in state taking images, we stay in that state, and we
        // block
        try {
            send(MinionGroup.CAMERA, Minion.RAFTS, "discardRows", n);
        } catch (OperationTimeoutException e) {
            log.error("timeout while waiting", e);
            throw e;
        } catch (InvalidStateException e) {
            log.error("subsytem not valid state for command", e);
            throw new RuntimeException("subsytem not valid state for command (" + e.getMessage() + ")");
        } catch (Exception e) {
            log.error("error ", e);
            throw new RuntimeException("error ", e);
        }
    }

    // @Override
    // public void tick() {
    // Map<String, Object> last = mu.getStatusAggregator().getAllLast();
    // for (Map.Entry<String, Object> ke : last.entrySet()) {
    // log.debug(" status " + ke.getKey() + " last " + ke.getValue() + " avg "
    // + mu.getStatusAggregator().getAverage(ke.getKey()) + " hist "
    // + mu.getStatusAggregator().getHistory(ke.getKey()).size());
    //
    // CameraStatus cs = new CameraStatus();
    // cs.setRaftTemperature(mu.getStatusAggregator().getAverage("raftsim/temperature"));
    // log.info("publish raft avg temp " + cs.getRaftTemperature());
    // KeyValueData d = new KeyValueData("cameraStatus", cs);
    // getSubsystem().publishSubsystemDataOnStatusBus(d);
    //
    // }
    // }

}
