package org.lsst.ccs.subsystem.mcm;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.function.Consumer;
import java.util.function.Predicate;
import javafx.util.converter.LocalDateStringConverter;

import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.messages.CommandRequest;

import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.messaging.ConcurrentMessagingUtils;
import org.lsst.ccs.utilities.logging.Logger;

public class AlertDispatcher implements HasLifecycle {


    @LookupField(strategy = Strategy.ANCESTORS)
    GenericMCM mcm;

    @LookupName
    String name;
    
    @LookupField(strategy = Strategy.CHILDREN)
    protected Map<String, AlertHandler> handlers = new HashMap<>();
    
    boolean isEnabled = true;
    
    private MCMUtilities<?, ?, ?> mu;
    
    private final Enum group;
    private final String groupName;

    public AlertDispatcher(Enum group) {
        this(null, group);
    }

    AlertDispatcher(MCMUtilities<?, ?, ?> mu2, Enum group) {
        this.mu = mu2;
        this.group = group;
        groupName = group.name();
    }

    public MCMUtilities getMcmUtilities() {
        return mu;
    }
    
    @Override
    public void postInit() {
        if ( mcm != null ) {
            if ( mu != null ) {
                throw new RuntimeException("**** Something went very wrong. This AlertDispatcher has already been initialized");
            }
            mu = mcm.getMcmUtilities();
        }
    }
    
    @Command
    public void enable() {
        this.isEnabled = true;
    }

    @Command
    public void disable(@Argument(defaultValue = "true") boolean disable) {
        this.isEnabled = !disable;
    }

    @Command
    public String status() {
        return status("");
    }
    
    String status(String indent) {
        StringBuilder sb = new StringBuilder();
        sb.append(indent).append("Name     : ").append(isEnabled).append("\n");
        sb.append(indent).append("Enabled  : ").append(isEnabled).append("\n");
        sb.append(indent).append("Group    : ").append(group).append("\n");
        sb.append(indent).append("Handlers :\n");
        for (AlertHandler h : handlers.values()) {
            sb.append(h.status(indent+"\t"));
        }
        return sb.toString();
    }

    public Enum getGroup() {
        return group;
    }
    
    public String getGroupName() {
        return groupName;
    }

    public void onAlert(AlertNotification notif) {
        if ( isEnabled ) {
            for (AlertHandler h : handlers.values()) {
                h.onAlert(notif);
            }
        }
    }

    private static Logger log = Logger.getLogger("org.lsst.ccs.subsystem.mcm");

    public static Logger getLogger() {
        return log;
    }

    private void addHandler(AlertHandler h) {
        handlers.put(h.getName(), h);
    }

    // convenience methods

    @Command
    public AlertHandler addAlertHandler(String name) {
        AlertHandler h = new AlertHandler(name);
        addHandler(h);
        return h;
    }

    @Command
    public AlertHandler addEmailAction(String handlerName, String email) {
        AlertHandler h = handlers.get(handlerName);
        h.addEmailAction(email);
        return h;
    }

    @Command
    public AlertHandler addCommandAction(String handlerName, String destination, String command, Object... parameters) {
        AlertHandler h = handlers.get(handlerName);
        h.addCommandAction(destination, command, parameters);
        return h;
    }

    @Command
    public AlertHandler setOriginSelector(String handlerName, String origin) {
        AlertHandler h = handlers.get(handlerName);
        h.setOriginSelector(origin);
        return h;
    }

    @Command
    public AlertHandler setOriginAndAlertIdSelector(String handlerName, String origin, String alertId) {
        AlertHandler h = handlers.get(handlerName);
        h.setOriginAndAlertIdSelector(origin, alertId);
        return h;
    }

    
    @Command
    public String[] getAlertHandlers() {
        return handlers.keySet().toArray(new String[0]);
    }

    
    public static class AlertHandler {
        public AlertHandler(String name) {
            this.handlerName = name;
        }

        public AlertHandler addAction(AlertAction action) {
            actions.add(action);
            return this;
        }

        // Convenience methods
        public AlertHandler addEmailAction(String email) {
            return addAction(new EmailAlertAction(email));
        }

        public AlertHandler addCommandAction(String destination, String command, Object... parameters) {
            return addAction(new CommandAlertAction(destination, command, parameters));
        }

        public String getName() {
            return handlerName;
        }

        public AlertHandler setSelector(Predicate<AlertNotification> selector) {
            this.selector = selector;
            return this;
        }

        public AlertHandler orSelector(Predicate<AlertNotification> or) {
            if (this.selector == null) {
                this.selector = or;
            } else {
                this.selector = this.selector.or(or);
            }
            return this;
        }

        public AlertHandler andSelector(Predicate<AlertNotification> and) {
            if (this.selector == null) {
                this.selector = and;
            } else {
                this.selector = this.selector.and(and);
            }
            return this;
        }

        public Predicate<AlertNotification> originSelector(String origin) {
            return a -> a.getOrigin().equals(origin);
        }

        public Predicate<AlertNotification> alertIdSelector(String id) {
            return a -> a.getAlert().getAlertId().equals(id);
        }

        public Predicate<AlertNotification> alertLevelSelector(AlertState severity) {
            return a -> a.getSeverity().ordinal() >= severity.ordinal();
        }

        // convenience
        public AlertHandler setOriginSelector(String origin) {
            return setSelector(originSelector(origin));
        }

        public AlertHandler setOriginAndAlertIdSelector(String origin, String alertId) {
            return setSelector(originSelector(origin).and(alertIdSelector(alertId)));
        }

        String handlerName;

        boolean enabled = true;

        @LookupField(strategy = LookupField.Strategy.CHILDREN)
        List<AlertAction> actions = new ArrayList<AlertAction>();

        Predicate<AlertNotification> selector;

        @Command
        public void enable() {
            this.enabled = true;
        }

        @Command
        public void disable(@Argument(defaultValue = "true") boolean disable) {
            this.enabled = !disable;
        }

        @Command
        public String status() {
            return status("");
        }

        String status(String indent) {
            StringBuilder sb = new StringBuilder();
            sb.append(indent).append("Name     : ").append(handlerName).append("\n");
            sb.append(indent).append("Enabled  : ").append(enabled).append("\n");
            sb.append(indent).append("Selector : ").append(selector).append("\n");
            sb.append(indent).append("Actions  :\n");
            if ( actions.size() > 0 ) {
            for (AlertAction h : actions ) {
                sb.append(h.status(indent + "\t"));
            }
            } else {
                sb.append(indent).append("\tNone defined");
            }
            sb.append("\n");
            return sb.toString();
        }
        
        
        
        public void onAlert(AlertNotification notif) {
            if (!enabled)
                return;            
//            if (notif.getSeverity().equals(AlertState.NOMINAL)) {
//                return;
//            }
            if (selector != null && !selector.test(notif)) {
                return;
            }
            for (AlertAction a : actions) {
                try {
                    a.accept(notif);
                } catch (Throwable e) {
                    log.error("Error handling action " + e, e);
                    e.printStackTrace();
                }
            }
        }

    }

    // AlertHandler => named
    // AlertSelector
    // boolean function with origin and alert object BiPredicate<String,Alert>
    // combinator
    public static abstract class AlertAction implements Consumer<AlertNotification>, HasLifecycle {
        // implementations: email, SMS, command on subsystem
        final Logger ACTION_LOG = Logger.getLogger("org.lsst.ccs.mcm.action");

        @LookupName
        String name;
        
        protected Logger getLogger() {
            return ACTION_LOG;
        }

        @Command
        public String status() {
            return status("");
        }

        String status(String indent) {
            StringBuilder sb = new StringBuilder();
            sb.append(indent).append("Name        : ").append(name).append("\n");
            sb.append(indent).append("Enabled     : ").append(enabled).append("\n");
            sb.append(indent).append("Description : ").append(getDescription()).append("\n");
            return sb.toString();
        }

        boolean enabled = true;

        @Command
        public void enable() {
            this.enabled = true;
        }
        
        @Command
        public void disable(@Argument(defaultValue = "true") boolean disable) {
            this.enabled = !disable;
        }
        
        public abstract String getDescription();
    }

    public static class EmailAlertActionFromFile extends EmailAlertAction {
                
        File file = null;
        long lastModified;
        String filePath;
        

        EmailAlertActionFromFile() {
            super(new ArrayList<>());
        }
        
        @Override
        public void init() {
            super.init();
            if ( filePath != null ) {
                try {
                    file = new File(filePath);
                    updateEmailList(file);
                } catch (Exception e) {

                }
            }
        }            
        
        private void updateEmailList(File f) {
            log.info("Updating email list form "+f.getAbsolutePath()+" "+f.lastModified());
            try {
                FileInputStream fis = new FileInputStream(f);
                InputStreamReader in = new InputStreamReader(fis);
                BufferedReader reader = new BufferedReader(in);
                List<String> addresses = new ArrayList<>();
                for (;;) {
                    String line = reader.readLine();
                    if (line == null) {
                        break;
                    } else if (line.startsWith("#")) {
                        continue;
                    }
                    line = line.trim();
                    int indx = line.indexOf("#");
                    if ( indx > -1 ) {
                        line = line.substring(0, indx);
                        line = line.trim();
                    }
                    addresses.add(line);
                }        
                configureEmailAddress(addresses);
                lastModified = f.lastModified();
            } catch (IOException fnf) {
                fnf.printStackTrace();
            }
        }
        
        @Override
        public void accept(AlertNotification notif) {
            if ( file != null && file.exists() ) {
                if (lastModified != file.lastModified()) {
                    updateEmailList(file);
                }
                super.accept(notif);
            }
        }        

        @Override
        public String getDescription() {
            return super.getDescription()+" from file "+file.getAbsolutePath();
        }
    }
    
    public static class EmailAlertAction extends AlertAction {

        protected String fromEmail = "mcm@no-reply.org";
        protected String replyToEmail = null;
        protected String smtpHost = "smtpunix.slac.stanford.edu";
        private DateTimeFormatter dtf = DateTimeFormatter.ofPattern("HH:mm:ss dd MMM yyyy");
        
        public EmailAlertAction(List<String> emails) {
            configureEmailAddress(emails);

        }

        @Override
        public String getDescription() {
            String description = "Email notification to [";
            for ( InternetAddress email: address) {
                description += email.getAddress()+", ";                
            }
            description = description.substring(0,description.length()-2);
            return description+="]";
        }
        
        @Override
        public void init() {
            Properties props = System.getProperties();
            props.put("mail.smtp.host", smtpHost);
            session = Session.getInstance(props, null);
            
            try {
                fromAddress = new InternetAddress(fromEmail);
            } catch (AddressException e) {
                log.error("Error while parsing address " + fromEmail, e);
                throw new RuntimeException("Error while parsing address " + fromEmail, e);
            }
        }            
        

        protected final void configureEmailAddress(List<String> emails) {
            address = new InternetAddress[emails.size()];
            int count = 0;
            for (String email : emails) {
                try {
                    address[count++] = new InternetAddress(email);
                } catch (AddressException e) {
                    log.error("Error while parsing address " + email, e);
                    throw new RuntimeException("Error while parsing address " + email, e);
                }
            }            
        }
        
        public EmailAlertAction(String email) {
            this( Arrays.asList(email) );
        }

        InternetAddress[] address;
        Session session;

        InternetAddress fromAddress;

        @Override
        public void accept(AlertNotification notif) {
            try {
                Message msg = new MimeMessage(session);
                msg.setFrom(fromAddress);
                msg.setRecipients(Message.RecipientType.TO, address);

                if ( replyToEmail != null ) {               
                    InternetAddress replyTo = new InternetAddress(replyToEmail);
                    msg.setReplyTo(new InternetAddress[]{replyTo});
                }
                
                //DO NOT CHANGE THE SUBJECT LINE AS IT IS PARSED BY USERS
                String subject = "CCS Alert "+notif.getSeverity()+" "+ notif.getOrigin() + " " + notif.getAlert().getAlertId();
                msg.setSubject(subject);
                
                LocalDateTime date = LocalDateTime.ofInstant(Instant.ofEpochMilli(notif.getTimestamp()), ZoneId.systemDefault());
                String text = "CCS Alert \n" + //
                        "  origin:   " + notif.getOrigin() + "\n" + //
                        "  cause:    " + notif.getCause() + "\n" + //
                        "  id:       " + notif.getAlert().getAlertId() + "\n" + //
                        "  severity: " + notif.getSeverity() + "\n" + //
                        "  descr:    " + notif.getAlert().getDescription() + "\n" + //
                        "  time:     " + date.format(dtf) + "\n";
                msg.setText(text);
                msg.setSentDate(new Date());
                log.info("Sending email: \n"+subject+"\n"+text);
                Transport.send(msg);
            } catch (MessagingException e) {
                log.error("Failed sending email " + e, e);
                e.printStackTrace();
            }

        }

    }

    public static class CommandAlertAction extends AlertAction {

        String destination;
        String command;
        Object[] parameters;

        @LookupField(strategy = Strategy.TOP)
        private Subsystem s;

        public CommandAlertAction() {
            super();
        }
        public CommandAlertAction(String destination, String command, Object... parameters) {
            super();
            this.destination = destination;
            this.command = command;
            this.parameters = parameters;
        }

        @Override
        public String getDescription() {
            return "Command "+command+" sent to "+destination+" with parameters "+parameters;
        }

        @Override
        public void accept(AlertNotification t) {
            log.info("command on alert    " + t);
            log.info("command on alert -> " + destination + " " + command);
                new ConcurrentMessagingUtils(s.getMessagingAccess()).sendAsynchronousCommand(new CommandRequest(destination, command, parameters));
        }

    }

}
