package org.lsst.ccs.subsystem.imagehandling;

import java.io.File;
import java.io.IOException;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.logging.Level;
import java.util.logging.Logger;
import nom.tam.fits.FitsException;
import org.lsst.ccs.daq.guider.FitsWriterFactory;
import org.lsst.ccs.daq.guider.SeriesMetaData;
import org.lsst.ccs.daq.guider.StateMetaData;
import org.lsst.ccs.daq.ims.DAQException;
import org.lsst.ccs.daq.ims.Guider;
import org.lsst.ccs.daq.ims.Store;
import org.lsst.ccs.daq.ims.channel.FitsIntWriter;
import org.lsst.ccs.daq.utilities.FitsServiceInterface;
import org.lsst.ccs.daq.utilities.FitsService;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.subsystem.imagehandling.data.FileList;
import org.lsst.ccs.utilities.ccd.CCD;
import org.lsst.ccs.utilities.ccd.FocalPlane;
import org.lsst.ccs.utilities.ccd.Reb;
import org.lsst.ccs.utilities.image.FitsHeaderMetadataProvider;
import org.lsst.ccs.utilities.image.HeaderSpecification;
import org.lsst.ccs.utilities.location.SensorLocation;
import org.lsst.ccs.utilities.location.SensorLocationSet;
import org.lsst.ccs.utilities.readout.GeometryFitsHeaderMetadataProvider;

/**
 * Handle the work of setting up and handling the guide sensors
 * @author tonyj
 */
class GuiderHandling {

    private static final Logger LOG = Logger.getLogger(GuiderHandling.class.getName());

    private final ImageHandlingConfig config;
    private final ExecutorService executor;
    private final PostImageFileHandling postImageFileHandling;
    private final List<Future<Object>> activeTasks = new ArrayList();
    private final FitsService fitsService;
    private Store guiderStore;
    private final FocalPlane geometry;

    /**
     * Create new instance of Guider Handling. Created once during subsystem startup
     * @param config The configuration
     * @param executor An executor to use for background tasks, in particular waiting for guider data to appear
     * @param geometry The focal-plan geometry in use. (Defined once during subsystem build and then immutable)
     * @param fitsService The FitsService to use for getting FITS header data
     * @param postImageFileHandling Post image handling, including patching up headers based on header service data, and running post image commands
     * @throws DAQException If an error talking to configuring the DAQ
     */
    GuiderHandling(ImageHandlingConfig config, ExecutorService executor, FocalPlane geometry, FitsService fitsService, PostImageFileHandling postImageFileHandling) throws DAQException {
        this.config = config;
        this.executor = executor;
        this.postImageFileHandling = postImageFileHandling;
        this.fitsService = fitsService;
        this.geometry = geometry;
    }

    /**
     * Start the GuiderHandling. Called during subsystem startup and also as a result of a subsequent change in the guider configuration.
     * @throws DAQException 
     */
    void start() throws DAQException {        
        this.guiderStore = new Store(config.getGuiderPartition());
        Guider guider = guiderStore.getGuider();
        SensorLocationSet guiderLocations = config.getGuiderLocations();
                        
        // Keep track of temporary file locations, if any
        Map<File, File> tmpMap = new HashMap<>();
        // We need to add a subscriber for each guiderLocation
        String partition = guider.getPartition();
        for (SensorLocation sl : guiderLocations) {
            final Reb reb = geometry.getRebAtLocation(sl.getRebLocation());
            fitsService.addReb(reb, "guider");
            FitsServiceInterface fitsServiceForReb = fitsService.getFitsServiceForReb(reb);
            final CCD ccd = reb.getCCDs().get(sl.getSensor());
            List<FitsHeaderMetadataProvider> providers = new ArrayList<>();
            providers.add(fitsServiceForReb.getFitsHeaderMetadataProvider(ccd.getUniqueId()));
            providers.add(new GeometryFitsHeaderMetadataProvider(ccd));           
            FitsIntWriter.FileNamer fileNamer = config.getFileNamer(tmpMap, true);
            Map<String, HeaderSpecification> headerSpecifications = fitsServiceForReb.getHeaderSpecificationMap();
            GuiderFitsWriterHandlerFactory handler = new GuiderFitsWriterHandlerFactory(partition, fileNamer, headerSpecifications, providers, postImageFileHandling, config.getIncludeRawStamps());
            // Currently each subscriber needs its own thread to wait for events
            activeTasks.add(executor.submit(() -> {
                try (Guider.Subscriber subscriber = guider.subscribe(Collections.singleton(sl), ByteOrder.BIG_ENDIAN, handler)) {
                    for (;!Thread.currentThread().isInterrupted();) {
                        try {
                            subscriber.waitForGuider();
                        } catch (DAQException x) {
                            LOG.log(Level.SEVERE, "Error during guider data processing for "+sl, x);
                        }
                    }
                }
                return null;
            }));
        }
        // ToDo: 
        //   * Deal with renaming temporary files (if needed)
        //   
    }
    
    /**
     * Undo the actions of start, to be (possibly) followed by another start
     * @throws DAQException 
     */
    void stop() throws DAQException {
        for (Future<Object> task : activeTasks) {
            task.cancel(true);
        }
        // Wait for tasks to finish?
        activeTasks.clear();
        if (guiderStore != null) {
            guiderStore.close();
            guiderStore = null;
        }
        fitsService.clearAllRebs();
    }

    private static class GuiderFitsWriterHandlerFactory extends FitsWriterFactory {

        private final PostImageFileHandling postImageFileHandling;
        private final List<FitsHeaderMetadataProvider> metaDataProviders;
        
        public GuiderFitsWriterHandlerFactory(String partition, FitsIntWriter.FileNamer fileNamer, Map<String, HeaderSpecification> headerSpecifications, List<FitsHeaderMetadataProvider> metaDataProviders, PostImageFileHandling postImageFileHandling, boolean includeRawStamps) {
            super(partition, fileNamer, headerSpecifications, includeRawStamps);
            this.postImageFileHandling = postImageFileHandling;
            this.metaDataProviders = metaDataProviders;
        }

        @Override
        protected FitsWriter createFitsFileWriter(StateMetaData state, SeriesMetaData series, String partition, FitsIntWriter.FileNamer fileNamer, Map<String, HeaderSpecification> headerSpecifications) throws IOException, FitsException {    
            return new GuiderFitsWriter(state, series, partition, fileNamer, headerSpecifications, metaDataProviders, postImageFileHandling); 
        }

    }
        
    private static class GuiderFitsWriter extends FitsWriterFactory.FitsWriter {
        private final PostImageFileHandling postImageFileHandling;
        private final SensorLocation sensorLocation;

        public GuiderFitsWriter(StateMetaData state, SeriesMetaData series, String partition, FitsIntWriter.FileNamer fileNamer, Map<String, HeaderSpecification> headerSpecifications, List<FitsHeaderMetadataProvider> metaDataProviders, PostImageFileHandling postImageFileHandling) throws IOException, FitsException {
            super(state, series, partition, fileNamer, headerSpecifications, metaDataProviders);
            this.postImageFileHandling = postImageFileHandling;
            sensorLocation = series.getLocation().getLocation();
        }

        @Override
        public void close() throws IOException, FitsException {
            super.close(); 
            try {
                ImageName obsId = new ImageName(getImageName());
            
                /*
                 * We creae a future file list here, so it can be passed to handleHeaderServiceData. We only
                 * have a single file, and it already exists, so this is not really needed, but it keeps compatibility
                 * with the existing code for non-guider data.
                 */
                FileList fl = new FileList();
                fl.add(getFileName()); 
                CompletableFuture<FileList> futureFileList = CompletableFuture.completedFuture(fl);
                futureFileList = postImageFileHandling.handleHeaderServiceData(futureFileList, getImageName());
                /*
                 * Similarly we pass the fileList to handleFitsFileCommands, and pretend we don't already know the sensor location
                 * (it gets recreated in handleFitsFileCommands). This could be cleaned up, but not easily.
                 */
                postImageFileHandling.handleFitsFileCommands(futureFileList, sensorLocation.getRebLocation(), obsId, "GUIDER");
            } catch (IllegalArgumentException x) {
                LOG.log(Level.WARNING, "Invalid obsid for guider image, ignored: "+getImageName(),x); 
            }
         }
        
    }

}
