package org.lsst.ccs.subsystem.imagehandling;

import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.Collections;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentCategory;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.daq.ims.ImageMetaData;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.description.ComponentNode;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.subsystem.imagehandling.data.FileList;
import org.lsst.ccs.utilities.location.Location;
import org.lsst.ccs.utilities.location.LocationSet;

public class ImageHandlingSubsystem extends Subsystem implements HasLifecycle {

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private ImageHandlingClient imageHandlingClient;

    public ImageHandlingSubsystem() {
        super("image-handling", AgentInfo.AgentType.SERVICE);
        getAgentInfo().getAgentProperties().setProperty(AgentCategory.AGENT_CATEGORY_PROPERTY, AgentCategory.IMAGE_HANDLER.name());
    }

    @Override
    public void build() {
        ComponentLookup lookupService = getComponentLookup();
        ComponentNode thisComponent = lookupService.getComponentNodeForObject(this);

        //Add listener to focal-plane configuration changes
        ComponentNode focalPlaneConfigurationHandler = new ComponentNode("configHandler", new BusMessageHandler());
        lookupService.addComponentNodeToLookup(thisComponent, focalPlaneConfigurationHandler);
    }

    @Override
    public void start() {
        // Turn off annoying messages from FITS (LSSTCCSRAFTS-656)
        Logger.getLogger("nom.tam.fits.HeaderCard").setLevel(Level.WARNING);
    }

    @Command(type = Command.CommandType.ACTION, description = "Fetch a named image from the 2-day store")
    public FileList fetchImage(String imageName) {
        return helper()
                .precondition(imageHandlingClient != null, "No image handling client registered")
                .enterFaultOnException(true)
                .action(() -> {
                    return imageHandlingClient.fetchImage(imageName);
                });
    }

    @Command(type = Command.CommandType.ACTION, description = "Simulate an image arriving from the 2-day store")
    public void simulate(String imageName, String dataFolder) {
        helper()
            .precondition(imageHandlingClient != null, "No image handling client registered")
            .enterFaultOnException(true)
            .action(() -> {
                Location location = Location.of("R00/RebW");
                ImageMetaData meta = new ImageMetaData(imageName, "raw", dataFolder, 7, new LocationSet(Collections.singleton(location)));
                Path metaPath = Path.of(dataFolder).resolve(imageName+"_R00_RebW.meta");
                if (!Files.exists(metaPath)) {
                    throw new IllegalArgumentException("Path "+metaPath+" does not exist");
                }
                Path rawData = Path.of(dataFolder).resolve(imageName+"_R00_RebW.raw");
                if (!Files.exists(rawData)) {
                    throw new IllegalArgumentException("Path "+rawData+" does not exist");
                }
                String line = Files.newBufferedReader(metaPath).readLine();
                int[] registerList = Arrays.stream(line.substring(1, line.length() - 1).split(",")).map(String::trim).mapToInt(Integer::parseInt).toArray();
                imageHandlingClient.simulateTrigger(location, meta, registerList, rawData);
            });
    }
}
