package org.lsst.ccs.subsystem.imagehandling;

import org.lsst.ccs.subsystem.imagehandling.data.FileList;
import java.io.IOException;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.logging.Logger;
import org.lsst.ccs.daq.ims.DAQException;
import org.lsst.ccs.daq.ims.Image;
import org.lsst.ccs.utilities.location.Location;
import org.lsst.ccs.daq.ims.Source;

/**
 * Coordinates the processing of a single image. Note since it is possible that
 * there are multiple images in flight at once, more than one instance of this
 * class may be running simultaneously.
 *
 * @author tonyj
 */
class ImageHandler implements Callable<FileList> {

    private static final Logger LOG = Logger.getLogger(ImageHandler.class.getName());

    private final Image image;
    private final ImageHandlingConfig config;
    private final ExecutorService daqExecutor;
    private final List<RebNode> rebs;
    private final boolean isStreaming;
    private final CountDownLatch darkTime = new CountDownLatch(1);
    private final Map<String,Serializable> imageMetaData;
    private final PostImageFileHandling postImageFileHandling;

    ImageHandler(Map<String,Serializable> imageMetaData, Image image, ExecutorService daqExecutor, ImageHandlingConfig config, List<RebNode> rebs, boolean isStreaming, PostImageFileHandling postImageFileHandling) {
        this.imageMetaData = imageMetaData;
        this.image = image;
        this.config = config;
        this.daqExecutor = daqExecutor;
        this.rebs = rebs;
        this.isStreaming = isStreaming;
        this.postImageFileHandling = postImageFileHandling;
    }

    @Override
    public FileList call() throws IOException, DAQException, InterruptedException, ExecutionException {
        // Image has one source for each REB associated with the image
        List<Source> sources = image.listSources();
        Set<Location> locationsToProcess = config.getLocationsToProcess();
        List<CompletableFuture<FileList>> lffl = new ArrayList<>();
        sources.stream()
                .filter((source) -> (locationsToProcess.contains(source.getLocation())))
                .forEach((source) -> {
                    Location location = source.getLocation();
                    for (RebNode r : rebs) {
                        if (r.getLocation().equals(location)) {
                            final String imageName = image.getMetaData().getName();
                            // Source handler is a callable, which creates a FileList by reading data from the DAQ
                            final SourceHandler sourceHandler = new SourceHandler(imageMetaData, darkTime, imageName, source, config, r, isStreaming);
                            // This executes the callable on the DAQ executor thread, and handles IOExceptions
                            CompletableFuture<FileList> futureFilelist = CompletableUtils.asyncCallable(daqExecutor, sourceHandler);
                            // Everything downstream of this should NOT be executed on the DAQ executor thread.
                            CompletableFuture<FileList> asyncFutureFilelist = futureFilelist.thenApplyAsync(fl->fl);
                            asyncFutureFilelist = postImageFileHandling.handleHeaderServiceData(asyncFutureFilelist, imageName);
                            lffl.add(asyncFutureFilelist);
                            postImageFileHandling.handleFitsFileCommands(asyncFutureFilelist, location, imageName, "SCIENCE");
                        }
                    }
                });
        FileList result = new FileList();
        for (Future<FileList> ffl : lffl) {
            FileList fl = ffl.get();
            result.addAll(fl);
        }
        return result;
    }
    
    void darkTimeArrived() {
        darkTime.countDown();
    }

}
