package org.lsst.ccs.subsystem.imagehandling;

import java.io.Serializable;
import java.nio.file.Path;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.Semaphore;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.daq.ims.DAQException;
import org.lsst.ccs.daq.ims.Folder;
import org.lsst.ccs.daq.ims.Image;
import org.lsst.ccs.daq.ims.ImageListener;
import org.lsst.ccs.daq.ims.ImageMetaData;
import org.lsst.ccs.daq.ims.Store;
import org.lsst.ccs.daq.utilities.FitsServiceInterface;
import org.lsst.ccs.daq.utilities.FitsService;
import org.lsst.ccs.services.AgentStatusAggregatorService;
import org.lsst.ccs.subsystem.imagehandling.data.FileList;
import org.lsst.ccs.subsystem.imagehandling.data.ImageReceivedEvent;
import org.lsst.ccs.utilities.ccd.FocalPlane;
import org.lsst.ccs.utilities.ccd.Reb;
import org.lsst.ccs.utilities.location.Location;
import org.lsst.ccs.utilities.location.LocationSet;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 *
 * @author tonyj
 */
public class ScienceHandling {

    private static final Logger LOG = Logger.getLogger(ScienceHandling.class.getName());

    private final ImageHandlingConfig imageHandlingConfig;
    private final FitsService fitsService;
    private final PostImageFileHandling postImageFileHandling;
    private final FocalPlane geometry;
    private final Agent agent;
    private final AgentStatusAggregatorService statusAggregator;
    private final Map<String, PerImageInfo> perImageInfo = new ConcurrentHashMap<>();
    private static class PerImageInfo {
        private volatile Map<String, Serializable> metaDataSets;
        // Shows that image handler was created before darkTimeArrived was called
        private ImageHandler imageHandlerPendingDarktime;
        // Shows that dark time has arrived before imageHandlerCreated was called
        private boolean darkTimeArrived;     
        private final String imageName;
        
        PerImageInfo(String imageName) {
            this.imageName = imageName;
            metaDataSets = new ConcurrentHashMap<>();
            imageHandlerPendingDarktime = null;
            darkTimeArrived = false;
        }

        private synchronized void darkTimeArrived() {
            if (imageHandlerPendingDarktime != null) {
                LOG.log(Level.FINE, "darkTimeArrived for {0}, calling darkTimeArrived for imageHandler {1}", new Object[]{imageName, imageHandlerPendingDarktime.getImageName()});
                imageHandlerPendingDarktime.darkTimeArrived();
                imageHandlerPendingDarktime = null;
            } else {
                LOG.log(Level.FINE, "darkTimeArrived for {0}, setting darkTimeArrived=true", imageName);
                darkTimeArrived = true;
            }
        }

        private synchronized void imageHandlerCreated(ImageHandler imageHandler) {
            if (darkTimeArrived) {
                LOG.log(Level.FINE, "Creating image handler for {0}, darkTimeArrived==true", imageHandler.getImageName());
                imageHandler.darkTimeArrived();
                darkTimeArrived = false;
            } else {
                LOG.log(Level.FINE, "Creating image handler for {0}, darkTimeArrived==false, imageHandlerPendingDarktime=", new Object[]{imageName, imageHandler.getImageName()});
                imageHandlerPendingDarktime = imageHandler;
            }
        }
    }

    private ThreadPoolExecutor daqExecutor;
    private Store scienceStore;

    private volatile String runNumber;
    private String previousRunNumber;
    private List<RebNode> rebNodes;

    ScienceHandling(ImageHandlingConfig imageHandlingConfig, FocalPlane geometry, FitsService fitsService, PostImageFileHandling postImageFileHandling, Agent agent, AgentStatusAggregatorService statusAggregator) {
        this.imageHandlingConfig = imageHandlingConfig;
        this.geometry = geometry;
        this.fitsService = fitsService;
        this.postImageFileHandling = postImageFileHandling;
        this.agent = agent;
        this.statusAggregator = statusAggregator;
    }

    void start() throws DAQException {
        Semaphore semaphore = new Semaphore(imageHandlingConfig.getDaqThreads());
        int nLocations = imageHandlingConfig.getLocationsToProcess().size();
        // We need to be able to handle multiple in-flight images.
        final int nThreads = Math.max(Integer.getInteger("org.lsst.ccs.subsystem.imagehandling.MinDAQThreads", 40), 2*(nLocations + 2)); 
        // pre-create all the stores we will need. We do this because creating new stores
        // as needed while the DAQ is busy often fails. We also need a semaphore to restrict
        // how many simulataneous threads are accessing the DAQ, since exceeding this can also
        // cause the DAQ to fail
        ConcurrentLinkedDeque<Store> stores = new ConcurrentLinkedDeque<>();
        for (int i = 0; i < nThreads; i++) {
            stores.add(new Store(imageHandlingConfig.getDaqPartition()));
        }
        ThreadFactory readThreadFactory = (Runnable r) -> new ReadThread(r, stores, semaphore);

        daqExecutor = new ThreadPoolExecutor(nThreads, nThreads, 60L,
                TimeUnit.SECONDS, new SynchronousQueue<>(), readThreadFactory);

        // RebNodes are no longer created in the groovy, so we make them here
        rebNodes = new ArrayList<>();
        for (Location location : imageHandlingConfig.getLocationsToProcess()) {
            final Reb reb = geometry.getRebAtLocation(location);
            fitsService.addReb(reb, "science");
            FitsServiceInterface fitsServiceForReb = fitsService.getFitsServiceForReb(reb);
            RebNode rebNode = new RebNode(reb, fitsServiceForReb);
            rebNodes.add(rebNode);
        }

        scienceStore = new Store(imageHandlingConfig.getDaqPartition());
        scienceStore.addImageListener(new ImageListener() {

            @Override
            public void imageCreated(Image image) {
                if (imageHandlingConfig.isUseStreaming()) {
                    if (!checkFolder(image)) {
                        return;
                    }
                    handleImage(image, true);
                }
            }

            @Override
            public void imageComplete(Image image) {
                KeyValueData kvd = new KeyValueData(ImageReceivedEvent.EVENT_KEY, new ImageReceivedEvent(image.getMetaData()));
                agent.publishSubsystemDataOnStatusBus(kvd);
                if (!imageHandlingConfig.isUseStreaming()) {
                    if (!checkFolder(image)) {
                        return;
                    }
                    handleImage(image, false);
                }
            }

            private void handleImage(Image image, boolean isStreaming) {
                LocationSet locationsWritten = new LocationSet(image.getMetaData().getLocations());
                locationsWritten.retainAll(imageHandlingConfig.getLocationsToProcess());
                ImageHandler imageHandler;
                final String imageName = image.getMetaData().getName();
                synchronized (ScienceHandling.this) {
                    if (!Objects.equals(runNumber, previousRunNumber)) {
                        LOG.log(Level.INFO, () -> String.format("Run number changed from %s to %s, clearing status aggregators", previousRunNumber, runNumber));
                        previousRunNumber = runNumber;
                        Instant cutoff = Instant.now().minus(Duration.ofMinutes(1));
                        for (String subsystemToClear : imageHandlingConfig.getSubsystemsToClear()) {
                            long start = System.nanoTime();
                            statusAggregator.clearAgentDataOlderThan(subsystemToClear, cutoff);
                            LOG.log(Level.INFO, () -> String.format("Clearing status aggregator for %s took %,dns", subsystemToClear, System.nanoTime() - start));
                        }
                        runNumber = null;
                    }

                    imageHandler = new ImageHandler(getMetaDataSet(imageName), image, daqExecutor, imageHandlingConfig, rebNodes, isStreaming, postImageFileHandling);
                    getPerImageInfo(imageName).imageHandlerCreated(imageHandler);
                }
                Future<FileList> future = daqExecutor.submit(imageHandler);
                daqExecutor.submit(() -> {
                    try {
                        FileList filelist = future.get(Integer.getInteger("org.lsst.ccs.subsystem.imagehandling.FitsTimeoutSeconds", 30), TimeUnit.SECONDS);
                        ImageEventSender sender = new ImageEventSender(filelist, null, agent, locationsWritten, imageName);
                        sender.run();
                    } catch (InterruptedException | ExecutionException | TimeoutException x) {
                        ImageEventSender sender = new ImageEventSender(null, x, agent, locationsWritten, imageName);
                        sender.run();
                    } finally {
                        perImageInfo.remove(imageName);
                    }
                });
            }

            private boolean checkFolder(Image image) {
                // Check if image is in expected folder
                final String daqFolder = imageHandlingConfig.getDaqFolder();
                if (daqFolder != null && !"".equals(daqFolder)) {
                    if (!imageHandlingConfig.getDaqFolder().equals(image.getMetaData().getCreationFolderName())) {
                        return false;
                    }
                }
                return true;
            }
        });

    }

    void stop() throws DAQException {
        // If we were already started, shutdown existing thread pool
        if (daqExecutor != null) {
            daqExecutor.shutdown();
            try {
                LOG.log(Level.INFO, "Shutting down science handling executor with {0} active tasks", daqExecutor.getActiveCount());
                boolean success = daqExecutor.awaitTermination(30, TimeUnit.SECONDS);
                if (!success) {
                    LOG.log(Level.WARNING, "Timed out waiting for science handling executor to shutdown");
                }
            } catch (InterruptedException x) {
                LOG.log(Level.WARNING, "Unexpected interrupt exception while waiting for restart", x);
            }
            daqExecutor = null;
        }
        if (scienceStore != null) {
            scienceStore.close();
            scienceStore = null;
        }
    }

    /**
     * Called when darktime arrives.
     */
    void darkTimeArrived(String imageName) {
        synchronized (this) {
            // If ExposureTime was not explicitly set, then we attempt to set it here
            Map<String, Serializable> existingMap = getMetaDataSet(imageName);
            if (!existingMap.containsKey("ExposureTime")) {
                Object start = existingMap.get("startIntegrationTime");
                Object end =  existingMap.get("endIntegrationTime");
                // Note instanceof fails if null
                if (start instanceof CCSTimeStamp && end instanceof CCSTimeStamp) {
                    CCSTimeStamp tsStart = (CCSTimeStamp) start;
                    CCSTimeStamp tsEnd = (CCSTimeStamp) end;
                    double exposureTime = Duration.between(tsStart.getTAIInstant(), tsEnd.getTAIInstant()).toMillis()/1000.0;
                    LOG.log(Level.INFO, "Setting missing ExposureTime to {0}", exposureTime);
                    existingMap.put("ExposureTime", exposureTime);
                }
            }
            getPerImageInfo(imageName).darkTimeArrived();
        }
    }

    void runNumberArrived(String runNumber) {
        synchronized (this) {
            this.runNumber = runNumber;
        }
    }

    // LSSTCCSRAFTS-678 Store the metadata for a give imageName.
    void addMetaData(String imageName, Map<String, Serializable> headerMap) {
        LOG.log(Level.INFO, "Adding metadata keys for image {0}: {1}", new Object[]{imageName, headerMap.keySet()});
        synchronized (this) {
            Map<String, Serializable> existingMap = getMetaDataSet(imageName);
            existingMap.putAll(headerMap);
        }
    }

    PerImageInfo getPerImageInfo(String imageName) {
        return perImageInfo.computeIfAbsent(imageName, (in) -> new PerImageInfo(in));
    }
    
    // LSSTCCSRAFTS-678 Get the metadata for a given imageName.
    Map<String, Serializable> getMetaDataSet(String imageName) {
        return getPerImageInfo(imageName).metaDataSets;
    }

    void simulateTrigger(Location location, ImageMetaData meta, int[] registerList, Path rawData) throws DAQException {
        scienceStore.simulateTrigger(location, meta, registerList, rawData);
    }

    FileList fetchImage(String imageName) throws DAQException, InterruptedException, ExecutionException {
        Store store = new Store(imageHandlingConfig.getDaqPartition());
        final String daqFolder = imageHandlingConfig.getDaqFolder();
        Folder imageFolder = store.getCatalog().find(daqFolder);
        if (imageFolder == null) {
            throw new RuntimeException("Folder " + daqFolder + " not found");
        }
        Image image = imageFolder.find(imageName);
        if (image == null) {
            throw new RuntimeException("Image " + imageName + " not found");
        }
        LocationSet locationsWritten = new LocationSet(image.getMetaData().getLocations());
        locationsWritten.retainAll(imageHandlingConfig.getLocationsToProcess());
        Future<FileList> future = daqExecutor.submit(new ImageHandler(getMetaDataSet(imageName), image, daqExecutor, imageHandlingConfig, rebNodes, false, postImageFileHandling));
        return future.get();
    }

}
