package org.lsst.ccs.subsystem.imagehandling;

import com.fasterxml.jackson.core.JsonProcessingException;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import nom.tam.fits.BasicHDU;
import nom.tam.fits.BinaryTableHDU;
import nom.tam.fits.Fits;
import nom.tam.fits.FitsException;
import nom.tam.fits.Header;
import nom.tam.fits.HeaderCard;
import nom.tam.util.AsciiFuncs;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.imagenaming.Controller;
import org.lsst.ccs.imagenaming.Source;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.imagehandling.config.WaitForHeaderService;
import org.lsst.ccs.subsystem.imagehandling.data.AdditionalFile;
import org.lsst.ccs.subsystem.imagehandling.data.FileList;
import org.lsst.ccs.subsystem.imagehandling.data.ImageHeaderData;
import org.lsst.ccs.subsystem.imagehandling.data.JsonFile;
import org.lsst.ccs.utilities.image.FitsCheckSum;
import org.lsst.ccs.utilities.location.Location;
import org.lsst.ccs.subsystem.imagehandling.data.MeasuredShutterTime;
import org.lsst.ccs.utilities.location.LocationSet;

/**
 * Handles post image processing of FITS files, including adding header data
 * from the header service, and executing commands on generated FITS files. This
 * class is used for science images and guider images.
 *
 * @author tonyj
 */
class PostImageFileHandling {

    private static final Logger LOG = Logger.getLogger(PostImageFileHandling.class.getName());

    private final ImageHandlingConfig config;
    private final AtomicBoolean isHeaderServiceEnabled;

    private static class FutureShutterProfiles extends CompletableFuture<List<JsonFile>> {

        private final List<JsonFile> files = new ArrayList<>(2);

        void add(JsonFile file) {
            files.add(file);
            if (files.size() == 2) {
                this.complete(files);
            }
        }

        @Override
        public FutureShutterProfiles completeOnTimeout(List<JsonFile> value, long timeout, TimeUnit unit) {
            super.completeOnTimeout(value, timeout, unit);
            return this;
        }
    }
    private final LocationSet locationSet;
    
    /**
     * These map an image name to a future which will complete when the
     * specified data arrives.
     */
    private final Map<ImageName, CompletableFuture<List<ImageHeaderData.Header>>> futureHeaderDataForImage = new ConcurrentHashMap<>();
    private final Map<ImageName, CompletableFuture<Double>> futureMeasuredShutterTimeForImage = new ConcurrentHashMap<>();
    private final Map<ImageName, FutureShutterProfiles> futureShutterMotionProfileForImage = new ConcurrentHashMap<>();
    private final Map<ImageName, ImageProblematicHeaders> problematicHeadersForImage = new ConcurrentHashMap<>();
    private final CommandExecutor commandExecutor;
    private final AlertService alertService;

    PostImageFileHandling(ImageHandlingConfig config, CommandExecutor commandExecutor, AlertService alertService) {
        this.config = config;
        this.isHeaderServiceEnabled = new AtomicBoolean(true);
        this.commandExecutor = commandExecutor;
        this.alertService = alertService;
        this.locationSet = config.getLocationsToProcess();
    }

    /**
     * Given a list of files, this method will return a future which will
     * complete after the header service or shutter time data data has been
     * received and merged. This method is called from GuiderHandling and from
     * (Science) ImageHandling.
     *
     * @param filesToProcess The list of files to process
     * @param imageName The image name
     * @return The Future.
     */
    CompletableFuture<FileList> handleAsynchronousData(CompletableFuture<FileList> filesToProcess, ImageName imageName) {
        // We only need to wait for the header service data if configured to do so
        boolean needToWaitForHeaderService = config.getWaitForHeaderService() == WaitForHeaderService.ALWAYS
                || (config.getWaitForHeaderService() == WaitForHeaderService.AUTO && isHeaderServiceEnabled.get());
        // Never wait for header service data if image is triggered by CCS(LSSTCCSRAFTS-780)
        needToWaitForHeaderService = needToWaitForHeaderService && imageName.getController() != Controller.CCS;
        boolean needToWaitForMeasuredShutterTime = imageName.getSource() == Source.MainCamera; // Only if lsstcam
        return handleAsynchronousData(filesToProcess, imageName, needToWaitForHeaderService, needToWaitForMeasuredShutterTime);

    }

    private static class FileListAndShutterTime {

        private final FileList filelist;
        private final Double shutterTime;

        public FileListAndShutterTime(FileList filelist, Double shutterTime) {
            this.filelist = filelist;
            this.shutterTime = shutterTime;
        }

        public FileList getFilelist() {
            return filelist;
        }

        public Double getShutterTime() {
            return shutterTime;
        }

    }

    CompletableFuture<FileList> handleAsynchronousData(CompletableFuture<FileList> filesToProcess, ImageName imageName, boolean needToWaitForHeaderService, boolean needToWaitForMeasuredShutterTime) {
        CompletableFuture<FileList> resultingFileList = filesToProcess;

        // If we need to wait for header service data, then this code returns a future which will complete when 
        // * the original filesToProcess future is complete AND
        // * header service has arrived, and the resulting data base been merged into the FITS file, or the data arrival timed out
        if (needToWaitForHeaderService) {
            CompletableFuture<List<ImageHeaderData.Header>> futureHeaderData = getHeaderDataFutureForImage(imageName);
            resultingFileList = resultingFileList.thenCombine(futureHeaderData, (fl, headerData) -> {
                return mergeHeaderServiceData(imageName, fl, headerData);
            });
        }
        // If we need to wait for measured shutter time, then this code returns a future which will complete when
        // * The original filesToProcess future is complete AND
        // * The measured shutter time has arrived, and the resulting data has been merged into the header service, or the data arrival timed out
        if (needToWaitForMeasuredShutterTime) {
            CompletableFuture<Double> futureMeasuredShutterTimeData = getMeasuredShutterTimeFutureForImage(imageName);
            CompletableFuture<FileListAndShutterTime> combined = resultingFileList.thenCombine(futureMeasuredShutterTimeData, (fl, measuredShutterTime) -> {
                return new FileListAndShutterTime(mergeShutterTimeData(imageName, fl, measuredShutterTime), measuredShutterTime);
            });
            // Now deal with shutterMotionProfiles
            FutureShutterProfiles futureShutterMotionProfile = getShutterMotionProfilesForImage(imageName).completeOnTimeout(null, 15, TimeUnit.SECONDS);
            resultingFileList = combined.thenCombine(futureShutterMotionProfile, (fileListAndShutterTime, jsonFiles) -> {
                final Double shutterTime = fileListAndShutterTime.getShutterTime();
                if (shutterTime != null && shutterTime  > 0) {
                    return mergeShutterMotionProfiles(imageName, fileListAndShutterTime.getFilelist(), jsonFiles);
                } else {
                    return fileListAndShutterTime.getFilelist();
                }
            });
        }
                
        return resultingFileList;
    }

    FileList mergeHeaderServiceData(ImageName imageName, FileList fl, List<ImageHeaderData.Header> headerData) {
        String message = null;
        if (headerData == null) {
            message = String.format("Header service data for %s did not arrive in a timely manner", imageName);
            LOG.log(Level.WARNING, message);

        } else {
            try {
                fixupFitsFiles(fl, headerData, false, imageName);
            } catch (FitsException | IOException x) {
                message = "Error while fixing FITS headers";
                LOG.log(Level.WARNING, message, x);
            }
        }
        if (message != null && alertService != null && config.getRaiseAlertOnMissingOrBadHeaderServiceData()) {
            Alert alert = new Alert(ImageHandlingClient.MISSING_HEADERSERVICEDATA_ALERTID, "Missing or malformed header service data");
            alertService.raiseAlert(alert, AlertState.ALARM, message);
        }
        return fl;

    }

    FileList mergeShutterTimeData(ImageName imageName, FileList fl, Double measuredShutterTime) {
        String message = null;
        if (measuredShutterTime == null) {
            message = String.format("The measured shutter time for %s did not arrive in a timely manner", imageName);
            LOG.log(Level.WARNING, message);
        } else {
            try {
                List<ImageHeaderData.Header> changedHeaders = new ArrayList();
                ImageHeaderData.Header header = new ImageHeaderData.Header("SHUTTIME", measuredShutterTime, "Shutter Exposure Time");
                changedHeaders.add(header);
                if (measuredShutterTime > 0) {
                    ImageHeaderData.Header header2 = new ImageHeaderData.Header("XPOSURE", measuredShutterTime, "Effective exposure duration");
                    changedHeaders.add(header2);
                }
                fixupFitsFiles(fl, changedHeaders, true, imageName);
            } catch (FitsException | IOException x) {
                message = "Error while fixing FITS headers";
                LOG.log(Level.WARNING, message, x);
            }
        }
        if (message != null && alertService != null && config.getRaiseAlertOnMissingOrBadHeaderServiceData()) {
            // TODO: Use a different alert id?
            Alert alert = new Alert(ImageHandlingClient.MISSING_HEADERSERVICEDATA_ALERTID, "Missing measured shutter data");
            alertService.raiseAlert(alert, AlertState.ALARM, message);
        }
        return fl;

    }

    FileList mergeShutterMotionProfiles(ImageName imageName, FileList fl, List<JsonFile> jsonFiles) {
        if (jsonFiles == null || jsonFiles.size() < 2) {
            String message = String.format("The shutter motion profiles for %s did not arrive in a timely manner jsonFiles=%s", imageName, jsonFiles);
            LOG.log(Level.WARNING, message);
        } else {
            try {
                JsonFile openJson = jsonFiles.get(0);
                JsonFile closeJson = jsonFiles.get(1);
                // Seems unlikely,  but let check if they came in the expected order
                if (openJson.getFileName().endsWith("Close")) {
                    openJson = jsonFiles.get(1);
                    closeJson = jsonFiles.get(0);
                }
                addShutterMotionProfilesToFitsFile(fl, openJson, closeJson, imageName);
            } catch (FitsException | IOException  x) {
                String message = "Error while fixing FITS headers";
                LOG.log(Level.WARNING, message, x);
            }
        }
        return fl;
    }

    void handleFitsFileCommands(CompletableFuture<FileList> filesToProcess, Location location, ImageName imageName, String mode) {
        if (!config.getCommands().isEmpty()) {
            filesToProcess.thenAccept(fl -> {
                Map<String, String> env = new HashMap<>();
                env.put("BOARD", location.getBoardName());
                env.put("RAFT", location.getRaftName());
                env.put("IMAGENAME", imageName.toString());
                env.put("MODE", mode);
                env.put("SEQNO", imageName.getNumberString());
                env.put("DATE", imageName.getDateString());
                env.put("CONTROLLER", imageName.getController().getCode());
                env.put("SOURCE", imageName.getSource().getCode());
                commandExecutor.executeFitsFileList(fl, env, location, imageName);
            });
        }
    }

    
    void completePostImageFileHandlingForImage(ImageName imageName) {
        ImageProblematicHeaders imageProblematicHeaders = getImageProblematicHeaders(imageName);
        imageProblematicHeaders.logIfNeeded();
        
        //We are done for the image: cleaning up the maps
        futureHeaderDataForImage.remove(imageName);
        futureMeasuredShutterTimeForImage.remove(imageName);
        futureShutterMotionProfileForImage.remove(imageName);
        problematicHeadersForImage.remove(imageName);
    }
    
    private static class ImageProblematicHeaders {
        
        private final Map<String,List<CardValues>> warningsDifferMap = new ConcurrentHashMap<>();
        private final Map<String,List<CardValues>> warningsNullCardMap = new ConcurrentHashMap<>();
        private final ImageName imageName;
                
        ImageProblematicHeaders(ImageName imageName) {
            this.imageName = imageName;
        }
        
        void addCardValue(CardValues cardValues) {
            String keyword = cardValues.headerKeyword;
            Map<String,List<CardValues>> map = cardValues.ccsValue != null ? warningsDifferMap : warningsNullCardMap;            
            List<CardValues> list = map.computeIfAbsent(keyword, (k) -> new CopyOnWriteArrayList<>());
            if (!list.contains(cardValues)) {
                list.add(cardValues);
            }
        }        
        
        void logIfNeeded() {        
            if (!warningsDifferMap.isEmpty() || !warningsNullCardMap.isEmpty()) {
                StringBuilder sb = new StringBuilder("The following header cards inconsistencies have been detected between the CCS value and the header service provided value for image " + imageName + ":\n");
                if (!warningsDifferMap.isEmpty()) {
                    sb.append("\nThe following header keywords have different values:\n");
                    for (List<CardValues> list : warningsDifferMap.values()) {
                        for (CardValues cv : list) {
                            sb.append("   ").append(cv.toString()).append("\n");
                        }
                    }
                }
                if (!warningsNullCardMap.isEmpty()) {
                    sb.append("\nThe following header keywords are null in CCS:\n");
                    for (List<CardValues> list : warningsNullCardMap.values()) {
                        for (CardValues cv : list) {
                            sb.append("   ").append(cv.toString()).append("\n");
                        }
                    }
                }
                sb.append("\n");
                LOG.log(Level.INFO, sb.toString());
            }
        }
    }
    
    //There is only one instance of this class per ImageHandler.
    //This method is invoked multiple times for each of the Rebs handled by the 
    //ImageHandler.
    void fixupFitsFiles(FileList fileList, List<ImageHeaderData.Header> headerData, boolean overwrite, ImageName imageName) throws FitsException, IOException {
        
        ImageProblematicHeaders imageProblematicHeaders = getImageProblematicHeaders(imageName);
        
        for (File file : fileList) {
            Fits fits = new Fits(file);
            final BasicHDU<?> hdu = fits.getHDU(0);
            Header header = hdu.getHeader();
            long deltaCheckSum = 0;
            for (ImageHeaderData.Header headerServiceHeader : headerData) {
                String keyword = headerServiceHeader.getKeyword();
                if (keyword == null || "COMMENT".equals(keyword) || "FILENAME".equals(keyword)) {
                    continue;
                }
                if (keyword.startsWith("HIERARCH")) {
                    keyword = keyword.replace(" ", ".");
                }
                HeaderCard card = header.findCard(keyword);
                Object newValue = headerServiceHeader.getValue();
                String newValueAsString = newValue == null ? null : newValue.toString();

                if (card != null) {
                    String oldValue = card.getValue();

                    //Set the value provided by the header service only if the existing value
                    //is either null or empty and the service provided value is not null and is not empty.
                    if ((oldValue == null || oldValue.isEmpty() || overwrite) && newValue != null) {
                        long oldCheckSum = FitsCheckSum.checksum(AsciiFuncs.getBytes(card.toString()));
                        if (newValue instanceof String newValueString) {
                            if (!newValueString.isEmpty()) {
                                if (card.isStringValue()) {
                                    card.setValue(newValueString);
                                } else {
                                    card = new HeaderCard(card.getKey(), newValueString, card.getComment());
                                    header.updateLine(card.getKey(), card);
                                }
                            }
                        } else {
                            if (card.isStringValue()) {
                                // This should never happen
                                LOG.log(Level.WARNING, "Header service keyword {0} of type {1} but isString is true", new Object[]{keyword, newValue.getClass()});
                            }
                            if (newValue instanceof Double newValueDouble) {
                                if (!Double.isNaN(newValueDouble)) {
                                    card.setValue(newValueDouble);
                                }
                            } else if (newValue instanceof Boolean newValueBoolean) {
                                card.setValue(newValueBoolean);
                            } else if (newValue instanceof Integer newValueInteger) {
                                card.setValue(newValueInteger);
                            } else {
                                LOG.log(Level.WARNING, "Header service keyword {0} of unexpected type {1} ignored", new Object[]{keyword, newValue.getClass()});
                            }
                        }
                        long newCheckSum = FitsCheckSum.checksum(AsciiFuncs.getBytes(card.toString()));
                        deltaCheckSum += newCheckSum - oldCheckSum;
                    } else {
                        Class cardType = card.valueType();
                        //Check here if the header service provided value differs from the ccs value
                        boolean differ = true;
                        if (oldValue != null && newValue != null) {
                            //First we trim the strings
                            oldValue = oldValue.trim();
                            //newValue = newValue.trim();
                            if (cardType == Boolean.class) {
                                //Booleans in the HeaderCard are represented as "T" and "F"
                                differ = oldValue.toLowerCase().charAt(0) != newValueAsString.toLowerCase().charAt(0);
                            } else {
                                differ = !Objects.equals(oldValue, newValueAsString);
                            }
                        }
                        if ( differ ) {
                            CardValues cardValues = new CardValues(keyword,newValueAsString, oldValue);
                            imageProblematicHeaders.addCardValue(cardValues);
                        }
                    }
                } else { // No existing card
                    CardValues cardValues = new CardValues(keyword,newValueAsString, null);
                    imageProblematicHeaders.addCardValue(cardValues);
                }
            }
            FitsCheckSum.updateCheckSum(header, deltaCheckSum);
            header.rewrite();
        }
        
    }
    
    private void addShutterMotionProfilesToFitsFile(FileList fileList, JsonFile openFile, JsonFile closeFile, ImageName imageName) throws FitsException, JsonProcessingException, IOException {
        BinaryTableHDU openHDU = ShutterMotionProfileHandler.createHDU(openFile.toString());
        BinaryTableHDU closeHDU = ShutterMotionProfileHandler.createHDU(closeFile.toString());

        for (File file : fileList) {
            ShutterMotionProfileHandler.appendHDUsToFitsFile(file, openHDU, closeHDU);
        }
        // If we want to write anything into the primary header now is the time        
        List<String> primaryHeaderEntries = new ArrayList<>();
        primaryHeaderEntries.add("HIERARCH.SHUTTER.STARTTIME.TAI.ISOT");
        primaryHeaderEntries.add("HIERARCH.SHUTTER.STARTTIME.TAI.MJD");
        primaryHeaderEntries.add("HIERARCH.SHUTTER.SIDE");
        primaryHeaderEntries.add("HIERARCH.SHUTTER.MODEL");
        primaryHeaderEntries.add("HIERARCH.SHUTTER.HALLSENSORFIT.MODELSTARTTIME");
        primaryHeaderEntries.add("HIERARCH.SHUTTER.HALLSENSORFIT.PIVOTPOINT1");
        primaryHeaderEntries.add("HIERARCH.SHUTTER.HALLSENSORFIT.PIVOTPOINT2");
        primaryHeaderEntries.add("HIERARCH.SHUTTER.HALLSENSORFIT.JERK0");
        primaryHeaderEntries.add("HIERARCH.SHUTTER.HALLSENSORFIT.JERK1");
        primaryHeaderEntries.add("HIERARCH.SHUTTER.HALLSENSORFIT.JERK2");
        List<ImageHeaderData.Header> headers = new ArrayList();
        for (String primaryHeaderEntry : primaryHeaderEntries) {
            String openSide = openHDU.getHeader().getStringValue(primaryHeaderEntry);
            if (openSide != null) {
                ImageHeaderData.Header header1 = new ImageHeaderData.Header(primaryHeaderEntry.replace("SHUTTER.", "SHUTTER.OPEN."), openSide, "");
                headers.add(header1);
            } else {
                double openDouble = openHDU.getHeader().getDoubleValue(primaryHeaderEntry, -999.0);
                if (openDouble != -999.0) {
                    ImageHeaderData.Header header2 = new ImageHeaderData.Header(primaryHeaderEntry.replace("SHUTTER.", "SHUTTER.OPEN."), openDouble, "");
                    headers.add(header2);
                }
            }

            String closeSide = closeHDU.getHeader().getStringValue(primaryHeaderEntry);
            if (closeSide != null) {
                ImageHeaderData.Header header1 = new ImageHeaderData.Header(primaryHeaderEntry.replace("SHUTTER.", "SHUTTER.CLOSE."), closeSide, "");
                headers.add(header1);
            } else {
                double closeDouble = closeHDU.getHeader().getDoubleValue(primaryHeaderEntry, -999.0);
                if (closeDouble != -999.0) {
                    ImageHeaderData.Header header2 = new ImageHeaderData.Header(primaryHeaderEntry.replace("SHUTTER.", "SHUTTER.CLOSE."), closeDouble, "");
                    headers.add(header2);
                }
            }
        }
        fixupFitsFiles(fileList, headers, true, imageName);
    }

    private static class CardValues {

        private final String serviceValue;
        private final String ccsValue;
        private final String headerKeyword;

        CardValues(String headerKeyword, String serviceValue, String ccsValue) {
            this.serviceValue = serviceValue;
            this.ccsValue = ccsValue;
            this.headerKeyword = headerKeyword;
        }

        @Override
        public boolean equals(Object obj) {
            CardValues in = (CardValues) obj;
            if (!Objects.equals(in.serviceValue, serviceValue)) {
                return false;
            }
            if (!Objects.equals(in.headerKeyword, headerKeyword)) {
                return false;
            }
            return Objects.equals(in.ccsValue, ccsValue);
        }

        @Override
        public int hashCode() {
            int hash = 7;
            hash = 73 * hash + Objects.hashCode(this.serviceValue);
            hash = 73 * hash + Objects.hashCode(this.ccsValue);
            hash = 173 * hash + Objects.hashCode(this.headerKeyword);
            return hash;
        }

        @Override
        public String toString() {
            return headerKeyword + "\t header_service: " + serviceValue + "\t ccs: " + ccsValue;
        }

    }

    /**
     * Look for an existing, or create a new, future which is waiting for the
     * header data for this image, and complete it by providing the measured
     * data.
     *
     * @param imageName The corresponding image name
     * @param headers The newly arrived headers.
     */
    void headerDataArrived(ImageName imageName, List<ImageHeaderData.Header> headers) {
        CompletableFuture<List<ImageHeaderData.Header>> futureHeaderData = getHeaderDataFutureForImage(imageName);
        futureHeaderData.complete(headers);
    }

    /**
     * Look for an existing, or create a new, future which is waiting for the
     * measured shutter time for this image, and complete it by providing the
     * measured data.
     *
     * @param imageName The corresponding image name
     * @param headers The newly arrived headers.
     */
    void measuredShutterTimeArrived(MeasuredShutterTime shutterTime) {
        CompletableFuture<Double> futureMeasuredShutterTime = getMeasuredShutterTimeFutureForImage(shutterTime.getImageName());
        futureMeasuredShutterTime.complete(shutterTime.getMeasuredShutterOpenTime());
    }

    void shutterMotionProfileArrived(JsonFile jsonFile) {
        // We need an open and a close
        FutureShutterProfiles futureShutterMotionProfile = getShutterMotionProfilesForImage(jsonFile.getObsId());
        futureShutterMotionProfile.add(jsonFile);
    }

    /**
     * Look up or create a Future which will complete when either the header
     * data arrives, or after a timeout.
     *
     * @param imageName The corresponding imageName
     * @return The future
     */
    private CompletableFuture<List<ImageHeaderData.Header>> getHeaderDataFutureForImage(ImageName imageName) {
        return futureHeaderDataForImage.computeIfAbsent(imageName, s -> new CompletableFuture<List<ImageHeaderData.Header>>().completeOnTimeout(null, Integer.getInteger("org.lsst.ccs.subsystem.imagehandling.headerTimeoutSeconds", 15), TimeUnit.SECONDS));
    }

    /**
     * Look up or create a Future which will complete when either the measured
     * shutter time arrives, or after a timeout.
     *
     * @param imageName The corresponding imageName
     * @return The future
     */
    private CompletableFuture<Double> getMeasuredShutterTimeFutureForImage(ImageName imageName) {
        return futureMeasuredShutterTimeForImage.computeIfAbsent(imageName, s -> new CompletableFuture<Double>().completeOnTimeout(null, Integer.getInteger("org.lsst.ccs.subsystem.imagehandling.headerTimeoutSeconds", 15), TimeUnit.SECONDS));
    }

    private FutureShutterProfiles getShutterMotionProfilesForImage(ImageName imageName) {
        // We cant use complete on timeout here, because the time starts when the first motion profile arrives
        return futureShutterMotionProfileForImage.computeIfAbsent(imageName, s -> new FutureShutterProfiles());
    }    
    
    /**
     * Get an instance of the problematic headers for a given ImageName
     * @param imageName the corresponding imageName
     * @return the problematic headers instance
     */
    private ImageProblematicHeaders getImageProblematicHeaders(ImageName imageName) {
        return problematicHeadersForImage.computeIfAbsent(imageName, s -> new ImageProblematicHeaders(s));
    }

    void setHeaderServiceEnabled(boolean enabled) {
        isHeaderServiceEnabled.set(enabled);
    }

    CompletableFuture<File> handleAdditionalFile(AdditionalFile additionalFile) {

        if (!config.getAdditionalFileCommands().isEmpty()) {
            Map<String, String> props = new HashMap<>();
            props.put("FileName", additionalFile.getFileName());
            props.put("FileType", additionalFile.getFileType());
            ImageName obsId = additionalFile.getObsId();
            props.put("ImageName", obsId.toString());
            props.put("ImageDate", obsId.getDateString());
            props.put("ImageNumber", obsId.getNumberString());
            props.put("ImageController", obsId.getController().getCode());
            props.put("ImageSource", obsId.getSource().getCode());
            // For compatibility with FITSFILES
            props.put("SEQNO", obsId.getNumberString());
            props.put("DATE", obsId.getDateString());
            props.put("CONTROLLER", obsId.getController().getCode());
            props.put("SOURCE", obsId.getSource().getCode());
            props.put("MODE", "OTHER");
            File file = config.getAdditionalFileName(additionalFile, props);
            try {
                // TODO: Write file asynchronously?
                try (BufferedOutputStream out = new BufferedOutputStream(new FileOutputStream(file))) {
                    additionalFile.writeFile(out);
                }
                Map<String, String> env = props.entrySet().stream().collect(Collectors.toMap(e -> e.getKey().toUpperCase(), e -> e.getValue()));
                return commandExecutor.executeAdditional(file, obsId, additionalFile.getFileType(), env);
            } catch (IOException x) {
                LOG.log(Level.SEVERE, "Error handling additional file: " + additionalFile.getFileName(), x);
                return CompletableFuture.failedFuture(x);
            }
        }
        return CompletableFuture.completedFuture(null);
    }

}
