package org.lsst.ccs.subsystem.imagehandling;

import org.lsst.ccs.subsystem.imagehandling.data.FileList;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.daq.ims.DAQException;
import org.lsst.ccs.daq.ims.Image;
import org.lsst.ccs.daq.ims.ImageListener;
import org.lsst.ccs.daq.ims.Store;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.subsystem.imagehandling.data.ImageReceivedEvent;

public class ImageHandlingClient  implements HasLifecycle {

    private static final Logger LOG = Logger.getLogger(ImageHandlingClient.class.getName());
    
    // TODO: Should we be using a core provided executor service?
    private final ExecutorService executor = Executors.newCachedThreadPool();

    @LookupPath
    private String path;
    
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private ImageHandlingConfig imageHandlingConfig;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private List<RebNode> rebs = new ArrayList<>();

    @LookupField(strategy = LookupField.Strategy.TOP)
    private Agent agent;

    @Override
    public void build() {
    }
    
    @Override
    public void start() {
        LOG.log(Level.FINE, "Starting ImageHandling client {0})", path);
    }

    @Override
    public void postStart() {
        try {
            Store store = new Store(imageHandlingConfig.getDaqPartition());
            store.addImageListener(new ImageListener(){
                @Override
                public void imageCreated(Image image) {
                    // Ignore for now, will need to use this to support streaming.
                }

                @Override
                public void imageComplete(Image image) {
                    KeyValueData kvd = new KeyValueData(ImageReceivedEvent.EVENT_KEY, new ImageReceivedEvent(image.getMetaData()));
                    agent.publishSubsystemDataOnStatusBus(kvd);
                    
                    // Check if image is in expected folder
                    final String daqFolder = imageHandlingConfig.getDaqFolder();
                    if (daqFolder != null && !"".equals(daqFolder)) {
                        if (!imageHandlingConfig.getDaqFolder().equals(image.getMetaData().getCreationFolderName())) {
                            return;
                        }
                    }
                    Future<FileList> future = executor.submit(new ImageHandler(image, executor, imageHandlingConfig, rebs));
                    executor.submit(new ImageEventSender(future, agent, imageHandlingConfig.getLocationsToProcess(), image.getMetaData().getName()));
                }
            });
        } catch (DAQException x) {
            // TODO: Is this the right thing to do? unfortunately HasLifecycle 
            // javadoc does not say what happens if an exception is thrown.
            throw new RuntimeException("Failed to connect to DAQ", x);
        }
    }
    
    ExecutorService getExecutorService() {
        return executor;
    }
    
}
