package org.lsst.ccs.subsystem.imagehandling;

import java.util.Arrays;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.AgentCategory;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;
import org.lsst.ccs.bus.messages.StatusConfigurationInfo;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.StatusMessageListener;

/**
 * A class that listens to Configuration messages coming from the focal-plane
 * subsystem.
 *
 * @author The LSST CCS Team
 */
class FocalPlaneConfigurationHandler implements HasLifecycle {
        
    private volatile String[] metaDataRegisters;
    private ConfigurationListener configListener;
    private FocalPlaneAgentPresenceListener focalPlanePresenceListener;
    private static final Logger LOG = Logger.getLogger(FocalPlaneConfigurationHandler.class.getName());

    
    @LookupField(strategy = LookupField.Strategy.TOP)
    Agent agent;

    @Override
    public void start() {
        focalPlanePresenceListener = new FocalPlaneAgentPresenceListener(agent);
        agent.getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(focalPlanePresenceListener);
    }

    @Override
    public void shutdown() {
        agent.getMessagingAccess().getAgentPresenceManager().removeAgentPresenceListener(focalPlanePresenceListener);
    }

    public String[] getMetaDataRegisters() {
        return metaDataRegisters;
    }
    
    private static boolean isFocalPlaneAgent(AgentInfo agentInfo) {
        return AgentCategory.FOCAL_PLANE.name().equals(agentInfo.getAgentProperty(AgentCategory.AGENT_CATEGORY_PROPERTY));
    }
    
    private class ConfigurationListener implements StatusMessageListener {

        @Override
        public void onStatusMessage(StatusMessage msg) {
            if ( ! (msg instanceof StatusConfigurationInfo) ) {
                return;
            }
            
            //Questions that might affect the implementation:
            //- are the configuration parameters final?
            //If so we don't need to listen to all status messages.
            if (isFocalPlaneAgent(msg.getOriginAgentInfo())) {
                ConfigurationInfo ci = ((StatusConfigurationInfo)msg).getConfigurationInfo();
                for ( ConfigurationParameterInfo cpi : ci.getAllParameterInfo() ) {
                    if ( cpi.getComponentName().equals("focalPlaneConfig") ) {
                        if ( cpi.getParameterName().equals("metaDataRegisters") ) {
                            metaDataRegisters = (String[]) cpi.getCurrentValueObject();
                            LOG.log(Level.FINEST, "Updating metaDataRegister to {0}", new Object[]{Arrays.asList(metaDataRegisters)});
                        }
                    }
                }
            }
        }
    }
    
    
    private class FocalPlaneAgentPresenceListener implements AgentPresenceListener {

        private final Agent agent;
        
        FocalPlaneAgentPresenceListener(Agent a) {
            this.agent = a;
        }

        @Override
        public void connecting(AgentInfo... agents) {
            for ( AgentInfo a : agents ) {
                if ( isFocalPlaneAgent(a) ) {
                    if ( configListener != null ) {
                        throw new RuntimeException("A configuration listener is already registered. There should be only one!");
                    }
                    configListener = new ConfigurationListener();                    
                    agent.getMessagingAccess().addStatusMessageListener(configListener);
                }                
            }
        }

        @Override
        public void disconnected(AgentInfo... agents) {
            for ( AgentInfo a : agents ) {
                if ( isFocalPlaneAgent(a) ) {
                    if ( configListener == null ) {
                        throw new RuntimeException("The configuration listener has already been removed!");
                    }
                    agent.getMessagingAccess().removeStatusMessageListener(configListener);
                    configListener = null;
                    metaDataRegisters = null;
                }
            }
        }
    }
    
}
