package org.lsst.ccs.subsystem.imagehandling.data;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.Serializable;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.lsst.ccs.bus.annotations.SkipEncoding;
import org.lsst.ccs.imagenaming.ImageName;

/**
 * A simple implementation to support sending  ecsv files to USDF
 * @author tonyj
 */
@SkipEncoding
public class ECSVFile extends AdditionalFile {

    private final List<Column> columns;
    private String delimiter = ",";
    private final String data;
    private Map<String, Serializable> meta;
    private static final long serialVersionUID = 1;
    static final String MIME_TYPE = "text/csv";

    public ECSVFile(List<Column> columns, String data, String fileName, String fileType, ImageName obsId) {
        super(fileName, fileType, obsId, DEFAULT_VERSION, MIME_TYPE);
        this.columns = columns;
        this.data = data;
    }

    public String getDelimiter() {
        return delimiter;
    }

    public void setDelimiter(String delimiter) {
        this.delimiter = delimiter;
    }
    
    public void addMetaData(Map<String, Serializable> meta) {
        this.meta = meta;
    }

    @Override
    public void writeFile(OutputStream out) throws IOException {
        Map<String, Object> yaml = new LinkedHashMap<>();
        Map<String, Serializable> header = new LinkedHashMap<>();
        header.put("fileName", getFileName());
        header.put("fileType", getFileType());
        header.put("obsId", getObsId().toString());
        header.put("version", String.valueOf(getVersion()));
        if (meta != null) {
            header.putAll(meta);
        }
        yaml.put("datatype", columns);
        yaml.put("delimiter", delimiter);
        yaml.put("meta", header);

        try (PrintStream print = new PrintStream(out)) {
            print.println("# %ECSV 1.0");

            StringWriter writer = new StringWriter();
            ObjectMapper om = new ObjectMapper(new YAMLFactory());
            om.writeValue(writer, yaml);
            writer.close();
            BufferedReader reader = new BufferedReader(new StringReader(writer.toString()));
            for (;;) {
                String line = reader.readLine();
                if (line == null) {
                    break;
                }
                print.print("# ");
                print.println(line);
            }
            print.println(columns.stream().map(Column::getName).collect(Collectors.joining(delimiter)));
            print.print(data);
        }
    }

    public static class Header {

        private final String fileName;
        private final String fileType;
        private final String obsId;

        public Header(String fileName, String fileType, String obsId) {
            this.fileName = fileName;
            this.fileType = fileType;
            this.obsId = obsId;
        }

        public String getFileName() {
            return fileName;
        }

        public String getFileType() {
            return fileType;
        }

        public String getObsId() {
            return obsId;
        }

    }

    public static class Column implements Serializable {

        private final String name;
        private final String format;
        private final String datatype;
        private final String description;
        private final String units;
        private static final long serialVersionUID = 1;

        public Column(String name, String format, String datatype, String description) {
            this(name, format, datatype, description, "unitless");
        }

        public Column(String name, String format, String datatype, String description, String units) {
            this.name = name;
            this.format = format;
            this.datatype = datatype;
            this.description = description;
            this.units = units;
        }

        public String getName() {
            return name;
        }

        public String getFormat() {
            return format;
        }

        public String getDatatype() {
            return datatype;
        }

        public String getDescription() {
            return description;
        }

        public String getUnits() {
            return units;
        }
    }

}
