/*
 * Decompiled with CFR 0.152.
 */
package org.lsst.ccs.visualization.server;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.nio.file.FileStore;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileTime;
import java.time.Duration;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;

public class CacheManager
implements Closeable {
    private static final Logger logger = Logger.getLogger(CacheManager.class.getName());
    private final FileStore fileStore;
    private final Path cachePath;
    private final long freeSpaceTarget;
    private final ScheduledExecutorService scheduler = Executors.newSingleThreadScheduledExecutor();

    public CacheManager(File dir, long freeSpaceTarget) throws IOException {
        this.freeSpaceTarget = freeSpaceTarget;
        this.cachePath = dir.toPath();
        this.fileStore = Files.getFileStore(dir.toPath());
        logger.info(String.format("Directory %s, totalSpace=%,d unallocatedSpece=%,d usableSpace=%,d\n", dir, this.fileStore.getTotalSpace(), this.fileStore.getUnallocatedSpace(), this.fileStore.getUsableSpace()));
        this.makeSpace(freeSpaceTarget);
    }

    public void start(Duration scanPeriod) {
        this.scheduler.scheduleAtFixedRate(this::makeSpace, scanPeriod.getSeconds(), scanPeriod.getSeconds(), TimeUnit.SECONDS);
    }

    public void scanNow() {
        this.scheduler.execute(this::makeSpace);
    }

    private void makeSpace() {
        try {
            this.makeSpace(this.freeSpaceTarget);
        }
        catch (IOException ex) {
            logger.log(Level.SEVERE, "Unable to meet free space target", ex);
        }
    }

    private void makeSpace(long size) throws IOException {
        if (size < this.fileStore.getUsableSpace()) {
            return;
        }
        if (size > this.fileStore.getTotalSpace()) {
            throw new RuntimeException(String.format("Filestore not big enough to free %,d bytes", size));
        }
        SortedSet<FileWithCreationDate> files = this.scanFiles();
        for (FileWithCreationDate file : files) {
            Files.delete(file.path);
            logger.log(Level.INFO, "Deleting file {0} free space {1}", new Object[]{file, this.fileStore.getUsableSpace()});
            if (this.fileStore.getUsableSpace() < size) continue;
            break;
        }
    }

    private SortedSet<FileWithCreationDate> scanFiles() throws IOException {
        final TreeSet<FileWithCreationDate> files = new TreeSet<FileWithCreationDate>();
        Files.walkFileTree(this.cachePath, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                if (!attrs.isDirectory()) {
                    files.add(new FileWithCreationDate(file, attrs));
                }
                return FileVisitResult.CONTINUE;
            }
        });
        return files;
    }

    @Override
    public void close() throws IOException {
        this.scheduler.shutdownNow();
        try {
            this.scheduler.awaitTermination(10L, TimeUnit.SECONDS);
        }
        catch (InterruptedException ex) {
            InterruptedIOException x = new InterruptedIOException("Error while shutting down cache scheduler");
            x.initCause(ex);
            throw x;
        }
    }

    public static void main(String[] args) throws IOException, InterruptedException {
        try (CacheManager cacheManager = new CacheManager(new File("/mnt/ramdisk/"), 2000000000L);){
            SortedSet<FileWithCreationDate> scanFiles = cacheManager.scanFiles();
            scanFiles.forEach(f -> System.out.println(f));
            cacheManager.start(Duration.ofSeconds(10L));
            Thread.sleep(60000L);
        }
    }

    private static class FileWithCreationDate
    implements Comparable<FileWithCreationDate> {
        private final FileTime creationTime;
        private final Path path;
        private final long size;

        private FileWithCreationDate(Path file, BasicFileAttributes attrs) {
            this.path = file;
            this.creationTime = attrs.creationTime();
            this.size = attrs.size();
        }

        @Override
        public int compareTo(FileWithCreationDate other) {
            return this.creationTime.compareTo(other.creationTime);
        }

        public String toString() {
            return "FileWithCreationDate{creationTime=" + this.creationTime + ", file=" + this.path + ", size=" + this.size + '}';
        }
    }
}

