package org.lsst.ccs.subsystems.fcs.common;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.ControlWord;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposState;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposParameter;
import org.lsst.ccs.subsystems.fcs.FcsAlerts;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 *
 * @author virieux
 */
public interface EPOSControllerForCarousel extends EPOSController {

    int getPositionSensorType();

    void setPositionSensorType(int pst);

    void setHoldingBrakes(boolean holdingBrake);

    boolean isHoldingBrakes();

    /**
     * For carousel, status word is read by PDO so no need to read by SDO. During
     * actions, read by PDO are done every 250ms.
     *
     * @return status word
     */
    @Override
    default boolean isTargetReached() {
        /* target is reached when bit10 of status word is 1 */
        return ((getStatusWord() >> 10) & 1) == 1;
    }

    /**
     * Set PositionSensorType to Absolute encoder SSI (value=4)
     * See https://jira.slac.stanford.edu/browse/LSSTCCSFCS-667 to understand why this function is so complex.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "GoToSwitchOnDisabled and set PositionSensorType to Absolute encoder SSI : 4")
    default void setPositionSensorTypeEncoderSSI() {
        FCSLOG.info(getName() + " setting PositionSensorType Absolute encoder SSI");

        long beginTime = System.currentTimeMillis();

        int nbTryMax = 15; // Number of times we try to switch mode
        int nbRecoverMax = 4; // Number of times we try to recover controller if faulted

        boolean controlerFailed = false; // Check of controller already failed
        boolean swapDone = false; // Check if swap is done

        int i = 0;
        while ((i++ < nbTryMax) && !swapDone) {
            goToSwitchOnDisabled();
            doSetEncoderSSI();

            // Check if there is a bug in the controller
            int ssiPosition = readSSIPosition();
            if (Math.abs(ssiPosition) <= 1) {
                /* there is a bug in the controller in reading SSI position */
                /* switch back to Incremental Encoder */
                FCSLOG.info("Error after doSetEncoderSSI, ssiPosition = " + ssiPosition + ". We go back to TypeSinusIncrementalEncoder and retry ("+i+"/"+nbTryMax+")");
                doSetSinusIncrementalEncoder();
            } else {
                swapDone = true;
            }
            /* We need to check if the controller faulted after the swap, this is rare but it can happen.
             * If so, we try to fix the controller fault the first time we ecounter it.
             */
            updateEposState();
            boolean inFault = isInState(EposState.FAULT);
            if (inFault){
                // We are in fault, we must assume swap failed and try again later
                swapDone = false;
                // controler in Fault we should fix it first
                FCSLOG.info("StatusWord= " + readStatusWord() + ", FAULT= " + inFault);
                // Check if the controller already faulted. If so, something bad is going on, we stop !
                if (controlerFailed) {
                    this.raiseAlarm(FcsAlerts.IN_FAULT, "Controller faulted too many times while trying to change Carousel mode. Carousel's controller in fault, should be resetted manually.", getName());
                    checkFault(); // this will raise an Exception
                    return;
                }
                controlerFailed = true;

                /* We will try to recover the controller fault,
                 * by resetting it, but we allow some time around the reset to digest the error.
                 */
                updateEposState();

                int j = 0;
                while ((j++ < nbRecoverMax) && isInState(EposState.FAULT)) {
                    FcsUtils.sleep(200, "Sleep before fault reset");
                    // Do the reset manually and minimally (cannot publish or clear alert as we)
                    writeControlWord(ControlWord.DISABLE_VOLTAGE);
                    FcsUtils.sleep(10, "Sleep after controler disable voltage");
                    writeControlWord(ControlWord.FAULT_RESET);

                    FcsUtils.sleep(200, "Sleep after fault reset");
                    updateEposState();
                }
                // Check if the state at the end of the loop
                if (isInState(EposState.FAULT)) {
                    this.raiseAlarm(FcsAlerts.IN_FAULT,
                        "We failed to clear the controller's error " + nbRecoverMax + " times in a row for " + j*2*200 + "ms, while trying to change Carousel mode. Carousel's controller in fault, should be resetted manually.",
                         getName()
                    );
                    checkFault(); // this will raise an Exception
                } else {
                    // TODO: create a specific alert 'CONTROLLER_RECOVERY_SUCCESS' to distinguish more easily LSSTCCSFCS-694
                    this.raiseWarning(FcsAlerts.HARDWARE_ERROR, "We successfully cleared the controller's error in " + j + " tries in " + j*2*200 + "ms.",  getName());
                }
                // If we succeded to clear the error we will re-do the swap.
                // So lets go back to incremental first to be sure.
                doSetSinusIncrementalEncoder();
            }
        }
        // Check swap status at the end of the loop
        if (swapDone) {
            FCSLOG.info("We successfully switched the controller mode to absolute/SSI. It took " + i + " tries in" +  (beginTime - System.currentTimeMillis()) + "ms.");
            return;
        }
        // If we get out, it means we never succeded to switch despite many retries
        this.raiseAlarm(FcsAlerts.HARDWARE_ERROR, " after setPositionSensorTypeEncoderSSI, ssiPosition = " + readSSIPosition() + " we go back to TypeSinusIncrementalEncoder", getName());
        doSetSinusIncrementalEncoder();
    }

    /**
     * Generic function to switch position encoder mode
     * See doc EPOS2_Firmware_Specification_En.pdf pages 171-172
     * @param val value corresponding to the mode we want
     * @return
     */
    default void doSetEncoder(int val) {
        int controllerStructure = readControllerStructure();
        int valueToWrite = (controllerStructure == 0) ? val : val + 0x200;

        writeParameter(EposParameter.PositionSensorType, valueToWrite);
        /* Delay is to give time to the controller to limit error rate, see LSSTCCSFCS-667 */
        FcsUtils.sleep(80, "Switch encoder mode");
        /* check that position sensor type is specified encoder. */
        checkPositionSensorType(valueToWrite);
    }

    /**
     * Switch encoder mode to absolute/SSI
     * Value to send is 4 (0b010)
     */
    default void doSetEncoderSSI() {
        doSetEncoder(4);
    }

    /**
     * Set PositionSensorType to Sinus Incremental Encoder(value=8)
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "GoToSwitchOnDisabled "
            + "and set PositionSensorType to Sinus Incremental Encoder : 8")
    default void setPositionSensorTypeSinusIncrementalEncoder() {
        FCSLOG.info(getName() + " setting PositionSensorType Sinus Incremental Encoder");
        goToSwitchOnDisabled();
        doSetSinusIncrementalEncoder();
    }

    /**
     * Switch encoder mode to incremental
     * Value to send is 8 (0b100)
     */
    default void doSetSinusIncrementalEncoder() {
        // This could be doSetEncoder(8), but as requested by Pierre, we do not want to wait between write and check in this mode.
        int val = 8;
        int controllerStructure = readControllerStructure();
        int valueToWrite = (controllerStructure == 0) ? val : val + 0x200;

        writeParameter(EposParameter.PositionSensorType, valueToWrite);
        /* check that position sensor type is specified encoder. */
        checkPositionSensorType(valueToWrite);
    }

    default void checkPositionSensorType(int sensortype) {
        FcsUtils.checkAndWaitConditionWithTimeoutAndFixedDelay(
            () -> sensortype == readParameter(EposParameter.PositionSensorType),
            null,
            getName() + " check position sensory type is " + sensortype + " after switching mode ",
            getName() + ": couldn't change position sensor type during time allocated of 500ms",
            500,
            2
        );

        setPositionSensorType(sensortype);
    }

    /**
     *
     * @param position
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT, description = "Define the actual position as position given as argument.")
    @Override
    default void defineAbsolutePosition(int position) {
        FCSLOG.finer(() -> getName() + " Defining Absolute Position:" + position);
        this.changeMode(EposMode.HOMING);

        // TODO REVIEW CAROUSEL Added at SLAC july 21 for tests. Should be removed eventually or be incorporated to a safer method
        if (EposMode.HOMING != readMode()) {
            // second chance
            FcsUtils.sleep(500, "Controller change mode");
            this.changeMode(EposMode.HOMING);
        }
        checkEposMode(EposMode.HOMING);
        this.writeParameter(EposParameter.HomePosition, position);
        this.writeParameter(EposParameter.HomingMethod, 35);
        this.goToOperationEnable();
        /* start homing */
        this.writeControlWord(ControlWord.ABSOLUTE_POSITION);
        checkHomingDone();
    }

    /**
     * Activate POWER SAVE in order to permit shutter to be enabled.
     */
    default void activatePowerSave() {
        int val = (int) readParameter(EposParameter.DigitalOutputFonctionnalityState);
        writeParameter(EposParameter.DigitalOutputFonctionnalityState, FcsUtils.force2one(val, 13));
        publishData();
        FCSLOG.info(getName() + ": POWER SAVE activated.");
    }

    /**
     * Deactivate POWER SAVE to be able to rotate carousel.
     */
    default void deactivatePowerSave() {
        int val = (int) readParameter(EposParameter.DigitalOutputFonctionnalityState);
        writeParameter(EposParameter.DigitalOutputFonctionnalityState, FcsUtils.force2zero(val, 13));
        publishData();
        FCSLOG.info(getName() + ": POWER SAVE deactivate command executed.");
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Read bit holding brake into DigitalOutputFonctionnalityState. For carousel. index:2078 subindex:1 bit:2")
    default boolean readHoldingBrakes() {
        int digitalOutput = (int) readParameter(EposParameter.DigitalOutputFonctionnalityState);
        boolean holdingBrakes = ((digitalOutput >> 2) & 1) == 1;
        setHoldingBrakes(holdingBrakes);
        return holdingBrakes;
    }
}
