package org.lsst.ccs.subsystems.fcs;

import java.time.Duration;
import java.util.logging.Logger;

import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.aiousb.USB_DIO_96;
import org.lsst.ccs.drivers.aiousb.USB_DIO_96_Interface;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * This is the Main Module for the Loader control software when the loader is in
 * standalone.
 *
 *
 * @author virieux
 */
public class LoaderMain extends MainModule implements USB_DIO_96_Interface {
    private static final Logger FCSLOG = Logger.getLogger(LoaderMain.class.getName());

    @LookupField(strategy = LookupField.Strategy.CHILDREN, pathFilter = "loader")
    private Loader loader;

    @LookupField(strategy = LookupField.Strategy.CHILDREN, pathFilter = "autochanger")
    private FilterHolder autochanger;

    private USB_DIO_96_Interface usbdio96Card;

    private static final Duration LOAD_UNLOAD_FILTER_MAX_DURATION = Duration.ofSeconds(480);

    @Override
    public void init() {
        /*
         * define a role for my subsystem in order to make LoaderGUI listen to my
         * subsystem
         */
        subs.getAgentService(AgentPropertiesService.class).setAgentProperty("org.lsst.ccs.subsystem.fcs.loader", "loader");
        super.init();
    }

    /**
     * Update loader state in reading the sensors.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Update loader state in reading the sensors.")
    @Override
    public void updateStateWithSensors() {
        // update loader sensors is CANbus is connected
        if (loader.isCANbusConnected()) { // return boolean in memory.. but might be outdated !
            int i = 0;
            boolean done=false;
            while (i++<10 && !done) {
                try {
                    loader.updateStateWithSensors();
                    done = true;
                } catch (Exception e) {
                    FCSLOG.warning("Cannot contact CANbus. " + e.getMessage() + " Retrying "+i+"/10");
                    FcsUtils.sleep(20, name);
                }
            }
            if (!done) {
                FCSLOG.info("Loader does not respond. Connexion may have been cut physically. We disconnect CANbus.");
                disconnectLoaderStandaloneCANbus();
            }
        }
        autochanger.updateStateWithSensors();
    }


    @Override
    public void initializeHardware() {
        loader.initializeHardware();
    }

    /**
     * Disconnect the loader hardware.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "STANDALONE ONLY: Disconnect the loader hardware and unload configuration.")
    public void disconnectLoaderStandaloneCANbus() {
        loader.disconnectLoaderCANbus();
        loader.deInitializeHardware();

        /* after loader disconnection update state for GUI*/
        updateStateWithSensors();
    }

    /**
     * Connect the loader hardware.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "STANDALONE ONLY: Load the loader configuration and connect the hardware.")
    public void connectLoaderStandaloneCANbus(String loaderIdentifier) {
        loader.connectLoaderCANbus(loaderIdentifier);
        /* after loader connection update state for GUI*/
        updateStateWithSensors();
    }

    /**
     * In Standalone mode, nothing special to load.
     * We will connect to CANbus explicitly by choosing loader in GUI.
     */
    @Override
    public void postStart() {
        super.postStart();
    }

    /**
     * Print list of hardware with initialization information.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Print the list of hardware with initialization information.")
    public String printHardwareState() {
        StringBuilder sb = new StringBuilder(this.bridge.printHardwareState());
        sb.append("\n");
        sb.append(loader.printHardwareState());
        return sb.toString();
    }

    /**
     * Connect to the card USB_DIO_96 card.
     */
    public void connectUSB_DIO_96card() {
        if (usbdio96Card == null) {
            usbdio96Card = new USB_DIO_96();
            try {
                ((USB_DIO_96) usbdio96Card).connect();
            } catch (DriverException ex) {
                throw new FcsHardwareException("Could not connect to DIO_USB_96. "
                        + "Check that a DIO_USB_96 is present on this HCU and connected throw a USB cable.", ex);
            } catch (Exception ex) {
                throw new FcsHardwareException("Could not connect to DIO_USB_96. "
                        + " because : " + ex.toString());
            }
        }
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(name);
        sb.append("\n").append(loader.toString());
        return sb.toString();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE,
            description = "Print PLC loader signals read on card USB_DIO_96.")
    public String printUSBDIO96Signals() {
        connectUSB_DIO_96card();
        return usbdio96Card.toString();
    }

    /**
     * Simulate that there is no filter at HANDOFF on the test bench : AP2 & AF0
     *
     * Signals to simulate : NOPERMIT=0, ENG=0, AP2=1, AF0=1, AF1=1, AF3=0
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE,
            description = "Simulate PLC signals that say there is no filter at HANDOFF (AP2 & AF0 & NOLOCKOUT).")
    @Override
    public void simulateAutochangerEmpty() {
        connectUSB_DIO_96card();
        usbdio96Card.simulateAutochangerEmpty();
        FcsUtils.sleep(20, name);
        updateStateWithSensors();
    }

    /**
     * Simulate that a filter is at HANDOFF on the test bench and is held : AP2
     * & AF3
     *
     * Signals to simulate : NOPERMIT=0, ENG=0, AP2=1, AF0=0, AF1=0, AF3=1
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE,
            description = "Simulate PLC signals that say there is a filter at HANDOFF and it is held (AP2 & AF3 & NOLOCKOUT).")
    @Override
    public void simulateAutochangerHoldFilter() {
        connectUSB_DIO_96card();
        usbdio96Card.simulateAutochangerHoldFilter();
        FcsUtils.sleep(20, name);
        updateStateWithSensors();
    }

    /**
     * Load a filter into bench or storage box.
     *
     * This command simulates signals coming from autochanger.
     * To be used in standalone mode only.
     *
     * At the end of this command, the loader carrier is empty
     * at storage position.
     *
     */
    @SuppressWarnings("unchecked")
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL,
            description = "Load a filter into bench or storage box. Simulate signals coming from autochanger.",
            timeout = 480000, autoAck = false)
    public void loadFilterIntoBenchOrStorageBox() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("loadFilterIntoBenchOrStorageBox")) {
            updateStateWithSensors();
            subs.helper()
                .precondition(!agentStateService.isInState(AlertState.ALARM),
                        "can't execute commands in ALARM state.")
                .duration(LOAD_UNLOAD_FILTER_MAX_DURATION)
                .enterFaultOnException(true)
                .action(() -> {
                    simulateAutochangerEmpty();
                    if ( loader.getCarrier().isLooslyBetweenStorageAndEngaged() &&
                        ( !loader.isClampedOnFilter() || !loader.getClamp().hasClampEnoughForce() ) ) {
                        FCSLOG.info(name + " loader between STORAGE and ENGAGED with hooks not CLAMPED or with not enough force margins "
                            + "Starting recovery to get back to a CLAMPED state to continue the load operation. ");
                        // TODO add SAME counter as un FCSMain loadFilter method
                        loader.getClamp().recoveryUnclamp();
                        FcsUtils.sleep(50, name);
                        updateStateWithSensors();
                        loader.getClamp().recoveryClamp();
                        FcsUtils.sleep(50, name);
                        updateStateWithSensors();
                    }

                    if (loader.isClampedOnFilter() || loader.isClosedOnFilter()) {
                        loader.moveFilterToHandoff();
                    }
                    simulateAutochangerHoldFilter();
                    loader.openClampAndMoveEmptyToStorage();
                });
        }
    }

    /**
     * Unload a filter from bench or storage box.
     *
     * This command simulates signals coming from autochanger.
     * To be used in standalone mode only.
     *
     * At the end of this command, hooks are CLOSED on filter,
     * and carrier is at STORAGE position.
     *
     */
    @SuppressWarnings("unchecked")
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL,
            description = "Unload a filter from bench or storage box. Simulate signals coming from autochanger.",
            timeout = 480000, autoAck = false)
    public void unloadFilterFromBenchOrStorageBox() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("unloadFilterFromBenchOrStorageBox")) {
            updateStateWithSensors();
            subs.helper()
                .precondition(!agentStateService.isInState(AlertState.ALARM),
                        "can't execute commands in ALARM state.")
                .duration(LOAD_UNLOAD_FILTER_MAX_DURATION)
                .enterFaultOnException(true)
                .action(() -> {
                    simulateAutochangerHoldFilter();
                    if (loader.isEmpty()) {
                        loader.moveEmptyToHandoffAndClose();
                    }
                    simulateAutochangerEmpty();
                    loader.moveFilterToStorage();
                });
        }
    }

}
