
package org.lsst.ccs.subsystems.fcs;

import java.util.ArrayList;
import java.util.List;
import java.util.TreeMap;
import java.util.logging.Logger;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterFamily;

/**
 * This class's goal is to manage the filters.
 *
 * @author virieux
 */
public class FilterManager implements HasLifecycle {
    private static final Logger FCSLOG = Logger.getLogger(FilterManager.class.getName());
    @LookupName
    private String name;

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private final List<Filter> filters = new ArrayList<>();

    /**
     * A map to store the filters that FCS manages. The key of this map is the filter name.
     */
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    protected final TreeMap<String, Filter> filtersMapByName = new TreeMap<>();

    /**
     * A map to store the filters that FCS manages. The key of this map is the filter ID.
     */
    private final TreeMap<Integer, Filter> filtersMapByID = new TreeMap<>();

    /**
     * A map to store the filters that FCS manages. The key of this map is the filter ObservatoryName.
     */
    private final TreeMap<String, Filter> filtersMapByObservatoryName = new TreeMap<>();

    /**
     * A getter for the Filters Map sorted by filter name.
     */
    public TreeMap<String, Filter> getFiltersMapByName() {
        @SuppressWarnings("unchecked")
        TreeMap<String, Filter> filtersMap = (TreeMap<String, Filter>) filtersMapByName.clone();
        return filtersMap;
    }

    /**
     * A getter for the Filters Map sorted by filter ID
     */
    public TreeMap<Integer, Filter> getFiltersMapByID() {
        @SuppressWarnings("unchecked")
        TreeMap<Integer, Filter> filtersMap = (TreeMap<Integer, Filter>) filtersMapByID.clone();
        return filtersMap;
    }

    @Override
    public void init() {
        /**
         * Creation of the HashMap filtersMap. To move this initialization to the
         * constructor is a bad idea. It has been tested and it doesn't work because in
         * the construction phase the environment of the objects Filter is not set, so
         * the method getName() returns null.
        */
        for (Filter f : filters) {
            FCSLOG.info("Add filter to filterManager:" + f.getName());
            filtersMapByID.put(f.getFilterID(), f);
            filtersMapByObservatoryName.put(f.getObservatoryName(), f);
        }
    }

    public List<String> getFilterNames() {
        return filtersMapByName.keySet().stream().toList();
    }

    /**
     * Return a filter which name is given as argument.
     *
     * @param filterName
     * @return a filter instance
     */
    public Filter getFilterByName(String filterName) {
        if ( filtersMapByName.containsKey(filterName) ) {
            return filtersMapByName.get(filterName);
        } else if ( filtersMapByObservatoryName.containsKey(filterName) ) {
            return filtersMapByObservatoryName.get(filterName);
        } else {
            throw new IllegalArgumentException("The filter '" + filterName + "' does not exist");
        }
    }

    public int getFilterID(String filterName) {
        return getFilterByName(filterName).getFilterID();
    }

    /**
     * Return Return filter which id is given as argument.
     *
     * @param id
     * @return
     */
    public Filter getFilterByID(int id) {
        if ( containsFilterID(id) ) {
            return filtersMapByID.get(id);
        } else {
            FCSLOG.severe(name + " filterId=" + id + " is unknown in fcs configuration.");
            Filter unknownF = new Filter(id, 0, FilterFamily.U, 0);
            unknownF.setName("filterID_" + id);
            filtersMapByID.put(id, unknownF);
            return unknownF;
        }
    }

    /**
     * Return true if a filterID is in the map.
     *
     * @param id
     * @return true if filtersMapByID contains this id.
     */
    public boolean containsFilterID(int id) {
        return filtersMapByID.containsKey(id);
    }

    /**
     * return true if filterName is in FilterManager map.
     * @param filterName
     * @return
     */
    public boolean containsFilterName(String filterName) {
        return filtersMapByName.containsKey(filterName) || filtersMapByObservatoryName.containsKey(filterName);
    }

    /**
     * Return Return filter name which correspond to ID.
     *
     * @param id
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return filter name which corresponds to the filterID.")
    public String getFilterNameByID(int id) {
        if ( containsFilterID(id) ) {
            return filtersMapByID.get(id).getName();
        } else {
            FCSLOG.warning(name + ": " + id + "  unknown filter ID. Please update filterManager configuration.");
            return "UnknownFilterID_" + id;
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return filter observatory (OCS) name which corresponds to ID.")
    public String getObservatoryNameByID(int id) {
        if ( containsFilterID(id) ) {
            return filtersMapByID.get(id).getObservatoryName();
        } else {
            FCSLOG.warning(name + ": " + id + "  unknown filter ID. Please update filterManager configuration.");
            return "UnknownFilterID_" + id;
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return a printed list of filters with their properties.", alias = "printFilterList")
    @Override
    public String toString() {
        String fmt = "%-9s%-7s%-14s%-20s\n";
        StringBuilder sb = new StringBuilder();
        sb.append(String.format(fmt, "Name", "ID", "Weight (kg)", "Family"));
        sb.append(String.format(fmt, "----", "--", "-----------", "------"));
        filtersMapByName.forEach(
            (name, filter) -> {
            sb.append(String.format(fmt, name, filter.getFilterID(), filter.getWeight(), filter.getFamily().getFamilyName()));
        });
        return sb.toString();
    }
}
