/*
 * Decompiled with CFR 0.152.
 */
package org.lsst.ccs.subsystems.fcs;

import java.io.Serializable;
import java.time.Duration;
import java.util.EnumMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusStateChangeNotification;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.Persist;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.ConcurrentMessagingUtils;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.HasDataProviderInfos;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.ocsbridge.states.CameraMotionState;
import org.lsst.ccs.subsystems.fcs.Autochanger;
import org.lsst.ccs.subsystems.fcs.AutochangerThreeOnlineClamps;
import org.lsst.ccs.subsystems.fcs.AutochangerTwoTrucks;
import org.lsst.ccs.subsystems.fcs.Carousel;
import org.lsst.ccs.subsystems.fcs.CarouselSocket;
import org.lsst.ccs.subsystems.fcs.FcsActions;
import org.lsst.ccs.subsystems.fcs.FcsAlerts;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations;
import org.lsst.ccs.subsystems.fcs.Filter;
import org.lsst.ccs.subsystems.fcs.FilterManager;
import org.lsst.ccs.subsystems.fcs.Loader;
import org.lsst.ccs.subsystems.fcs.MainModule;
import org.lsst.ccs.subsystems.fcs.common.BridgeToLoader;
import org.lsst.ccs.subsystems.fcs.common.PersistentCounter;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.states.AutochangerInclinationState;
import org.lsst.ccs.subsystems.fcs.states.CarouselPowerState;
import org.lsst.ccs.subsystems.fcs.states.FcsState;
import org.lsst.ccs.subsystems.fcs.states.FilterReadinessState;
import org.lsst.ccs.subsystems.fcs.states.ObservatoryFilterState;
import org.lsst.ccs.subsystems.fcs.states.TmaRotatorState;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

public class FcsMain
extends MainModule
implements HasDataProviderInfos {
    private static final long serialVersionUID = 7669526660659959402L;
    private static final Logger FCSLOG = Logger.getLogger(FcsMain.class.getName());
    @LookupField(strategy=LookupField.Strategy.TREE)
    private BridgeToLoader bridgeToLoader;
    @LookupField(strategy=LookupField.Strategy.CHILDREN)
    private Carousel carousel;
    @LookupField(strategy=LookupField.Strategy.CHILDREN)
    private Autochanger autochanger;
    @LookupField(strategy=LookupField.Strategy.CHILDREN)
    private Loader loader;
    @LookupField(strategy=LookupField.Strategy.CHILDREN)
    private FilterManager filterManager;
    @LookupField(strategy=LookupField.Strategy.TREE)
    private AlertService alertService;
    private volatile TmaRotatorState tmaAndRotatorMotionStatus = TmaRotatorState.UNKNOWN;
    private final Predicate<AgentInfo> ocsBridgePredicate = ai -> ai.getAgentProperty("agentType").equals(AgentInfo.AgentType.OCS_BRIDGE.name());
    @ConfigurationParameter(range="0..200000", description="Expected max duration for a filter exchange at zenith", units="millisecond")
    public volatile int setFilterFastDuration = 95000;
    @ConfigurationParameter(range="0..200000", description="Expected max duration for a filter exchange at horizon", units="millisecond")
    public volatile int setFilterSlowDuration = 120000;
    @ConfigurationParameter(range="0..500000", description="Expected max duration of a filter swap with loader (load or unload)", units="millisecond")
    public volatile int loadUnloadFilterMaxDuration = 480000;
    @Persist
    private volatile int previousSocketID = -1;
    private boolean firstMcmPublication = true;
    private Map<FcsActions.GeneralAction, PersistentCounter> movementCounter;
    private AtomicLong lastUpdateStateWithSensors = new AtomicLong(0L);
    private final StatusMessageListener cameraMotionStatusMessageListener = new StatusMessageListener(){
        private final AtomicBoolean isFirstStatusStateChangeNotification = new AtomicBoolean(true);

        public void onStatusMessage(StatusMessage msg) {
            CameraMotionState cameraMotionState;
            StateBundle stateToProcess = null;
            if (this.isFirstStatusStateChangeNotification.getAndSet(false)) {
                stateToProcess = msg.getState();
            } else if (msg instanceof StatusStateChangeNotification) {
                StatusStateChangeNotification stateChange = (StatusStateChangeNotification)msg;
                stateToProcess = stateChange.getNewState().diffState(stateChange.getOldState());
            }
            if (stateToProcess != null && (cameraMotionState = (CameraMotionState)stateToProcess.getState(CameraMotionState.class)) != null) {
                FCSLOG.finer("Camera Motion State received from OCS-Bridge = " + cameraMotionState.name());
                switch (cameraMotionState) {
                    case LOCKED: {
                        FcsMain.this.updateTmaAndRotatorStatus(TmaRotatorState.LOCKED);
                        break;
                    }
                    case UNLOCKED: {
                        FcsMain.this.updateTmaAndRotatorStatus(TmaRotatorState.UNLOCKED);
                        break;
                    }
                    default: {
                        FCSLOG.warning(FcsMain.this.name + "Unknown Camera Motion (software lock) state: " + cameraMotionState.name());
                        FcsMain.this.updateTmaAndRotatorStatus(TmaRotatorState.UNKNOWN);
                    }
                }
            }
        }
    };

    @Override
    public void build() {
        if (this.bridgeToLoader == null) {
            throw new RuntimeException("The FcsMain expected a bridge to the Loader, but none was found.");
        }
        if (this.bridgeToLoader.equals(this.bridge)) {
            throw new RuntimeException("In FcsMain both bridges are the same instance. We expected two different bridges.");
        }
        super.build();
        KeyValueData data = this.getDataForMcm();
        this.dataProviderDictionaryService.registerData(data);
        String mainPath = data.getKey();
        KeyValueDataList dataList = (KeyValueDataList)data.getValue();
        for (KeyValueData d : dataList) {
            String path = mainPath + "/" + d.getKey();
            DataProviderInfo info = this.dataProviderDictionaryService.getDataProviderDictionary().getDataProviderInfoForPath(path);
            String units = "unitless";
            String description = "";
            switch (d.getKey()) {
                case "filter_on_autochanger": {
                    description = "OCS name of the filter currently on the autochanger";
                    break;
                }
                case "filter_previous_socketID": {
                    description = "ID of the socket on which the current filter used to sit";
                    break;
                }
                case "autochanger_trucks_position": {
                    description = "Absolute position of the autochanger trucks";
                    units = "micron";
                    break;
                }
                case "autochanger_trucks_state": {
                    description = "Known autochanger position (STANDBY, HANDOFF, ONLINE) or IN_TRAVEL";
                    break;
                }
                case "proximity": {
                    description = "Value of the autochanger proximity sensor";
                    break;
                }
                default: {
                    units = "unitless";
                    description = "";
                }
            }
            info.addAttribute(DataProviderInfo.Attribute.UNITS, units);
            info.addAttribute(DataProviderInfo.Attribute.DESCRIPTION, description);
        }
        this.agentStateService.updateAgentState(new Enum[]{ObservatoryFilterState.UNKNOWN});
    }

    @Override
    public void init() {
        super.init();
        ((AgentPropertiesService)this.subs.getAgentService(AgentPropertiesService.class)).setAgentProperty("org.lsst.ccs.subsystem.fcs.wholefcs", "fcs");
        this.movementCounter = new EnumMap<FcsActions.GeneralAction, PersistentCounter>(FcsActions.GeneralAction.class);
        for (FcsActions.GeneralAction action : new FcsActions.GeneralAction[]{FcsActions.GeneralAction.SET_FILTER, FcsActions.GeneralAction.SET_FILTER_AT_HANDOFF_FOR_LOADER, FcsActions.GeneralAction.STORE_FILTER_ON_CAROUSEL, FcsActions.GeneralAction.SET_NO_FILTER, FcsActions.GeneralAction.DISENGAGE_FILTER_FROM_CAROUSEL, FcsActions.GeneralAction.LOAD_FILTER, FcsActions.GeneralAction.UNLOAD_FILTER, FcsActions.GeneralAction.CONNECT_LOADER, FcsActions.GeneralAction.DISCONNECT_LOADER}) {
            this.movementCounter.put(action, PersistentCounter.newCounter(action.getCounterPath(), this.subs, action.name()));
            action.registerDurationTopLevel(this.dataProviderDictionaryService);
        }
        ClearAlertHandler alwaysClear = new ClearAlertHandler(){

            public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
            }
        };
        this.alertService.registerAlert(FcsAlerts.HARDWARE_ERROR.getAlert(this.name), alwaysClear);
        this.alertService.registerAlert(FcsAlerts.CA_LOCKING_ISSUE.getAlert(this.name), alwaysClear);
        this.alertService.registerAlert(FcsAlerts.HARDWARE_ERROR.getAlert("carousel"), alwaysClear);
    }

    private void updateTmaAndRotatorStatus(TmaRotatorState tmaRotatorState) {
        TmaRotatorState currentMotionStatus = TmaRotatorState.valueOf(this.tmaAndRotatorMotionStatus.name());
        this.tmaAndRotatorMotionStatus = tmaRotatorState;
        if (!currentMotionStatus.equals((Object)this.tmaAndRotatorMotionStatus)) {
            FCSLOG.info(this.name + String.format(" Registering a Camera Motion State (software lock) change from %s to %s", currentMotionStatus.name(), this.tmaAndRotatorMotionStatus.name()));
        }
    }

    public void postInit() {
        super.postInit();
        this.getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(new AgentPresenceListener(){

            public void connecting(AgentInfo ... agents) {
                for (AgentInfo agent : agents) {
                    if (!FcsMain.this.ocsBridgePredicate.test(agent)) continue;
                    FcsMain.this.getMessagingAccess().addStatusMessageListener(FcsMain.this.cameraMotionStatusMessageListener, m -> FcsMain.this.ocsBridgePredicate.test(m.getOriginAgentInfo()));
                    break;
                }
            }

            public void disconnecting(AgentInfo agent) {
                if (FcsMain.this.ocsBridgePredicate.test(agent)) {
                    FcsMain.this.getMessagingAccess().removeStatusMessageListener(FcsMain.this.cameraMotionStatusMessageListener);
                    FcsMain.this.updateTmaAndRotatorStatus(TmaRotatorState.UNKNOWN);
                }
            }
        });
    }

    public void postStart() {
        AgentInfo agent;
        super.postStart();
        Iterator iterator = this.getMessagingAccess().getAgentPresenceManager().listConnectedAgents().iterator();
        if (iterator.hasNext() && this.ocsBridgePredicate.test(agent = (AgentInfo)iterator.next())) {
            ConcurrentMessagingUtils messagingUtils = new ConcurrentMessagingUtils(this.getMessagingAccess());
            CommandRequest cmd = new CommandRequest(agent.getName(), "publishState");
            FCSLOG.info("Requesting the OCS-Bridge to publish its states");
            messagingUtils.sendAsynchronousCommand(cmd);
        }
        if (FcsUtils.isProto()) {
            this.updateTmaAndRotatorStatus(TmaRotatorState.PROTOTYPE);
        }
    }

    public void shutdown() {
        if (this.isPowerSaveActivated()) {
            this.carousel.powerOn();
        }
        this.getMessagingAccess().removeStatusMessageListener(this.cameraMotionStatusMessageListener);
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="Print the current software lock status on the telescope and rotator motion")
    public String getTelescopeSoftwareLockStatus() {
        return this.tmaAndRotatorMotionStatus.name();
    }

    @Override
    public void updateFCSStateToReady() {
        if (this.carousel.isInitialized() && this.autochanger.isInitialized() && this.loader.isInitialized()) {
            this.updateAgentState(FcsState.READY);
            this.updateAgentState(FilterReadinessState.READY);
        }
    }

    @Command(type=Command.CommandType.ACTION, level=3, description="Sets the filters state to READY even if some sensors are still missing in autochanger.")
    public void forceFilterReadinessStateToReady() {
        this.updateAgentState(FcsState.READY);
        this.updateAgentState(FilterReadinessState.READY);
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="Return true if the changer is connected.")
    public boolean isChangerConnected() {
        return this.bridge.isReady();
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="Return true if the loader is connected.")
    public boolean isLoaderConnected() {
        return this.autochanger.isLoaderConnected();
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="Return true if the hardware of the changer is ready.")
    public boolean isChangerReady() {
        return this.bridge.allDevicesBooted();
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="Return true if the hardware of the loader is ready.")
    public boolean isLoaderReady() {
        return this.bridgeToLoader.allDevicesBooted();
    }

    @Command(type=Command.CommandType.ACTION, level=0, description="Disconnect the loader hardware.")
    public void disconnectLoaderCANbus() {
        this.movementCounter.get(FcsActions.GeneralAction.DISCONNECT_LOADER).increment();
        long beginTime = System.currentTimeMillis();
        this.loader.disconnectLoaderCANbus();
        FcsActions.GeneralAction.DISCONNECT_LOADER.publishDurationTopLevel(this.subs, System.currentTimeMillis() - beginTime);
        this.updateStateWithSensors();
    }

    @Command(type=Command.CommandType.ACTION, level=0, description="Connect the loader hardware.")
    public void connectLoaderCANbus() {
        this.movementCounter.get(FcsActions.GeneralAction.CONNECT_LOADER).increment();
        long beginTime = System.currentTimeMillis();
        this.loader.connectLoaderCANbus();
        FcsActions.GeneralAction.CONNECT_LOADER.publishDurationTopLevel(this.subs, System.currentTimeMillis() - beginTime);
        this.updateStateWithSensors();
    }

    @Override
    @Command(type=Command.CommandType.QUERY, level=0, description="Return the list of LOADER CANopen hardware that this subsystem manages.")
    public List<String> listLoaderHardwareNames() {
        return this.bridgeToLoader.listHardwareNames();
    }

    @Override
    @Command(type=Command.CommandType.QUERY, level=0, description="Return the list of names of sensors plugged on Loader pluto gateway.")
    public List<String> listLoSensorsNames() {
        return this.bridgeToLoader.listLoSensorsNames();
    }

    public boolean isFilterInCamera(int filterID) {
        return this.carousel.isFilterOnCarousel(filterID) || this.autochanger.isFilterOnTrucks(filterID);
    }

    public boolean isFilterAvailable(int filterID) {
        return this.getAvailableFilterMap().entrySet().stream().anyMatch(e -> (Integer)e.getKey() == filterID);
    }

    @Command(type=Command.CommandType.ACTION, level=1, description="Store the filter on a given socket of the carousel and move the empty autochanger to HANDOFF position.", autoAck=false, timeout=60000)
    public void storeFilterOnSocket(String socketName) {
        this.storeFilterOnSocket(socketName, true);
    }

    private void storeFilterOnSocket(String socketName, boolean withPrecision) {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("storeFilterOnSocket");){
            this.updateStateWithSensors();
            FCSLOG.info(this.name + " === About to store filter on Carousel and go empty to HANDOFF " + (withPrecision ? "for loader " : "") + "===");
            String debugMsg = "\n" + this.name + " This command is used to store on Carousel the filter currently on the Autochanger.\n";
            this.movementCounter.get(FcsActions.GeneralAction.STORE_FILTER_ON_CAROUSEL).increment();
            this.carousel.getSocketAtStandby().storeFilterOnCarouselCounter.increment();
            long beginTime = System.currentTimeMillis();
            if (!FilterReadinessState.READY.equals((Object)this.getFilterReadinessState())) {
                throw new RejectedCommandException(debugMsg + "FilterReadinessState must be READY");
            }
            if (!this.autochanger.isAvailable()) {
                throw new RejectedCommandException(debugMsg + "Autochanger is currently unavailable");
            }
            if (!this.autochanger.isHoldingFilter()) {
                throw new RejectedCommandException(debugMsg + "Autochanger is not currently holding a filter");
            }
            if (this.autochanger.isAtStandby() && this.autochanger.isCarouselHoldingFilterAtStandby()) {
                throw new RejectedCommandException(debugMsg + "Autochanger is at STANDBY and its filter is already clamped on the Carousel. Please do things manually.");
            }
            CarouselSocket desiredSocket = this.carousel.getSocketByName(socketName);
            if (!desiredSocket.isAvailable()) {
                throw new RejectedCommandException(debugMsg + socketName + " is unavailable");
            }
            if (!desiredSocket.isEmpty()) {
                throw new RejectedCommandException(debugMsg + socketName + " should be empty");
            }
            if (!desiredSocket.isAtStandby() && !this.carousel.isAvailable()) {
                throw new RejectedCommandException(debugMsg + "The desired socket is not at STANDBY and the Carousel rotation is currently unavailable");
            }
            FCSLOG.info(this.name + " === All preconditions passed, starting the movements ===");
            this.agentStateService.updateAgentState(new Enum[]{ObservatoryFilterState.UNLOADING});
            if (!this.autochanger.isAtHandoff() && !this.autochanger.isAtOnline()) {
                FCSLOG.info(this.name + " === Moving Autochanger to HANDOFF to enable further movements ===");
                this.autochanger.goToHandOff();
            }
            this.carousel.rotateSocketToStandby(socketName);
            this.carousel.checkDeltaPosition();
            this.updateStateWithSensors();
            FCSLOG.info(this.name + " === Carousel is ready to receive a filter at STANDBY ===");
            if (this.autochanger.isAtOnline()) {
                AutochangerThreeOnlineClamps onlineClamps = this.autochanger.getOnlineClamps();
                onlineClamps.unlockAndOpen();
                this.updateStateWithSensors();
            }
            this.autochanger.waitForProtectionSystemUpdate();
            this.autochanger.moveFilterToStandbyPlusDelta(this.carousel.getSocketAtStandby().getDeltaAutochangerStandbyPosition());
            this.updateStateWithSensors();
            if (!this.carousel.isHoldingFilterAtStandby()) {
                this.recoveryLockingProcess();
                if (!this.carousel.isHoldingFilterAtStandby()) {
                    String msg = this.name + ": Carousel should be LOCKED_ON_FILTER when Autochanger is at STANDBY with a filter; recovery process didn't work";
                    this.raiseAlarm(FcsAlerts.CA_LOCKING_ISSUE, msg, this.name);
                    throw new FcsHardwareException(msg);
                }
            }
            this.carousel.getSocketAtStandby().updateFilterID();
            FCSLOG.info("filter " + this.filterManager.getObservatoryNameByID(this.carousel.getSocketAtStandby().getFilterID()) + " is now locked in the carousel");
            FCSLOG.info(this.name + ": is going to moveEmptyFromStandbyToHandoff");
            this.autochanger.waitForProtectionSystemUpdate();
            this.autochanger.moveEmptyFromStandbyToHandoff(withPrecision);
            this.agentStateService.updateAgentState(new Enum[]{ObservatoryFilterState.UNLOADED});
            this.previousSocketID = -1;
            this.publishDataForMcm();
            FcsActions.GeneralAction.STORE_FILTER_ON_CAROUSEL.publishDurationTopLevel(this.subs, System.currentTimeMillis() - beginTime);
            FcsActions.GeneralAction.STORE_FILTER_ON_CAROUSEL.publishDurationPerElement(this.carousel.getSocketAtStandby().getSubsystem(), System.currentTimeMillis() - beginTime, this.carousel.getSocketAtStandby().path);
        }
    }

    private void storeFilterOnCarouselRelaxed() {
        this.storeFilterOnCarousel(false);
    }

    @Command(type=Command.CommandType.ACTION, level=1, description="Store the filter in the carousel and move the empty autochanger to HANDOFF position for loader. Initial State for autochanger: a filter at HANDOFF or ONLINE. Final State for autochanger : empty at HANDOFF for loader.", autoAck=false, timeout=60000)
    public void storeFilterOnCarousel() {
        this.storeFilterOnCarousel(true);
    }

    private void storeFilterOnCarousel(boolean withPrecision) {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("storeFilterOnCarousel");){
            CarouselSocket desiredSocket = this.carousel.getSocketAtStandby();
            if (desiredSocket == null) {
                String msg = this.name + ": there is no socket at STANDBY. Please rotate Carousel to the desired socket or use 'storeFilterOnSocket()'";
                throw new FcsHardwareException(msg);
            }
            this.storeFilterOnSocket(desiredSocket.getName(), withPrecision);
        }
    }

    private void recoveryLockingProcess() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("recoveryLockingProcess");){
            if (this.carousel.getClampXplus().isLocked() && !this.carousel.getClampXminus().isLocked()) {
                FcsUtils.sleep(200, this.name);
                this.carousel.recoveryLockingXminus();
            }
        }
    }

    @Command(type=Command.CommandType.ACTION, level=1, autoAck=false, description="Unclamp filter from carousel, move autochanger to handoff position and release the carousel clamps", timeout=15000)
    public void disengageFilterFromCarousel() {
        this.disengageFilterFromCarousel(false, false);
    }

    private void disengageFilterFromCarousel(boolean toOnline, boolean doClampOnline) {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("disengageFilterFromCarousel");){
            long beginTime = System.currentTimeMillis();
            this.movementCounter.get(FcsActions.GeneralAction.DISENGAGE_FILTER_FROM_CAROUSEL).increment();
            this.updateStateWithSensors();
            if (!FilterReadinessState.READY.equals((Object)this.getFilterReadinessState())) {
                throw new RejectedCommandException(this.name + " FilterReadinessState must be READY to start movement, currently = " + this.getFilterReadinessState());
            }
            if (!this.carousel.isAtStandby()) {
                throw new RejectedCommandException(this.name + " Carousel should have a socket at STANDBY before issuing this command");
            }
            if (!this.autochanger.isAtStandby()) {
                throw new RejectedCommandException(this.name + " Autochanger should be at STANDBY position before issuing this command");
            }
            if (!this.autochanger.isAvailable()) {
                throw new RejectedCommandException(this.name + " Autochanger is not available, canceling this command");
            }
            this.carousel.unlockClamps();
            this.carousel.updateStateWithSensors();
            this.carousel.waitForProtectionSystemUpdate();
            if (!this.carousel.isUnclampedOnFilterAtStandby()) {
                String msg = String.format("%s carousel clamps still locked. Aborting autochanger movement because carousel is still holding the filter.", this.name);
                String debug = String.format("\nsocketAtStandby should be UNLOCKED_ON_FILTER but current state is %s", this.carousel.getClampsStateAtStandby());
                this.raiseAlarm(FcsAlerts.HARDWARE_ERROR, msg + debug, this.name);
                throw new FcsHardwareException(msg + debug);
            }
            this.autochanger.waitForProtectionSystemUpdate();
            this.previousSocketID = this.carousel.getSocketAtStandbyID();
            try (AutochangerTwoTrucks.ReleaseAutochangerBrakes g = new AutochangerTwoTrucks.ReleaseAutochangerBrakes(this.autochanger.getAutochangerTrucks());){
                this.autochanger.getAutochangerTrucks().moveToApproachStandbyPositionWithLowVelocity();
                this.publishDataForMcm();
                this.updateStateWithSensors();
                if (!this.carousel.isEmptyAtStandby()) {
                    String msg = String.format("Something went wrong when disengaging filter %s from carousel socket%d, aborting autochanger movement.", this.filterManager.getObservatoryNameByID(this.autochanger.getFilterID()), this.previousSocketID);
                    Object help = "\nThe autochanger was sent to approachStandby position after unlocking the carousel clamps ";
                    help = (String)help + "but the carousel clamps are still sensing the presence of the filter.";
                    help = (String)help + "\nReach out immediately to a FES expert to assess the current status of the Filter Exchange System.";
                    this.raiseAlarm(FcsAlerts.HARDWARE_ERROR, msg + (String)help, "carousel");
                    throw new FcsHardwareException(msg);
                }
                FcsUtils.asyncRun(() -> {
                    try {
                        this.carousel.releaseClamps();
                    }
                    catch (Exception ex) {
                        String msg = "A problem has appeared during carousel clamp release : " + ex.getMessage();
                        String debug = String.format("\nCurrent clamp status is %s, X- is %s, X+ is %s", this.carousel.getSocketAtStandby().getClampsState(), this.carousel.getClampXminus().getClampState(), this.carousel.getClampXplus().getClampState());
                        this.raiseWarning(FcsAlerts.HARDWARE_ERROR, msg + debug, "carousel");
                    }
                });
                if (toOnline) {
                    if (doClampOnline) {
                        this.autochanger.moveAndClampFilterOnline();
                    } else {
                        this.autochanger.goToOnline();
                    }
                } else {
                    this.autochanger.moveFilterToHandoff();
                }
            }
            this.updateStateWithSensors();
            FcsActions.GeneralAction.DISENGAGE_FILTER_FROM_CAROUSEL.publishDurationTopLevel(this.subs, System.currentTimeMillis() - beginTime);
        }
    }

    public boolean autochangerNotInTravel() {
        return this.autochanger.isAtHandoff() || this.autochanger.isAtOnline() || this.autochanger.isAtStandby();
    }

    public boolean latchesOpenOrClosed() {
        return this.autochanger.getLatches().isClosed() || this.autochanger.getLatches().isOpened();
    }

    public boolean onlineClampsOpenOrLocked() {
        return this.autochanger.getOnlineClamps().isLocked() || this.autochanger.getOnlineClamps().isOpened();
    }

    public boolean filterAtOnlineMustBeLocked() {
        return !this.autochanger.isAtOnline() || this.autochanger.isEmpty() || this.autochanger.getOnlineClamps().isLocked();
    }

    public boolean filterAtStandbyMustBeHeld() {
        return !this.autochanger.isAtStandby() || this.autochanger.isEmpty() || this.autochanger.isHoldingFilter() || this.carousel.isHoldingFilterAtStandby();
    }

    public boolean carouselHoldingFilterOrReadyToGrab() {
        return this.carousel.isHoldingFilterAtStandby() || this.carousel.isReadyToGrabAFilterAtStandby();
    }

    public boolean carouselReadyToClampAtStandby() {
        return this.autochanger.isAtStandby() || this.autochanger.isEmpty() || this.carousel.isReadyToGrabAFilterAtStandby();
    }

    public boolean areTmaAndRotatorLocked() {
        return this.tmaAndRotatorMotionStatus.equals((Object)TmaRotatorState.LOCKED) || this.tmaAndRotatorMotionStatus.equals((Object)TmaRotatorState.PROTOTYPE) || this.isLoaderConnected() || FcsUtils.isSimu();
    }

    public boolean trucksMustBeStraight() {
        return this.autochanger.getAutochangerInclinationState().equals((Object)AutochangerInclinationState.STRAIGHT);
    }

    @Command(type=Command.CommandType.ACTION, level=0, description="Move filter to ONLINE position.", timeout=180000, autoAck=false)
    public void setFilter(int filterID) {
        boolean doClampOnline;
        boolean bl = doClampOnline = !"autochanger-PROTO".equals(this.autochanger.getIdentifier());
        if (!this.filterManager.containsFilterID(filterID)) {
            throw new RejectedCommandException("Unknown filter ID: " + filterID);
        }
        if (filterID == 0) {
            this.setNoFilterAtHandoffOrOnline(true);
        } else {
            this.setFilterAtHandoffOrOnline(filterID, true, doClampOnline);
        }
    }

    @Command(type=Command.CommandType.ACTION, level=0, description="Select filter and move to HANDOFF position without going to ONLINE.", timeout=180000, autoAck=false)
    public void setFilterAtHandoff(int filterID) {
        if (!this.filterManager.containsFilterID(filterID)) {
            throw new RejectedCommandException("Unknown filter ID: " + filterID);
        }
        if (filterID == 0) {
            this.setNoFilterAtHandoffOrOnline(false);
        } else {
            this.setFilterAtHandoffOrOnline(filterID, false);
        }
    }

    @Command(type=Command.CommandType.ACTION, level=0, description="Store all filters on carousel and move autochanger trucks to ONLINE.", timeout=180000, autoAck=false)
    public void setNoFilter() {
        this.setFilter(0);
    }

    @Command(type=Command.CommandType.ACTION, level=0, description="Store all filters on carousel and move autochanger trucks to HANDOFF with loader precision.", timeout=180000, autoAck=false)
    public void setNoFilterAtHandoff() {
        this.setFilterAtHandoff(0);
    }

    private boolean isRequestedFilterAlreadyInPosition(int filterID, boolean toOnline) {
        if (filterID == 0) {
            if (toOnline) {
                return this.autochanger.isEmptyOnline();
            }
            return this.autochanger.isEmpty() && this.autochanger.isAtHandoffForLoader();
        }
        if (toOnline) {
            return this.autochanger.isFilterClampedOnline(filterID);
        }
        return this.autochanger.isFilterOnTrucks(filterID) && this.autochanger.isAtHandoffForLoader();
    }

    private void setFilterAtHandoffOrOnline(int filterID, boolean toOnline, boolean doClampOnline) {
        this.updateStateWithSensors();
        if (filterID == 0) {
            throw new RejectedCommandException(this.name + " This method is invalid for the No Filter case. Should use setNoFilterAtHandoffOrOnline instead");
        }
        String filterObsName = this.filterManager.getObservatoryNameByID(filterID);
        if (this.isRequestedFilterAlreadyInPosition(filterID, toOnline)) {
            FCSLOG.info(this.name + " requested filter " + filterObsName + " already in position");
            return;
        }
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("setFilter");){
            if (this.isPowerSaveActivated()) {
                this.wakeFilterChanger(FcsEnumerations.CarouselPowerMode.WAKE_UP);
                this.autochanger.waitForProtectionSystemUpdate();
            }
            this.publishDataForMcm();
            this.checkControllers();
            FCSLOG.info(this.name + " filter requested online: " + filterObsName);
            this.subs.helper().precondition(!this.agentStateService.isInState((Enum)AlertState.ALARM), "Can't execute commands in ALARM state.", new Supplier[0]).precondition(this.areTmaAndRotatorLocked(), "The telescope motion and rotator need to be locked before performing a filter exchange. Current status of the lock is %s", new Object[]{this.getTelescopeSoftwareLockStatus()}).precondition(this.trucksMustBeStraight(), "The FCS does not allow a filter exchange to be executed because the camera is currently rotated. Ensure the rotator is set back to 0 before trying again.", new Supplier[0]).precondition(this.filterManager.containsFilterID(filterID), "Unknown filter id : %s", new Object[]{filterID}).precondition(this.isFilterInCamera(filterID), "The filter %s is currently out of the camera", new Object[]{filterObsName}).precondition(this.isFilterAvailable(filterID), "The filter %s is currently not available", new Object[]{filterObsName}).precondition(this.carousel.isAtStandby(), "The carousel has no socket currently at STANDBY position", new Supplier[0]).precondition(this.autochangerNotInTravel(), "The autochanger trucks are not at STANDBY, HANDOFF or ONLINE position", new Supplier[0]).precondition(this.autochanger.isAvailable(), "The autochanger is not available", new Supplier[0]).precondition(this.latchesOpenOrClosed(), "The autochanger latches should either be OPENED or CLOSED to proceed. Current state: %s", new Object[]{this.autochanger.getLatches().getLockStatus()}).precondition(this.onlineClampsOpenOrLocked(), "The autochanger ONLINE clamps should either be OPENED or LOCKED during normal operations. Current state: %s", new Object[]{this.autochanger.getOnlineClamps().getLockStatus()}).precondition(this.filterAtStandbyMustBeHeld(), "When a filter is at STANDBY it must be held by the autochanger or by the carousel.", new Supplier[0]).precondition(this.autochanger.isNotHoldingFilter() || this.carouselHoldingFilterOrReadyToGrab(), "The carousel socket at STANDBY should be holding a filter or ready to receive a filter. Current state: %s", new Object[]{this.carousel.getClampsStateAtStandby()}).precondition(this.autochanger.isNotHoldingFilter() || this.carouselReadyToClampAtStandby(), "The carousel state for the socket at STANDBY should be READYTOCLAMP when a filter is on the autochanger at HANDOFF or ONLINE. Current state: %s", new Object[]{this.carousel.getClampsStateAtStandby()}).precondition(new Enum[]{FilterReadinessState.READY}).duration(Duration.ofMillis(this.setFilterSlowDuration)).enterFaultOnException(true).action(() -> {
                FcsActions.GeneralAction action = toOnline ? FcsActions.GeneralAction.SET_FILTER : FcsActions.GeneralAction.SET_FILTER_AT_HANDOFF_FOR_LOADER;
                this.movementCounter.get(action).increment();
                long beginTime = System.currentTimeMillis();
                if (this.autochanger.isHoldingFilter()) {
                    FCSLOG.info(this.name + ": autochanger is currently holding filter " + this.autochanger.getFilterOnTrucksObservatoryName());
                } else {
                    FCSLOG.info(this.name + ": autochanger is empty, filter " + this.filterManager.getObservatoryNameByID(filterID) + " is on carousel socket " + this.carousel.getFilterSocket(filterID).getId());
                }
                this.carousel.setControllerPositionSensorTypeEncoderSSI();
                if (!this.autochanger.isFilterOnTrucks(filterID)) {
                    if (this.autochanger.isAtStandby()) {
                        FCSLOG.info(this.name + " autochanger is at STANDBY, it has to be moved empty to HANDOFF");
                        this.autochanger.moveEmptyFromStandbyToHandoff();
                    } else if (this.autochanger.isHoldingFilter()) {
                        FCSLOG.info(this.name + ": is going to store filter " + this.autochanger.getFilterOnTrucksObservatoryName() + " on carousel");
                        this.storeFilterOnCarouselRelaxed();
                    }
                    this.publishDataForMcm();
                    if (!this.autochanger.isEmpty() || !this.autochanger.isAtHandoff() && !this.autochanger.isAtOnline()) {
                        throw new FcsHardwareException(this.name + ": autochanger should be empty at HANDOFF or ONLINE");
                    }
                    this.carousel.rotateSocketToStandby(this.carousel.getFilterSocket(filterID).getName());
                    this.publishDataForMcm();
                    FcsUtils.sleep(100, this.name);
                    this.updateStateWithSensors();
                    if (!this.carousel.isAtStandby()) {
                        this.agentStateService.updateAgentState(new Enum[]{ObservatoryFilterState.UNLOADED});
                        throw new FcsHardwareException(this.name + ": carousel should be at STANDBY after the rotateSocketToStandby command.");
                    }
                    if (this.carousel.getFilterAtStandbyID() == filterID) {
                        this.carousel.checkDeltaPosition();
                        this.autochanger.waitForProtectionSystemUpdate();
                        if (toOnline) {
                            this.agentStateService.updateAgentState(new Enum[]{ObservatoryFilterState.LOADING});
                        }
                    } else {
                        throw new FcsHardwareException(this.name + " filter in carousel at STANDBY is " + this.filterManager.getObservatoryNameByID(this.carousel.getFilterAtStandbyID()) + " but it should be " + filterObsName);
                    }
                    this.autochanger.grabFilterAtStandby();
                    this.updateStateWithSensors();
                    this.previousSocketID = this.carousel.getSocketAtStandbyID();
                    this.publishDataForMcm();
                    this.updateStateWithSensors();
                    if (!this.autochanger.isFilterOnTrucks(filterID)) {
                        throw new FcsHardwareException(this.name + " filter " + filterObsName + " should now be on the autochanger");
                    }
                }
                if (this.autochanger.isAtStandby()) {
                    this.disengageFilterFromCarousel(toOnline, doClampOnline);
                } else if (toOnline) {
                    if (doClampOnline) {
                        if (this.autochanger.isAtHandoff()) {
                            this.autochanger.moveAndClampFilterOnline();
                        } else if (this.autochanger.isAtOnline()) {
                            this.autochanger.lockFilterAtOnline();
                        }
                    } else {
                        this.autochanger.goToOnline();
                    }
                    this.agentStateService.updateAgentState(new Enum[]{ObservatoryFilterState.LOADED});
                } else {
                    this.autochanger.moveFilterToHandoff();
                    this.agentStateService.updateAgentState(new Enum[]{ObservatoryFilterState.UNLOADED});
                }
                FCSLOG.info(this.name + " the filter " + filterObsName + " is now at " + (toOnline ? "ONLINE" : "HANDOFF"));
                this.updateAgentState(FcsState.valueOf("ONLINE_" + this.filterManager.getFilterNameByID(filterID).toUpperCase()));
                this.publishData();
                if (toOnline && this.isPowerSaveWanted()) {
                    this.wakeFilterChanger(FcsEnumerations.CarouselPowerMode.GO_TO_SLEEP);
                }
                action.publishDurationTopLevel(this.subs, System.currentTimeMillis() - beginTime);
                this.publishData();
                this.publishDataForMcm();
            });
        }
    }

    private void setFilterAtHandoffOrOnline(int filterID, boolean toOnline) {
        this.setFilterAtHandoffOrOnline(filterID, toOnline, true);
    }

    private void setNoFilterAtHandoffOrOnline(boolean toOnline) {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("setNoFilter");){
            this.updateStateWithSensors();
            if (this.isPowerSaveActivated()) {
                this.wakeFilterChanger(FcsEnumerations.CarouselPowerMode.WAKE_UP);
            }
            this.publishDataForMcm();
            this.checkControllers();
            this.subs.helper().precondition(!this.agentStateService.isInState((Enum)AlertState.ALARM), "can't execute commands in ALARM state.", new Supplier[0]).precondition(this.areTmaAndRotatorLocked(), "The telescope motion and rotator need to be locked before performing a filter exchange. Current status of the lock is %s", new Object[]{this.getTelescopeSoftwareLockStatus()}).precondition(this.trucksMustBeStraight(), "The FCS does not allow a filter exchange to be executed because the autochanger/camera is currently tilted. Ensure the rotator is set back to 0 before trying again.", new Supplier[0]).precondition(this.carousel.isAtStandby(), "carousel not stopped at STANDBY position", new Supplier[0]).precondition(this.autochangerNotInTravel(), "autochanger trucks are not at a HANDOFF or ONLINE or STANDBY", new Supplier[0]).precondition(this.autochanger.isAvailable(), "autochanger is not available", new Supplier[0]).precondition(this.latchesOpenOrClosed(), "%s: bad state for autochanger latches - have to be OPENED or CLOSED", new Object[]{this.autochanger.getLatches().getLockStatus()}).precondition(this.onlineClampsOpenOrLocked(), "%s: bad state for autochanger ONLINE clamps - have to be OPENED or LOCKED", new Object[]{this.autochanger.getOnlineClamps().getLockStatus()}).precondition(this.filterAtOnlineMustBeLocked(), "%s: bad state for autochanger ONLINE clamps - at ONLINE with a filter should be LOCKED", new Object[]{this.autochanger.getOnlineClamps().getLockStatus()}).precondition(this.filterAtStandbyMustBeHeld(), "When a filter is at STANDBY it must be held by autochanger or by carousel.", new Supplier[0]).precondition(this.autochanger.isNotHoldingFilter() || this.carouselHoldingFilterOrReadyToGrab(), "%s: bad state for carousel : should be holding a filter or ready to receive a filter", new Object[]{this.carousel.getClampsStateAtStandby()}).precondition(this.autochanger.isNotHoldingFilter() || this.carouselReadyToClampAtStandby(), "%s: bad state for carousel when a filter is on autochanger at HANDOFF or ONLINE: should be READYTOCLAMP", new Object[]{this.carousel.getClampsStateAtStandby()}).precondition(new Enum[]{FilterReadinessState.READY}).duration(Duration.ofMillis(this.setFilterSlowDuration)).enterFaultOnException(true).action(() -> {
                this.movementCounter.get(FcsActions.GeneralAction.SET_NO_FILTER).increment();
                long beginTime = System.currentTimeMillis();
                if (this.autochanger.isHoldingFilter()) {
                    FCSLOG.info(this.name + ": autochanger is holding filter: " + this.autochanger.getFilterOnTrucksObservatoryName());
                    FCSLOG.info(this.name + ": is going to store a filter on carousel");
                    this.storeFilterOnCarousel();
                } else {
                    FCSLOG.info(this.name + " no filter to store on carousel");
                }
                if (!this.autochanger.isEmpty() || !this.autochanger.isAtHandoff() && !this.autochanger.isAtOnline()) {
                    throw new FcsHardwareException(this.name + ": autochanger is not empty at handoff");
                }
                if (toOnline) {
                    this.agentStateService.updateAgentState(new Enum[]{ObservatoryFilterState.LOADING});
                    this.autochanger.goToOnline();
                }
                this.updateAgentState(FcsState.ONLINE_NONE);
                this.agentStateService.updateAgentState(new Enum[]{ObservatoryFilterState.LOADED});
                if (this.isPowerSaveWanted()) {
                    this.wakeFilterChanger(FcsEnumerations.CarouselPowerMode.GO_TO_SLEEP);
                }
                this.publishDataForMcm();
                this.publishData();
                FcsActions.GeneralAction.SET_NO_FILTER.publishDurationTopLevel(this.subs, System.currentTimeMillis() - beginTime);
            });
        }
    }

    @Command(type=Command.CommandType.ACTION, level=3, description="change subsystem state.")
    public void changeState(String state) {
        this.updateAgentState(FcsState.valueOf(state));
        FCSLOG.fine(() -> "SUBSYSTEM STATE=" + this.isInState(FcsState.valueOf(state)));
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="List the possible filters names as defined in the filter manager.")
    public List<String> listAllFilterNames() {
        return this.filterManager.getFilterNames();
    }

    public Map<String, Filter> getFcsFilterMap() {
        TreeMap<String, Filter> lf = new TreeMap<String, Filter>();
        if (this.autochanger.getFilterID() != 0) {
            lf.put("autochanger", this.filterManager.getFilterByID(this.autochanger.getFilterID()));
        }
        this.carousel.getSocketsMap().values().stream().forEach(socket -> {
            if (!socket.isEmpty() && socket.getFilterID() != 0) {
                lf.put(socket.getName(), this.filterManager.getFilterByID(socket.getFilterID()));
            }
        });
        return lf;
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="Print information on the filters physically in the LSSTCam.")
    public String printFilterMap() {
        StringBuilder sb = new StringBuilder("Filters on FES: \n");
        this.getFcsFilterMap().entrySet().stream().forEach(e -> {
            String socketName = (String)e.getKey();
            if (socketName != "autochanger" && !this.carousel.getSocketByName(socketName).isAvailable()) {
                sb.append("* ");
            }
            sb.append(socketName);
            sb.append("\n\t");
            sb.append(((Filter)e.getValue()).toString());
            sb.append("\n");
        });
        sb.append("\n'*' marks the currently unavailable sockets with filters.\n");
        return sb.toString();
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="Print general information on a filter.")
    public String printFilterInfo(String filterName) {
        return this.filterManager.getFilterByName(filterName).toString();
    }

    private Map<Integer, String> getAvailableFilterMap() {
        TreeMap<Integer, String> availableFilterMap = new TreeMap<Integer, String>();
        if (!this.autochanger.isAvailable()) {
            if (this.autochanger.isAtOnline()) {
                availableFilterMap.put(this.autochanger.getFilterID(), this.autochanger.getFilterOnTrucksName());
            }
            return availableFilterMap;
        }
        if (!this.carousel.isAvailable()) {
            if (!this.carousel.socketAtStandby.isEmpty()) {
                if (this.carousel.socketAtStandby.isAvailable()) {
                    availableFilterMap.put(this.carousel.socketAtStandby.getFilterID(), this.carousel.socketAtStandby.getFilterName());
                }
                availableFilterMap.put(0, "NONE");
            } else if (!this.autochanger.isEmpty()) {
                availableFilterMap.put(this.autochanger.getFilterID(), this.autochanger.getFilterOnTrucksName());
                if (this.carousel.socketAtStandby.isAvailable()) {
                    availableFilterMap.put(0, "NONE");
                }
            }
            return availableFilterMap;
        }
        if (!this.autochanger.isEmpty() && !this.autochanger.isAtStandby()) {
            availableFilterMap.put(this.autochanger.getFilterID(), this.autochanger.getFilterOnTrucksName());
            if (!this.carousel.socketsMap.values().stream().anyMatch(socket -> socket.isEmpty() && socket.isAvailable())) {
                return availableFilterMap;
            }
        }
        this.carousel.getSocketsMap().values().stream().forEach(socket -> {
            if (!socket.isEmpty() && socket.getFilterID() != 0 && socket.isAvailable()) {
                Filter f = this.filterManager.getFilterByID(socket.getFilterID());
                availableFilterMap.put(f.getFilterID(), f.getName());
            }
        });
        availableFilterMap.put(0, "NONE");
        return availableFilterMap;
    }

    private List<Integer> getAvailableFiltersID() {
        return this.getAvailableFilterMap().keySet().stream().filter(filterID -> filterID != 0).toList();
    }

    @Command(type=Command.CommandType.ACTION, level=0, description="Move filter to ONLINE position.", timeout=180000, autoAck=false)
    public void setFilterByName(String filterName) {
        this.subs.helper().precondition(this.filterManager.containsFilterName(filterName), "%s: Unknown filter name : %s", new Object[]{this.name, filterName}).precondition(this.isInState(FilterReadinessState.READY)).action(() -> this.setFilter(this.filterManager.getFilterID(filterName)));
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="Provide a mapping carousel sockets to available filters ID in the Filter Exchange System")
    public Map<String, Integer> getSocketsToAvailableFiltersID() {
        List<Integer> availableFiltersID = this.getAvailableFiltersID();
        TreeMap<String, Integer> fm = new TreeMap<String, Integer>();
        this.carousel.getSocketsMap().values().stream().forEach(socket -> {
            if (socket.isEmpty()) {
                if (socket.isAtStandby() && availableFiltersID.contains(this.autochanger.getFilterID())) {
                    fm.put(socket.getName(), this.autochanger.getFilterID());
                }
            } else if (availableFiltersID.contains(socket.getFilterID())) {
                fm.put(socket.getName(), socket.getFilterID());
            }
        });
        return fm;
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="Provide the list of available filters in the Filter Exchange System formatted as wanted by the OCS Bridge")
    public List<String> getAvailableFilters() {
        return this.getAvailableFilterMap().keySet().stream().map(filterID -> this.filterManager.getObservatoryNameByID((int)filterID)).collect(Collectors.toList());
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="Provide the list of installed filters in the Filter Exchange System formatted as wanted by the OCS Bridge.")
    public List<String> getInstalledFilters() {
        return this.getFcsFilterMap().entrySet().stream().map(entry -> ((Filter)entry.getValue()).getObservatoryName()).collect(Collectors.toList());
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="Provide the name of the filter currently online, formatted as wanted by the OCS Bridge")
    public String getOnlineFilterName() {
        int filterId = 0;
        if (this.autochanger.isAtOnline()) {
            filterId = this.autochanger.getFilterID();
        }
        return this.filterManager.getObservatoryNameByID(filterId);
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="Provide the maximum angle from vertical at which a filter exchange can be executed as wanted by the OCS Bridge.")
    public double getMaxAngleForFilterChange() {
        return 90.0;
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="Provide the maximum angle from vertical at which a filter exchange can be executed in normal speed (fast) as wanted by the OCS Bridge.")
    public double getMaxAngleForFastFilterChange() {
        int airmass = 3;
        return Math.acos(1.0 / (double)airmass) * 180.0 / Math.PI;
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="Provide the maximum duration of a filter exchange in slow-mode conditions as wanted by the OCS Bridge.")
    public Duration getDurationForSlowFilterChange(String filterName) {
        return Duration.ofMillis(this.setFilterSlowDuration);
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="Provide the maximum duration of a filter exchange in normal conditions as wanted by the OCS Bridge.")
    public Duration getDurationForFastFilterChange(String filterName) {
        return Duration.ofMillis(this.setFilterFastDuration);
    }

    @Command(type=Command.CommandType.QUERY, level=0, description=" Is the power save (sleep) mode authorized in the FES configuration")
    public boolean isPowerSaveEnabled() {
        return this.carousel.isPowerSaveEnabled();
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="Does the carousel automatically goes to sleep after a filter change")
    public boolean isPowerSaveWanted() {
        return this.carousel.isPowerSaveWanted();
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="Is the carousel in power save (sleep) mode")
    public boolean isPowerSaveActivated() {
        return this.carousel.isPowerSaveActivated();
    }

    @Command(type=Command.CommandType.ACTION, level=0, description="Control the power state of the filter changer carousel. Options are 0: go to sleep, 1: wake up and go back to sleep after setFilter, 2: wake up and stay up")
    public void wakeFilterChanger(int mode) {
        try {
            FcsEnumerations.CarouselPowerMode wakeUpMode = FcsEnumerations.CarouselPowerMode.values()[mode];
            FCSLOG.info("wakeFilterChanger was issued with mode : " + wakeUpMode);
            this.wakeFilterChanger(wakeUpMode);
        }
        catch (ArrayIndexOutOfBoundsException ex) {
            throw new RejectedCommandException("The only available modes for wakeFilterChanger are 0: go to sleep, 1: wake up, 2: wake up and stay up");
        }
    }

    private void wakeFilterChanger(FcsEnumerations.CarouselPowerMode mode) {
        switch (mode) {
            case GO_TO_SLEEP: {
                if (!this.isPowerSaveEnabled()) {
                    FCSLOG.info(this.name + " power save was requested but the current configuration does not allow it");
                    return;
                }
                this.carousel.setPowerSaveWanted(true);
                if (this.agentStateService.isInState((Enum)CarouselPowerState.REGULAR)) {
                    this.carousel.powerSave();
                    break;
                }
                FCSLOG.info("Filter changer already in power save");
                break;
            }
            case WAKE_UP: {
                this.carousel.setPowerSaveWanted(true);
                if (this.agentStateService.isInState((Enum)CarouselPowerState.LOW_POWER)) {
                    this.carousel.powerOn();
                    break;
                }
                FCSLOG.info("Filter changer already waken up");
                break;
            }
            case STAY_UP: {
                this.carousel.setPowerSaveWanted(false);
                if (this.agentStateService.isInState((Enum)CarouselPowerState.LOW_POWER)) {
                    this.carousel.powerOn();
                    break;
                }
                FCSLOG.info("Filter changer already waken up");
            }
        }
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="Approximate duration of the wake up operation for the filter exchange system")
    public Duration getDurationForWakeUp(int mode) {
        return Duration.ofMillis(this.carousel.getWakeUpTimeout());
    }

    @Command(type=Command.CommandType.ACTION, level=0, description="Put the filter changer in sleep mode. It will naturally wake up before a filter change.")
    public void sleep() {
        this.wakeFilterChanger(FcsEnumerations.CarouselPowerMode.GO_TO_SLEEP);
    }

    @Command(type=Command.CommandType.ACTION, level=0, description="Wake up the filter changer. It will naturally go back to sleep mode after a filter change.")
    public void wakeUp() {
        this.wakeFilterChanger(FcsEnumerations.CarouselPowerMode.WAKE_UP);
    }

    @Command(type=Command.CommandType.ACTION, level=0, description="Wake up the filter changer and make sure it DOES NOT go back to sleep mode after a filter change.")
    public void wakeUpAndStayUp() {
        this.wakeFilterChanger(FcsEnumerations.CarouselPowerMode.STAY_UP);
    }

    @Command(type=Command.CommandType.ACTION, level=0, description="Load a filter from the loader to the camera. The loader must hold a filter at STORAGE position.The autochanger must be empty at HANDOFF, latches open. At the end of this command thefilter is held by autochanger at Handoff position, and the loader carrier is empty at STORAGE.", timeout=480000, autoAck=false)
    public void loadFilter() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("loadFilter");){
            this.updateStateWithSensors();
            this.subs.helper().precondition(!this.agentStateService.isInState((Enum)AlertState.ALARM), "can't execute commands in ALARM state.", new Supplier[0]).precondition(this.autochanger.isAtHandoff() && this.autochanger.isEmpty(), " autochanger is not empty at HANDOFF position; can't load a filter.", new Supplier[0]).precondition(this.autochanger.getLatches().isOpened(), "Autochanger latches must be open before loadFilter command.", new Supplier[0]).duration(Duration.ofMillis(this.loadUnloadFilterMaxDuration)).enterFaultOnException(true).action(() -> {
                this.movementCounter.get(FcsActions.GeneralAction.LOAD_FILTER).increment();
                long beginTime = System.currentTimeMillis();
                if (!this.autochanger.isAtHandoffForLoader()) {
                    FCSLOG.info(this.name + " autochanger not in correct position for exchange with loader, trying to align the trucks");
                    this.autochanger.goToHandOffForLoader();
                }
                if (this.loader.getCarrier().isLooslyBetweenStorageAndEngaged() && !this.loader.isClampedOnFilter()) {
                    FCSLOG.info(this.name + " loader between STORAGE and ENGAGED with hooks not CLAMPED. Starting recovery to get back to a CLAMPED state to continue the load operation. ");
                    this.loader.getClamp().recoveryUnclamp();
                    FcsUtils.sleep(50, this.name);
                    this.updateStateWithSensors();
                    this.loader.getClamp().recoveryClamp();
                    FcsUtils.sleep(50, this.name);
                    this.updateStateWithSensors();
                }
                this.loader.moveFilterToHandoff();
                FcsUtils.sleep(50, this.name);
                this.updateStateWithSensors();
                this.autochanger.closeLatches();
                FcsUtils.sleep(50, this.name);
                this.updateStateWithSensors();
                this.loader.openClampAndMoveEmptyToStorage();
                FcsUtils.sleep(50, this.name);
                this.updateStateWithSensors();
                FcsActions.GeneralAction.LOAD_FILTER.publishDurationTopLevel(this.subs, System.currentTimeMillis() - beginTime);
            });
        }
    }

    @Command(type=Command.CommandType.ACTION, level=0, description="Load a filter from the loader to the cameraand store it on socket which name is given as argument. The loader must hold a filter at STORAGE position.The autochanger must be empty at HANDOFF, latches open. At the end of this command thefilter is store on carousel socket which name is given as argument, autochanger is empty at Handoff position, and the loader carrier is empty at STORAGE.")
    public void loadFilterOnSocket(String socketName) {
        this.updateStateWithSensors();
        FcsUtils.checkSocketName(socketName);
        CarouselSocket socket = this.carousel.getSocketsMap().get(socketName);
        if (!socket.isEmpty()) {
            throw new RejectedCommandException(String.format("Can't execute loadFilterOnSocket because %s is not empty. Select an empty socket.", socketName));
        }
        if (!socket.isAvailable()) {
            throw new RejectedCommandException(String.format("Can't execute loadFilterOnSocket because %s is not available. Select an available socket.", socketName));
        }
        this.loadFilter();
        this.carousel.rotateSocketToStandby(socketName);
        this.storeFilterOnCarousel();
    }

    @Command(type=Command.CommandType.ACTION, level=0, description="Unload a filter from the camera to the loader. The camera has to be at horizontal position. The filter has to be on Autochanger at HANDOFF. Loader carrier must be empty at STORAGE. \nAt the end of this command, autochanger is empty at HANDOFF, latches are open, and loader carrier is holding the filter at STORAGE (hooks are clamped)", timeout=480000, autoAck=false)
    public void unloadFilter() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("unloadFilter");){
            this.updateStateWithSensors();
            this.subs.helper().precondition(!this.agentStateService.isInState((Enum)AlertState.ALARM), "can't execute commands in ALARM state.", new Supplier[0]).precondition(this.autochanger.isAtHandoff() && this.autochanger.isHoldingFilter(), " autochanger is not holding a filter at HANDOFF position; can't unload a filter.", new Supplier[0]).precondition(this.autochanger.isAvailable(), "autochanger is not available", new Supplier[0]).duration(Duration.ofMillis(this.loadUnloadFilterMaxDuration)).enterFaultOnException(true).action(() -> {
                this.movementCounter.get(FcsActions.GeneralAction.UNLOAD_FILTER).increment();
                long beginTime = System.currentTimeMillis();
                if (!this.autochanger.isAtHandoffForLoader()) {
                    if (this.autochanger.isLinearRailMotionAllowed()) {
                        this.autochanger.goToHandOffForLoader();
                    } else {
                        throw new RejectedCommandException("Autochanger trucks position is not correct for exchange with loader  but command goToHandOffForLoader can't be executed because PLC doesn't allow to move trucks.");
                    }
                }
                this.loader.moveEmptyToHandoffAndClose();
                FcsUtils.sleep(50, this.name);
                this.updateStateWithSensors();
                this.autochanger.openLatches();
                FcsUtils.sleep(50, this.name);
                this.updateStateWithSensors();
                this.loader.moveFilterToStorage();
                FcsUtils.sleep(50, this.name);
                this.updateStateWithSensors();
                FcsActions.GeneralAction.UNLOAD_FILTER.publishDurationTopLevel(this.subs, System.currentTimeMillis() - beginTime);
            });
        }
    }

    @Command(type=Command.CommandType.ACTION, level=1, description="Recover from a timeout or an exception raised during the carousel clamping process. This command moves the filter back to handoff position and releases both carousel clamps.")
    public void recoveryCarouselClamping() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("recoveryCarouselClamping");){
            this.carousel.unlockClamps();
            this.autochanger.moveToApproachStandbyPositionWithLowVelocity();
            this.carousel.releaseClamps();
            this.autochanger.moveToHandoffWithHighVelocity();
        }
    }

    @Override
    @Command(type=Command.CommandType.QUERY, level=0, description="check if controllers are in fault")
    public void checkControllers() {
        this.bridge.checkControllers();
    }

    @Override
    public synchronized void updateStateWithSensors() {
        long beginTime = System.currentTimeMillis();
        if (!FcsUtils.isSimu() && beginTime - this.lastUpdateStateWithSensors.get() < 100L) {
            FCSLOG.info(this.name + " do not repeat updateStateWithSensors; last one was executed less than 100ms ago.");
            return;
        }
        this.lastUpdateStateWithSensors.set(beginTime);
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("updateStateWithSensors-fcs");){
            this.carousel.updateStateWithSensors();
            if (this.carousel.isAtStandby()) {
                this.carousel.getSocketAtStandby().updateFilterID();
            }
            this.autochanger.updateStateWithSensors();
            if (this.autochanger.getAutochangerTrucks().isAtOnline() && this.autochanger.isHoldingFilter() && this.autochanger.getOnlineClamps().isLocked()) {
                String filterName = this.filterManager.getFilterNameByID(this.autochanger.getFilterID());
                this.updateAgentState(FcsState.valueOf("ONLINE_" + filterName.toUpperCase()));
            } else {
                this.updateAgentState(FcsState.valueOf("ONLINE_NONE"));
            }
            if (this.loader.isCANbusConnected()) {
                this.loader.updateStateWithSensors();
            }
        }
        if (this.firstMcmPublication) {
            this.publishDataForMcm();
            this.firstMcmPublication = false;
        }
    }

    @Override
    public void initializeHardware() {
        this.carousel.initializeHardware();
        this.autochanger.initializeHardware();
        this.loader.initializeHardware();
        this.postStart();
        FCSLOG.info("initializeHardware done");
    }

    @Override
    public void publishData() {
        super.publishData();
        this.bridgeToLoader.publishData();
    }

    public void publishDataForMcm() {
        if (this.autochanger.getAutochangerTrucks().getPosition() == 0) {
            FcsUtils.sleep(100, "main");
            this.autochanger.updateStateWithSensors();
        }
        FCSLOG.info(this.name + " publishing data for MCM..");
        this.updateAgentState(this.autochanger.getAutochangerInclinationState());
        this.subs.publishSubsystemDataOnStatusBus(this.getDataForMcm());
    }

    public void publishDataProviderCurrentData(AgentInfo ... agents) {
        for (AgentInfo agent : agents) {
            if (!AgentInfo.AgentType.MCM.equals((Object)agent.getType())) continue;
            FCSLOG.info(this.name + " published for MCM that just joined the buses.");
            this.publishDataForMcm();
            break;
        }
    }

    private KeyValueData getDataForMcm() {
        KeyValueDataList kvdl = new KeyValueDataList();
        kvdl.addData("filter_on_autochanger", (Serializable)((Object)this.filterManager.getObservatoryNameByID(this.autochanger.getFilterID())));
        kvdl.addData("filter_previous_socketID", (Serializable)Integer.valueOf(this.previousSocketID));
        kvdl.addData("autochanger_trucks_position", (Serializable)Integer.valueOf(this.autochanger.getAutochangerTrucks().getPosition()));
        kvdl.addData("autochanger_trucks_state", (Serializable)((Object)this.autochanger.getAutochangerTrucksState()));
        kvdl.addData("proximity", (Serializable)Integer.valueOf(this.autochanger.getOnlineProximityDistance()));
        return new KeyValueData("fcs/mcm", (Serializable)kvdl);
    }

    @Override
    public void postShutdown() {
        FCSLOG.info(this.name + " is shutting down.");
        this.bridge.doShutdown();
        if (this.loader.isCANbusConnected()) {
            this.bridgeToLoader.doShutdown();
        }
        FCSLOG.info(this.name + " is shutdown.");
    }

    public String vetoTransitionToNormalMode() {
        if (this.autochanger.isEmptyOnline() || this.autochanger.isFilterClampedOnline()) {
            this.updateAgentState(ObservatoryFilterState.LOADED);
        } else {
            this.updateAgentState(ObservatoryFilterState.UNLOADED);
        }
        if (!this.isChangerReady()) {
            String msg = "The filter exchange system has devices not fully booted, cannot switch to Normal mode";
            FCSLOG.info(this.name + " vetoTransitionToNormalMode failed:\n" + msg);
            return msg;
        }
        if (!this.agentStateService.isInState((Enum)ObservatoryFilterState.LOADED)) {
            String msg = String.format("The filter exchange system is currently in filter state = %s.\nSwitching to Normal mode requires it to be in a LOADED state.\nUse the autochanger level 1 command moveOnlineForScience, with or without filter, to prepare the system.", this.agentStateService.getState(ObservatoryFilterState.class));
            FCSLOG.info(this.name + " vetoTransitionToNormalMode failed:\n" + msg);
            return msg;
        }
        FCSLOG.info(this.name + " vetoTransitionToNormalMode succeeded: current filter ONLINE = " + this.getOnlineFilterName());
        this.publishDataForMcm();
        this.sleep();
        return super.vetoTransitionToNormalMode();
    }
}

