package org.lsst.ccs.subsystems.fcs.common;

import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.subsystem.common.service.DataAccumulationService;
import java.util.logging.Logger;

/**
 * Maintains persistent counters that are also trended. Uses both the
 * data accumulation and the trending services.
 * Inspired by org-lsst-ccs-subsystem-shutter class PersistentCounter
 */

public class PersistentCounter {
    private final String path;
    private final Subsystem subsys;
    private final DataAccumulationService dataAccumulationService;
    private static final Logger FCSLOG = Logger.getLogger(PersistentCounter.class.getName());

    private PersistentCounter(final String path, final Subsystem subsys) {
        this.path = path;
        this.subsys = subsys;
        this.dataAccumulationService = subsys.getAgentService(DataAccumulationService.class);
    }

    /**
     * Makes a new persistent counter. Must be called during the init() stage of the subsystem's life cycle.
     *
     * @param path The path name.
     */
    public static PersistentCounter newCounter(final String path, final Subsystem subsys, String actionName) {
        final PersistentCounter ctr = new PersistentCounter(path, subsys);
        ctr.register(actionName);
        return ctr;
    }

    /**
     * Registers the path with {@code DataProviderDictionaryService} and with {@code DataAccumulationService}.
     */
    private void register(String actionName) {
        final DataProviderDictionaryService dataSvc
            = subsys.getAgentService(DataProviderDictionaryService.class);
        // register data
        dataSvc.registerData(new KeyValueData(path, 0.0));
        DataProviderInfo info = dataSvc.getDataProviderDictionary().getDataProviderInfoForPath(path);
        info.addAttribute(DataProviderInfo.Attribute.DESCRIPTION, "Counter for the action " + actionName);
        info.addAttribute(DataProviderInfo.Attribute.UNITS, "unitless");
        info.addAttribute(DataProviderInfo.Attribute.TYPE, "double"); // accumulateData works with double
        // use accumulation service for this data
        dataAccumulationService.registerAccumulatedDataPath(path);
    }

    /**
     * Increments the counter value by 1 and publishes the new value (if available).
     */
    public void increment() {
        dataAccumulationService.accumulateData(path, 1.0); // accumulateData works with double
        final double sum = getValue();
        if (!Double.isNaN(sum)) {
            subsys.publishSubsystemDataOnStatusBus(new KeyValueData(path, sum));
        } else {
            FCSLOG.warning("AccumulatedValue is NaN, not published: check your trending DB");
        }
    }

    /**
     * Gets the current value of the counter.
     *
     * @return The value, or NaN if the REST file server is unreachable.
     */
    private double getValue() {
        return dataAccumulationService.getAccumulatedValueForPath(path);
    }
}
