
package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;

/**
 * This is the model for a single hook in the loader.
 * Each one is checked in CLOSED and OPENED position by 2 sensors.
 * These sensors have to give a consistent response otherwise the hook is in ERROR.
 * There is 4 hooks in the loader to hold a filter.
 *
 * @author virieux
 */
public class LoaderHook {

    @LookupName
    private String name;

    private final DigitalSensor closeSensor;
    private final DigitalSensor openSensor;

    private LockStatus lockStatus = LockStatus.UNKNOWN;

    private boolean inError;

    public LoaderHook(DigitalSensor closeSensor, DigitalSensor openSensor) {
        this.closeSensor = closeSensor;
        this.openSensor = openSensor;
    }

    public LockStatus getLockStatus() {
        return lockStatus;
    }

    /**
     * This methods updates lockStatus from the values return by the sensors.
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Update state from sensors values.")
    public void updateState() {
        inError = this.closeSensor.isOn() && this.openSensor.isOn();

        if (closeSensor.isOn() && openSensor.isOn()) {
            lockStatus = LockStatus.ERROR;
        } else if (closeSensor.isOn()) {
            lockStatus = LockStatus.CLOSED;
        } else if (openSensor.isOn()) {
            lockStatus = LockStatus.OPENED;
        } else {
            lockStatus = LockStatus.INTRAVEL;
        }
    }

    @Override
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "List and display Loader hook information.")
    public synchronized String toString() {
        updateState();
        StringBuilder sb = new StringBuilder("+> " + name + " <+").append("\n");
        sb.append("+ LockStatus: ").append(lockStatus.name()).append("\n");
        sb.append("+ OpenSensor: ").append(openSensor.isOn() ? "ON" : "OFF").append("\n");
        sb.append("+ CloseSensor: ").append(closeSensor.isOn() ? "ON" : "OFF").append("\n");
        return sb.toString();
    }

    /**
     * To create an object StatusDataPublishedByLoaderHook for publication on the
     * STATUS bus.
     *
     * @return
     */
    StatusDataPublishedByLoaderHook createStatusDataPublishedByLoaderHook() {
        return new StatusDataPublishedByLoaderHook(closeSensor.isOn(), openSensor.isOn(), inError, lockStatus);
    }

}
