package org.lsst.ccs.subsystems.fcs;

import java.time.Duration;
import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Logger;

import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.commons.annotations.Persist;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.GeneralAction;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert;
import org.lsst.ccs.subsystems.fcs.common.AlertRaiser;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.common.PersistentCounter;
import org.lsst.ccs.subsystems.fcs.common.PlutoGatewayInterface;
import org.lsst.ccs.subsystems.fcs.drivers.CanOpenSeneca4RTD;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.states.AutochangerInclinationState;
import org.lsst.ccs.subsystems.fcs.states.AutochangerTrucksState;
import org.lsst.ccs.subsystems.fcs.utils.ActionGuard;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * This class is used with the final hardware and prototype in which we have
 * online clamps to hold the filter at online position. It's not used for Single
 * Filter Test.
 *
 * CAMERA PROTECTION SYSTEM Signal coming from carousel : CS : Carousel stopped
 * on socket CF0: No filter in carousel socket CF1: Filer in carousel socket
 * CFC: Filter clamped on carousel socket
 *
 * Signal coming from loader : LPS: Loader at Storage position (loader is
 * connected) LRH: Loader at Handoff position, closed but not clamped
 *
 * Local Protection Module Signals emitted by autochanger: enableRailLin1 enable driver
 * linear rail (X+) enableRailLin2 enable follower linear rails (X-) enableClamps
 * enableLatches
 *
 * Sensors in AC1 and AC2 and not in PROTO :
 *
 * loaderPresenceSensors * lockOutSensors
 *
 *
 * @author virieux
 */
//TODO: Consider adding MobileItem to the implemented object
public class Autochanger implements FilterHolder, AlertRaiser, HasLifecycle {
    private static final Logger FCSLOG = Logger.getLogger(Autochanger.class.getName());
    @LookupField(strategy = LookupField.Strategy.TOP)
    private Subsystem subs;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private ConfigurationService configurationService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    protected DataProviderDictionaryService dataProviderDictionaryService;

    @LookupName
    protected String name;

    // TODO: Find proper way to add `path` variable to the Autochanger like other elements.
    @LookupField(strategy = LookupField.Strategy.SIBLINGS, pathFilter = FCSCst.CHANGER_TCPPROXY_NAME)
    private BridgeToHardware tcpProxy;

    private final PlutoGatewayInterface plutoGateway;

    @LookupField(strategy = LookupField.Strategy.SIBLINGS, pathFilter = "filterManager")
    private FilterManager filterManager;

    @LookupField(strategy = LookupField.Strategy.SIBLINGS, pathFilter = "filterIdentificator")
    protected FilterIdentificator filterIdentificator;

    @LookupField(strategy = LookupField.Strategy.CHILDREN, pathFilter = "autochangerTrucks")
    private AutochangerTwoTrucks autochangerTrucks;

    @LookupField(strategy = LookupField.Strategy.CHILDREN, pathFilter = "latches")
    protected AutochangerTwoLatches latches;

    @LookupField(strategy = LookupField.Strategy.CHILDREN, pathFilter = "onlineClamps")
    private AutochangerThreeOnlineClamps onlineClamps;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private MainModule main;

    @LookupField(strategy = LookupField.Strategy.TREE, pathFilter = ".*\\/tempSensorsDevice1")
    private CanOpenSeneca4RTD tempSensorsDevice1;

    @LookupField(strategy = LookupField.Strategy.TREE, pathFilter = ".*\\/tempSensorsDevice2")
    private CanOpenSeneca4RTD tempSensorsDevice2;

    @LookupField(strategy = LookupField.Strategy.TREE)
    protected AgentPeriodicTaskService periodicTaskService;

    @LookupField(strategy = LookupField.Strategy.SIBLINGS, pathFilter = ".*carousel")
    private FilterHolder carousel;

    @LookupField(strategy = LookupField.Strategy.SIBLINGS, pathFilter = ".*loader")
    private FilterHolder loader;

    @ConfigurationParameter(range = "0..5000", description = "Time to wait until protection system signals are updated", units = "millisecond", category = "autochanger")
    public volatile long timeToUpdateProtectionSystem = 2000;

    @ConfigurationParameter(range = "0..100", description = "Time to wait between activateBrake and disableOperation for online clamps", units = "millisecond", category = "autochanger")
    private volatile int waitTimeForBrakeOC = 20;

    @ConfigurationParameter(range = "0..100", description = "Time to wait between activateBrake and disableOperation for linear rails", units = "millisecond", category = "autochanger")
    private volatile int waitTimeForBrakeLR = 20;

    @Persist
    protected volatile int filterOnTrucksID;

    /**
     * Flag to be set by the Camera team to indicate whether we can use the autochanger
     * If we do want to prohibit filter changes for some reason, use the setAvailable
     * command to switch this flag to false, or back again to true.
     */
    @Persist
    private volatile boolean available = true;

    /**
     * CPS signal : carousel_CS Carousel stop on socket
     */
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS, pathFilter = ".*\\/carouselStoppedAtStandbySensors")
    private ComplementarySensors carouselStoppedAtStandbySensors;


    /**
     * Loader Presence Sensor : true if loader is connected on autochanger
     */
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS, pathFilter = ".*\\/loaderPresenceSensors")
    private ComplementarySensors loaderPresenceSensors;

    /**
     * Local Protection Module signals
     */
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS, pathFilter = ".*\\/lpmLinearRail1Status")
    private DigitalSensor lpmLinearRail1Status;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS, pathFilter = ".*\\/lpmLinearRail2Status")
    private DigitalSensor lpmLinearRail2Status;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS, pathFilter = ".*\\/lpmOnlineClampsStatus")
    private DigitalSensor lpmOnlineClampsStatus;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS, pathFilter = ".*\\/lpmLatchesStatus")
    private DigitalSensor lpmLatchesStatus;

    //AutochangerInclination for MCM
    //OUT_AIN (DigitalSensor)
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS, pathFilter = ".*\\/OUT_AIN")
    private DigitalSensor AIN;

    // Counter for actions
    private PersistentCounter grabFilterAtStandbyCounter;
    private PersistentCounter moveAndClampFilterOnlineCounter;

    // Variable used to distinguished between autochanger1, 2 and PROTO
    private String subsystemIdentifier = "";

    // Autochanger temperatures
    private double[] temperatures = new double[8];

    /* Store last time a function was executed to check for a minimal interval between execution */
    private AtomicLong lastUpdateStateWithSensors = new AtomicLong(0);

    public Autochanger(PlutoGatewayInterface plutoGateway) {
        this.plutoGateway = plutoGateway;
    }

    @Override
    public Subsystem getSubsystem() {
        return subs;
    }

    @Override
    public AlertService getAlertService() {
        return alertService;
    }

    public BridgeToHardware getTcpProxy() {
        return tcpProxy;
    }

    public int getWaitTimeForBrakeOC() {
        return waitTimeForBrakeOC;
    }

    public int getWaitTimeForBrakeLR() {
        return waitTimeForBrakeLR;
    }

    /**
     * For MCM
     *
     * @return
     */
    public AutochangerInclinationState getAutochangerInclinationState() {
        if (AIN.isOn()) {
            return AutochangerInclinationState.STRAIGHT;
        } else {
            return AutochangerInclinationState.TILTED;
        }
    }

    /**
     * For MCM
     *
     * @return trucks location
     */
    public AutochangerTrucksState getAutochangerTrucksState() {
        if (isAtHandoff()) {
            return AutochangerTrucksState.HANDOFF;
        } else if (isAtOnline()) {
            return AutochangerTrucksState.ONLINE;
        } else if (isAtStandby()) {
            return AutochangerTrucksState.STANDBY;
        } else {
            return AutochangerTrucksState.IN_TRAVEL;
        }
    }

    /**
     * For MCM
     *
     *
     * @return online proximity distance read on proximity sensor
     */
    public int getOnlineProximityDistance() {
        return autochangerTrucks.getOnlineProximityDistance();
    }

    @Override
    public void build() {
        dataProviderDictionaryService.registerClass(StatusDataPublishedByAutochangerSeneca.class, name + "/temperatures");
        periodicTaskService.scheduleAgentPeriodicTask(new AgentPeriodicTask("PublishAutochangerTemperatures", this::publishSenecaTemperatures)
            .withIsFixedRate(true).withPeriod(Duration.ofMinutes(10)));
        plutoGateway.registerSubsystemIdentifier(name);
        // Actions counters
        grabFilterAtStandbyCounter = PersistentCounter.newCounter(GeneralAction.GRAB_FILTER_AT_STANDBY.getCounterPath(), getSubsystem(), GeneralAction.GRAB_FILTER_AT_STANDBY.name());
        moveAndClampFilterOnlineCounter = PersistentCounter.newCounter(GeneralAction.MOVE_AND_CLAMP_FILTER_ONLINE.getCounterPath(), getSubsystem(), GeneralAction.MOVE_AND_CLAMP_FILTER_ONLINE.name());
        GeneralAction.GRAB_FILTER_AT_STANDBY.registerDurationTopLevel(dataProviderDictionaryService);
        GeneralAction.MOVE_AND_CLAMP_FILTER_ONLINE.registerDurationTopLevel(dataProviderDictionaryService);

    }

    @Override
    public void init() {

        ClearAlertHandler alwaysClear = new ClearAlertHandler() {
            @Override
            public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
            }
        };

        alertService.registerAlert(FcsAlert.HARDWARE_ERROR.getAlert(), alwaysClear);
        alertService.registerAlert(FcsAlert.HARDWARE_ERROR.getAlert(plutoGateway.getName()), alwaysClear);
        alertService.registerAlert(FcsAlert.AC_SENSOR_ERROR.getAlert(), alwaysClear);
        alertService.registerAlert(FcsAlert.AC_SENSOR_ERROR.getAlert(name), alwaysClear);
    }
    /**
     * Return true if an error has been detected on sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Return true if an error has been detected on sensors.")
    public boolean isSensorsInError() {
        return latches.isInError() || onlineClamps.isInError() || autochangerTrucks.isPositionSensorsInError();
    }

    /**
     * returns trucks to be able to send commands to trucks in fcs subsystem.
     *
     * @return trucks
     */
    public AutochangerTwoTrucks getAutochangerTrucks() {
        return autochangerTrucks;
    }

    /**
     * returns onlineClamps to be able to send commands to onlineClamps in fcs
     * subsystem.
     *
     * @return onlineClamps
     */
    public AutochangerThreeOnlineClamps getOnlineClamps() {
        return onlineClamps;
    }

    /**
     * return latches
     *
     * @return
     */
    public AutochangerTwoLatches getLatches() {
        return latches;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return the subsystem identifier autochanger1, 2 or PROTO")
    public String getIdentifier() {
        return subsystemIdentifier;
    }

    /**
     * If autochanger holds a filter, returns filterID, else returns 0.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "If autochanger holds a filter, return filterID, else return 0.")
    @Override
    public int getFilterID() {
        return filterOnTrucksID;
    }

    /**
     * Used by simulator only.
     *
     * @param filterOnTrucksID
     */
    public void setFilterOnTrucksID(int filterOnTrucksID) {
        int formerFilterOnTrucksID = this.filterOnTrucksID;

        if (formerFilterOnTrucksID != filterOnTrucksID) {
            this.filterOnTrucksID = filterOnTrucksID;
            subs.getAgentPersistenceService().persistNow();
        }
    }

    /**
     *
     * @param filterID
     * @return return true if filter with filterID is on autochanger
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "return true if filter with filterID is on autochanger")
    public boolean isFilterOnTrucks(int filterID) {
        return filterOnTrucksID == filterID && filterOnTrucksID != 0;
    }

    /**
     * Test to check the presence of a filter at ONLINE position
     *
     * @return true if filter ONLINE, false otherwise
     */
    public boolean isFilterOnline() {
        return this.isAtOnline() && this.isHoldingFilter();
    }

    /**
     * Test to check the presence of a precise filter at ONLINE position
     *
     * @param filterID
     * @return true if filterID is on Autochanger and trucks are ONLINE and latches
     *         are closed.
     */
    public boolean isFilterOnline(int filterID) {
        return isFilterOnline() && isFilterOnTrucks(filterID);
    }

    /**
     * Test to check if a filter is currently locked at ONLINE position
     *
     * @return true if filter is at ONLINE and online clamps are locked, false otherwise.
     */
    public boolean isFilterClampedOnline() {
        return isFilterOnline() && onlineClamps.isLocked();
    }

    /**
     *
     * @param filterID
     * @return true if filterID is on Autochanger and trucks are ONLINE and locked
     */
    public boolean isFilterClampedOnline(int filterID) {
        return isFilterOnline(filterID) && onlineClamps.isLocked();
    }

    /**
     * Test to check if the autochanger is empty at ONLINE position
     *
     * @return true if empty and at ONLINE position, false otherwise
     */
    public boolean isEmptyOnline() {
        return isEmpty() && isAtOnline();
    }

    /**
     * If autochanger holds a filter, return filter name else return null.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Name of filter in autochanger, or NONE if no filter")
    public String getFilterOnTrucksName() {
        return filterManager.getFilterNameByID(filterOnTrucksID);
    }

    /**
     * Return filter observatory name on trucks.
     * This handles cases when there is no filter.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Name of filter in autochanger, or NONE if no filter")
    public String getFilterOnTrucksObservatoryName() {
        return filterManager.getObservatoryNameByID(filterOnTrucksID);
    }


    /**
     * Returns the boolean field loaderConnected. If the loaderConnected boolean is
     * being updated and waits for a response from a sensor, this methods waits
     * until loaderConnected is updated. If the field loaderConnected is not being
     * updated, it returns immediately the field loaderConnected.
     *
     * @return atHandoff
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the loader is connected to the camera. "
            + "This command doesn't read again the sensors.")
    public boolean isLoaderConnected() {
        return loaderPresenceSensors.isOn() && !loaderPresenceSensors.isInError();
    }

    /**
     *
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the carousel is holding the filter at STANDBY position.")
    public boolean isCarouselHoldingFilterAtStandby() {
        return this.carousel.isAtStandby() && this.carousel.isHoldingFilter();
    }

    /**
     * Return true if loader is holding a filter. This command doesn't
     * read again the sensors.
     *
     * @return
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if loader is holding a filter. "
            + "This command doesn't read again the sensors.")
    public boolean isLoaderHoldingFilterAtHandoff() {
        return loader.isHoldingFilter();
    }

    /**
     * @return true if all autochanger hardware is initialized and bridge is
     *         hardwareReady.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if all autochanger CANopen devices are booted, identified and initialized.")
    public boolean isCANDevicesReady() {
        return tcpProxy.allDevicesBooted();
    }

    /**
     * @return if all autochanger CANopen devices are booted, identified and
     *         initialized and homing of the controllers is done.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if all autochanger CANopen devices are booted, identified and "
            + "initialized and homing of the controllers is done.")
    public boolean isHardwareReady() {
        return tcpProxy.allDevicesBooted() && isInitialized();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the autochanger is ready to use or false if not.")
    public boolean isAvailable() {
        return this.available;
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ADVANCED, description = "Flag autochanger as unavailable if false or available if true. "
            + "If available is false, this command should be used to return to 'available' only after FES engineers have been consulted.")
    public void setAvailable(boolean available) {
        this.available = available;
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE,
             description = "Flag the autochanger as NOT READY TO USE. " +
             "This cancels all possible filter changes until further notice and will freeze the FES in its current state, either with a filter ONLINE or no filters at all. " +
             "All filters but one should therefore disappear from the list of available filters. " +
             "This is a protective command that can do no harm but will give some time to FES experts to review the status of the autochanger before deciding to reactivate it.")
    public void setUnavailable() {
        setAvailable(false);
    }

    /**
     * Return true if plutoGateway, trucks, latches and onlineClamps are
     * initialized.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if all autochanger hardware is initialized.")
    public boolean isInitialized() {
        return plutoGateway.isInitialized() && autochangerTrucks.isInitialized() && latches.isInitialized()
                && onlineClamps.isInitialized();
    }

    public boolean isLinearRailMotionAllowed() {
        return this.lpmLinearRail1Status.isOn() && this.lpmLinearRail2Status.isOn();
    }

    public void waitForProtectionSystemUpdate() {
        waitForRailMotionAllowed(timeToUpdateProtectionSystem);
    }

    /**
     * wait until linear rail motion is authorized by local protection system.
     * because there are temporisation in local protection system, there is a delay
     * between sensors authorize us to move and local protection system authorizes.
     *
     * @param timeout after this delay don't wait anymore.
     */
    private void waitForRailMotionAllowed(long timeout) {
        final String methodName = "waitForRailMotionAllowed";
        final String className = Autochanger.class.getName();
        FCSLOG.entering(className, methodName, timeout);
        FcsUtils.waitCondition(() -> isLinearRailMotionAllowed(), () -> plutoGateway.updateValues(),
                className, timeout);

        // we don't trust the first read
        FcsUtils.sleep(10, className);
        if (!isLinearRailMotionAllowed()) {
            FCSLOG.finer("Attention: isLinearRailMotionAllowed glitch 10ms , waiting again");
        }
        FcsUtils.sleep(50, className);
        if (!isLinearRailMotionAllowed()) {
            FCSLOG.finer("Attention: isLinearRailMotionAllowed glitch 60ms, waiting again");
        }
        FcsUtils.waitCondition(() -> isLinearRailMotionAllowed(), () -> plutoGateway.updateValues(),
                "waitForRailMotionAllowed", timeout);
        FCSLOG.exiting(Autochanger.class.getName(), methodName);
    }

    @Override
    public void postStart() {
        final String methodName = "postStart";
        FCSLOG.entering(Autochanger.class.getName(), methodName);
        if (plutoGateway.isBooted()) {
            initializeGateway();
            try {
                // In postStart if a RuntimeException is thrown, fcs cannot starts. To avoid
                // this we have to catch the exception and raise an ALERT of level ALARM.
                updateStateWithSensors();
            } catch (Exception ex) {
                this.raiseAlarm(FcsAlert.HARDWARE_ERROR, name + " couldn't updateStateWithSensors in postStart ", ex);
            }
        } else {
            plutoGateway.raiseAlarmIfMissing();
        }
        this.subsystemIdentifier = plutoGateway.getSubsystemIdentifier();
        plutoGateway.publishSubsystemIdentifier();

        if (tempSensorsDevice1.isBooted()) { // if device1 is booted, it is expected that device2 is as well
            if ((this.subsystemIdentifier == "autochanger1") || (this.subsystemIdentifier == "autochanger2")) {
                // Fix the configuration of the channel 2 of tempSensorsDevice2, see LSSTCCSFCS-567
                tempSensorsDevice2.patchPT100(2);
            }
            checkSenecaTemperatures(); // first occurrence of the periodic task
        }
        FCSLOG.exiting(Autochanger.class.getName(), methodName);
    }

    private void initializeGateway() {
        try {
            this.plutoGateway.initializeAndCheckHardware();
        } catch (FcsHardwareException ex) {
            this.raiseAlarm(FcsAlert.HARDWARE_ERROR, name + " couldn't initialize gateway", plutoGateway.getName(), ex);
        }
    }

    /**
     * For end users, for tests and in engineering mode. This command can be used to
     * recover after a missing hardware during fcs startup. For example when fcs was
     * started before hardware power up. Check all hardware and publish data.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "For end users, for tests and in engineering mode. "
            + "This command can be used to recover after a missing hardware "
            + "during fcs startup. For example, when fcs was started before hardware "
            + "power up. Check all hardware and publish data.", timeout = 5000)
    public void initializeHardware() {
        FCSLOG.info(name + " BEGIN initializeHardware");

        // check that all hardware is booted and identified.
        tcpProxy.bootProcess();

        try {
            this.postStart();
            autochangerTrucks.postStart();
            onlineClamps.postStart();
            latches.postStart();

        } catch (FcsHardwareException ex) {
            this.raiseAlarm(FcsAlert.HARDWARE_ERROR, " couldn't initialize autochanger", ex);
        }
        FCSLOG.info(name + " END initializeHardware");
    }

    /**
     * Check if Local Protection Module allows linear rails motions.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Check if Local Protection Module allows linear rails motion.")
    public void checkLinearRailMotionAllowed() {
        FcsUtils.checkAndWaitCondition(() -> lpmLinearRail1Status.isOn() && lpmLinearRail2Status.isOn(),
                () -> plutoGateway.updateValues(), "checkLinearRailMotionAllowed",
                name + ": linear rails motion NOT allowed by Local Protection Module.");
    }

    /**
     * Check if Local Protection Module allows latches motion.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Check if Local Protection Module allows latches motion.")
    public void checkLatchMotionAllowed() {
        FcsUtils.checkAndWaitCondition(
            () -> lpmLatchesStatus.isOn(),
            () -> plutoGateway.updateValues(),
            "checkLatchMotionAllowed",
            name + ": latches open or close NOT allowed by Local Protection Module."
        );
    }

    /**
     * Check if Local Protection Module allows online clamps motion.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Check if Local Protection Module allows online clamps motion.")
    public void checkOnlineClampMotionAllowed() {
        FcsUtils.checkAndWaitCondition(
            () -> lpmOnlineClampsStatus.isOn(),
            () -> plutoGateway.updateValues(),
            "checkOnlineClampMotionAllowed",
            name + ": Online clamps open or close NOT allowed by Local Protection Module."
        );
    }

    /**
     * Check if latches can be opened. If latches can't be opened because the
     * conditions for opening are not filled, this command throws a
     * RejectedCommandException. If latches are in error, it throws a
     * FcsHardwareException.
     *
     * @param bypassSensorError
     * @throws FcsHardwareException,RejectedCommandException
     */
    public void checkConditionsForOpeningLatches(boolean bypassSensorError) {
        FCSLOG.info(name + " checking pre-conditions for opening latches");

        updateStateWithSensors();

        checkLatchesInitialized();

        if (latches.isInError() && !bypassSensorError) {
            throw new FcsHardwareException(name + ": latches are in ERROR state - can't open latches.");

        } else if (isEmpty()) {
            throw new RejectedCommandException(name + ": no filter in autochanger - can't open latches.");

        } else if (!autochangerTrucks.isAtStandby() && !autochangerTrucks.isAtHandoff()) {
            throw new RejectedCommandException(name + ": autochanger is loaded with a filter but is not "
                    + "at handoff position neither at standby - can't open latches.");

        } else if (autochangerTrucks.isAtStandby()) {
            checkConditionsForOpeningLatchesAtStandby();

        } else if (autochangerTrucks.isAtHandoff()) {
            checkConditionsForOpeningLatchesAtHandoff();
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Check if the latches can be opened.")
    public void checkConditionsForOpeningLatches() {
        checkConditionsForOpeningLatches(false);
    }

    private void checkLatchesInitialized() {
        if (!latches.isInitialized()) {
            throw new RejectedCommandException(
                    name + ": latches are not initialized. " + "Please initialize hardware first.");
        }
    }

    private void checkConditionsForOpeningLatchesAtStandby() {
        /* autochanger holds a filter at STANDBY position */
        if (isCarouselHoldingFilterAtStandby()) {
            FCSLOG.info(() -> name + " carousel is holding filter at STANDBY => latches can be open safely.");

        } else {
            throw new RejectedCommandException(name + ": autochanger is loaded with a filter and is  "
                    + "at STANDBY position but carousel doesn't hold the filter " + "- can't open latches.");
        }
    }

    private void checkConditionsForOpeningLatchesAtHandoff() {
        /* autochanger holds a filter at HANDOFF position */
        if (isLoaderHoldingFilterAtHandoff()) {
            FCSLOG.info(() -> name + " loader is holding filter at HANDOFF => latches can be open safely.");

        } else {
            throw new RejectedCommandException(name + ": autochanger is loaded with a filter and is  "
                    + "at HANDOFF position but loader doesn't hold the filter " + "- can't open latches.");
        }
    }

    /**
     * Check if Autochanger latches can be closed.
     *
     * @param bypassSensorError
     * @throws RejectedCommandException
     */
    public void checkConditionsForClosingLatches(boolean bypassSensorError) {
        FCSLOG.info(() -> name + " checking conditions for closing latches.");

        updateStateWithSensors();
        checkLatchesInitialized();

        if (latches.isInError() && !bypassSensorError) {
            throw new RejectedCommandException(name + ": latches are in ERROR state - can't close latches.");

        } else if (isEmpty()) {
            throw new RejectedCommandException(name + ": no filter in autochanger - can't close latches.");

        } else if (!autochangerTrucks.isAtStandby() && !autochangerTrucks.isAtHandoff()) {
            throw new RejectedCommandException(
                    name + ": autochanger is not " + "at handoff position neither at standby - can't close latches.");
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Check if Autochanger latches can be closed.")
    public void checkConditionsForClosingLatches() {
        checkConditionsForClosingLatches(false);
    }

    /**
     * log if actions on online clamps are allowed, throws an exception otherwise.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Check and log if actions on online clamps are allowed,"
            + "throws an exception otherwise.")
    public void checkConditionsForActioningOnlineClamps() {
        if (!isAtOnline()) {
            throw new RejectedCommandException(
                    name + " actions not allowed on ONLINE clamps " + "when trucks are not at ONLINE.");
        }
        if (isEmpty()) {
            throw new RejectedCommandException(
                    name + " actions not allowed on ONLINE clamps " + "when no filter on trucks.");
        }
        FCSLOG.info(() -> name + " autochangerTrucks are at ONLINE, a filter is there : actions are allowed.");
    }

    /**
     * Check if movement of the autochanger trucks with a filter is permitted
     */
    public void checkConditionsForTrucksMotionWithFilter() {
        FCSLOG.info(name + " Checking pre-conditions for moving the trucks with a filter");
        this.updateStateWithSensors();

        if ( autochangerTrucks.isPositionSensorsInError() ) {
            handleSensorsError();
        }

        if ( !isHoldingFilter() ) {
            throw new RejectedCommandException(name + " This command requires the autochanger to already hold a filter.");
        }

        if ( isAtStandby() || isCarouselHoldingFilterAtStandby() ) {
            throw new RejectedCommandException(name + " This command cannot be executed when the filter is held at STANDBY by the carousel. First use the disengageFilterFromCarousel command.");
        }
    }

    /**
     * This methods checks that filter can be moved by trucks. Returns immediately
     * if autochanger trucks are empty. Throws RejectedCommandException if a filter
     * is in trucks AND held both by autochanger and (loader or carousel).
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws RejectedCommandException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Check if its safe for filters to move autochanger trucks.")
    public void checkFilterSafetyBeforeMotion() {

        String msg = name + " can't move trucks because ";
        this.updateStateWithSensors();

        if (this.isSensorsInError()) {
            handleSensorsError();

        } else if (isEmpty()) {
            FCSLOG.info(() -> name + " trucks are empty - can move");

        } else if (!this.onlineClamps.isOpened()) {
            throw new RejectedCommandException(
                    name + " : a filter " + "is in trucks and ONLINE clamps are NOT OPENED.");

        } else if (isAtStandby()) {
            checkFilterSafetyAtStandby(msg);

        } else if (autochangerTrucks.isAtHandoff()) {
            FCSLOG.info(() -> name + " : trucks are AT HANDOFF ");
            checkFilterSafetyAtHandoff(msg);

        } else if (this.autochangerTrucks.isAtOnline()) {
            FCSLOG.info(() -> name + " : trucks are at ONLINE ");
            checkFilterSafetyAtOnline(msg);

        } else if (!latches.isHoldingFilter() && !carousel.isHoldingFilter()) {
            throw new RejectedCommandException(msg + " neither autochanger neither carousel "
                    + "is holding filter. Close latches or carousel clamps.");
        }
    }

    /**
     * Check if trucks can be moved when loaded with a filter at STANDBY.
     *
     * @param message
     * @throws FcsHardwareException
     */
    private void checkFilterSafetyAtStandby(String message) {
        if (isHoldingFilter() && isCarouselHoldingFilterAtStandby()) {
            throw new RejectedCommandException(message + "both carousel and autochanger are holding filter at STANDBY");

        } else if (!isHoldingFilter() && !isCarouselHoldingFilterAtStandby()) {
            throw new RejectedCommandException(
                    message + "neither carousel nor autochanger are holding filter at STANDBY");

        } else if (!(latches.isClosed() || latches.isOpened())) {
            throw new RejectedCommandException(message + " latches must be opened or closed.");

        } else {
            FCSLOG.info(() -> name + " filter safe at STANDBY - can move");
        }
    }

    /**
     * Check if trucks can be moved when loaded with a filter at HANDOFF.
     *
     * @param message
     * @throws FcsHardwareException
     */
    private void checkFilterSafetyAtHandoff(String message) {
        if (isHoldingFilter() && this.isCarouselHoldingFilterAtStandby()) {
            throw new RejectedCommandException(
                    message + "autochanger is holding filter at HANDOFF but another filter is at STANDBY "
                            + "- can't move trucks");
        }
        if (isHoldingFilter() && isLoaderHoldingFilterAtHandoff()) {
            throw new RejectedCommandException(message + "both loader and autochanger are holding filter at HANDOFF");
        } else if (!isHoldingFilter() && !isLoaderHoldingFilterAtHandoff()) {
            throw new RejectedCommandException(
                    message + "neither loader nor autochanger are holding filter at HANDOFF");
        } else {
            FCSLOG.info(() -> name + " filter safe at HANDOFF - can move");
        }
    }

    /**
     * Check if trucks can be moved when loaded with a filter at ONLINE. It can be
     * move : - if no filter is at STANDBY (otherwise risk of collision) - AND
     * ONLINE clamps are opened, - AND latches are CLOSED.
     *
     * @param message
     * @throws FcsHardwareException
     */
    private void checkFilterSafetyAtOnline(String message) {
        if (isHoldingFilter() && this.isCarouselHoldingFilterAtStandby()) {
            throw new RejectedCommandException(
                    message + "autochanger is holding filter at ONLINE but another filter is at STANDBY "
                            + "- can't move trucks");
        }

        if (isHoldingFilter() && !onlineClamps.isOpened()) {
            throw new RejectedCommandException(
                    message + "onlineClamps have to be opened on filter at ONLINE - can't move trucks.");
        }

        if (!isHoldingFilter() && !onlineClamps.isLocked()) {
            throw new RejectedCommandException(
                    message + "neither latches nor onlineClamps are holding filter at ONLINE");
        } else {
            FCSLOG.info(() -> name + " filter safe at ONLINE - can move");
        }
    }

    /**
     * Test in carousel position at STANDBY is correct to go and grab a filter
     * or store a filter on carousel.
     */
    public void checkCarouselDeltaPosition() {
        if (carousel instanceof Carousel) {
            ((Carousel) carousel).checkDeltaPosition();
        }
    }

    /**
     * Update and publish carousel's socket at STANDBY's data
     * Used in AutochangerTwoTrucks
     * This is needed because carousel object can not be accessed from AutochangerTwoTrucks
     */
    public void carouselUpdateStandbyState() {
        if (carousel instanceof Carousel) {
            ((Carousel) carousel).updateSocketAtStandbyState();
        }
    }

    /**
     * Raise ALERT and throw FcsHardwareException with a detailed message when error
     * is detected in sensors.
     *
     * @throws FcsHardwareException
     */
    private void handleSensorsError() {
        boolean transientError = false;
        String msg = name + " error detected in sensors :";
        if (this.autochangerTrucks.isPositionSensorsInError()) {
            msg += " trucks position sensors";
            transientError = autochangerTrucks.isPositionSensorErrorsTransient();
        }
        if (this.latches.isFilterEngagedInError()) {
            transientError = transientError || isLatchesErrorTransient();
        }
        if (this.latches.isInError()) {
            msg += " latches sensors";
        }
        if (this.onlineClamps.isInError()) {
            msg += " onlineClamps sensors";
        }
        if (transientError) {
            this.raiseWarning(FcsAlert.AC_SENSOR_ERROR, msg + " - can be a transient error. ", name);
        } else {
            this.raiseAlarm(FcsAlert.AC_SENSOR_ERROR, msg);
            throw new FcsHardwareException(msg);
        }
    }

    private boolean isLatchesErrorTransient() {
        return Math.abs(autochangerTrucks.getPosition() - 984000) < 5000;
    }

    /**
     * This method reads all the sensors and updates the trucks, latches and online
     * clamps state.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Update clamp state in reading sensors.")
    @Override
    public synchronized void updateStateWithSensors() {

        long beginTime = System.currentTimeMillis();

        // Check if the system is in simulation mode
        if (!FcsUtils.isSimu()) {
            // Check the time elapsed since the last sync was sent on the bus
            // If this amounts to less that 100ms, use the current values and do not send a sync
            // The 100 ms rate was decided so that it would be useful for the trending and still
            // be higher than the rate at which the system is able to receive consecutive syncs
            // which is ~ 50 ms according to Guillaume Daubard
            if (beginTime - lastUpdateStateWithSensors.get() < 100) {
               return;
            }
        }
        lastUpdateStateWithSensors.set(beginTime);

        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("updateStateWithSensors-ac")) {
            plutoGateway.checkInitialized();
            this.plutoGateway.updateValues();
            /* tcpProxy.updatePDOData() updates all devices values that can be updated by PDO */
            tcpProxy.updatePDOData();
            updateState();
            updateFilterOnTrucksID();
        }
    }

    /**
     * update filterOnTrucksID
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "update filterOnTrucksID.")
    public void updateFilterOnTrucksID() {
        if (!latches.isEmpty()) {
            setFilterOnTrucksID(filterIdentificator.getFilterId());
        } else {
            setFilterOnTrucksID(0);
        }
    }

    private void updateState() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("updateState-ac")) {
            this.filterIdentificator.retrieveFilterId();
            this.autochangerTrucks.updateState();
            this.latches.updateState();
            this.onlineClamps.updateState();
            if (this.isAtStandby() && carouselStoppedAtStandbySensors.isOn()) {
                carousel.updateStateWithSensors();
            }
            main.updateAgentState(getAutochangerInclinationState());
        }
    }

    /**
     * Update FCS state and FCS readiness state and publish on the status bus. Check
     * that Autochanger hardware is ready to be operated and moved. This means that
     * : - all CAN open devices are booted, identified and initialized, -
     * initialization has been done on the controllers.
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT, description = "Force FCS state and FCS readiness state to be ready and publishes on the status bus. TO BE USED ONLY IN EXTREME CASES.")
    public void updateFCSStateToReady() {
        if (this.isCANDevicesReady() && latches.isInitialized() && autochangerTrucks.isInitialized()
                && onlineClamps.isInitialized()) {
            main.updateFCSStateToReady();
        }
    }

    /****************************************************************************
     * Methods from FilterHolder
     ***************************************************************************
     */

    /**
     * Return true if a filter is in trucks and latches are LOCKED.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Return true if a filter is in trucks and latches are CLOSED.")
    @Override
    public boolean isHoldingFilter() {
        return latches.isHoldingFilter();
    }

    @Override
    public boolean isNotHoldingFilter() {
        return latches.isOpened();
    }

    /**
     * Return true if autochanger trucks are at HANDOFF. This command doesn't read
     * again the sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if autochanger trucks are at HANDOFF. "
            + "This command doesn't read again the sensors.")
    @Override
    public boolean isAtHandoff() {
        return autochangerTrucks.isAtHandoff();
    }

    /**
     * Return true if autochanger trucks are at STANDBY. This command doesn't read
     * again the sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if autochanger trucks are at STANDBY. "
            + "This command doesn't read again the sensors.")
    @Override
    public boolean isAtStandby() {
        return autochangerTrucks.isAtStandby();
    }

    /***************************************************************************
     * end of Methods from FilterHolder
     ***************************************************************************
     */

    /**
     * return true if autochanger trucks are at ONLINE.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if autochanger trucks are at ONLINE. "
            + "This command doesn't read again the sensors.")
    @Override
    public boolean isAtOnline() {
        return autochangerTrucks.isAtOnline();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if autochanger trucks position is around approachStandbyPosition. "
            + "This command doesn't read again the sensors.")
    public boolean isAtApproachStandbyPosition() {
        return Math.abs(autochangerTrucks.getPosition() - autochangerTrucks.getApproachStandbyPosition()) < 1000;
    }

    public boolean isAtHandoffForLoader() {
        return autochangerTrucks.isAtHandoffForLoader();
    }

    /****************************************************************************
     * AutochangerTrucks ENGINEERING_ROUTINE ACTION commands
     ***************************************************************************
     */

    /**
     * Move Autochanger trucks to the Handoff position. If Autochanger trucks are
     * already at HANDOFF position, it does nothing.
     *
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Move Autochanger trucks to the Handoff position.")
    public void goToHandOff() {
        autochangerTrucks.goToHandOff();
    }

    /**
     * Move Autochanger trucks to the Handoff position with a better accuracy.
     *
     *
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Move Autochanger trucks to the Handoff "
            + "position to prepare an exchange with the loader. The position accuracy requirement is tighter than for command goToHandoff (see positionRangeAtHandoffForLoader and positionRangeAtHandoff).")
    public void goToHandOffForLoader() {
        autochangerTrucks.goToHandOffForLoader();
    }

    /**
     * Move Autochanger trucks to the Online position. If Autochanger trucks are
     * already at ONLINE position, it does nothing.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Move Autochanger trucks to the Online position.")
    public void goToOnline() {
        autochangerTrucks.goToOnline();
    }

    /**
     * Move Autochanger trucks to the Standby position. If Autochanger trucks are
     * already at STANDBY position, it does nothing.
     *
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Move Autochanger trucks to the Standby position.")
    public void goToStandby() {
        checkCarouselDeltaPosition();
        autochangerTrucks.goToStandby();
    }

    /**
     * For end users at the console.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Move Autochanger trucks to the Approach Standby position. "
            + "Move fast if position > approachStandbyPosition, slow otherwise.")
    public void moveToApproachStandby() {
        checkCarouselDeltaPosition();
        /* trucks are between standby and approachStandby we have to move slowly*/
        if (autochangerTrucks.getPosition() > autochangerTrucks.getApproachStandbyPosition()) {
            autochangerTrucks.moveToApproachStandbyPositionWithLowVelocity();
        } else {
            autochangerTrucks.moveToApproachStandbyPositionWithHighVelocity();
        }
    }

    /**
     * Command used to move a filter ALREADY detached from the carousel to ONLINE and lock it in place
     *
     * @throws FcsHardwareException if autochanger trucks sensor in error
     * @throws RejectedCommandException with pre-conditions are not met
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, autoAck = false, timeout = 20000,
             description = "Move a filter already on autochanger to ONLINE and lock clamps. Cannot start from STANDBY position.")
    public void moveAndClampFilterOnline() {
        /* The goal of this early test is to avoid updating counter and duration if already at destination*/
        if ( isAtOnline() && onlineClamps.isLocked() ) {
            FCSLOG.info(name + " filter already locked ONLINE.");
            return;
        }

        checkConditionsForTrucksMotionWithFilter();

        moveAndClampFilterOnlineCounter.increment();
        final long beginTime = System.currentTimeMillis();
        // The method itself is autotimed
        autochangerTrucks.moveAndClampFilterOnline();
        GeneralAction.MOVE_AND_CLAMP_FILTER_ONLINE.publishDurationTopLevel(subs, System.currentTimeMillis() - beginTime);
    }

    /**
     * Command used to move a filter ALREADY detached from the carousel to ONLINE
     *
     * @throws FcsHardwareException if autochanger trucks sensor in error
     * @throws RejectedCommandException with pre-conditions are not met
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, autoAck = false, timeout = 20000,
             description = "Move a filter already on autochanger to ONLINE but DO NOT lock the clamps. Cannot start from STANDBY position.")
    public void moveFilterOnlineNoClamping() {
        /* The goal of this early test is to avoid updating counter and duration if already at destination*/
        if ( isAtOnline() ) {
            FCSLOG.info(name + " filter already at ONLINE.");
            return;
        }

        checkConditionsForTrucksMotionWithFilter();

        // The method itself is autotimed
        autochangerTrucks.moveFilterOnlineNoClamping();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "unlock and open clamps in doing homing if needed, "
            + "then move slowly to approach online position, then move fast to handoff", autoAck = false, timeout = 20000)
    public void unclampAndMoveFilterToHandoff() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("unclampAndMoveFilterToHandoff")) {
            autochangerTrucks.unclampAndMoveFilterToHandoff();
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "go slowly to approachStandbyPosition ")
    public void moveToApproachStandbyPositionWithLowVelocity() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("moveToApproachStandbyPositionWithLowVelocity")) {
            autochangerTrucks.moveToApproachStandbyPositionWithLowVelocity();
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "go fast to approachStandbyPosition ")
    public void moveToApproachStandbyPositionWithHighVelocity() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("moveToApproachStandbyPositionWithHighVelocity")) {
            autochangerTrucks.moveToApproachStandbyPositionWithHighVelocity();
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "go fast to approachOnlinePosition ")
    public void moveToApproachOnlinePositionWithHighVelocity() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("moveToApproachOnlinePositionWithHighVelocity")) {
            autochangerTrucks.moveToApproachOnlinePositionWithHighVelocity();
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "go slowly to slow motion position ")
    public void moveToSlowMotionPositionWithLowVelocity() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("moveToSlowMotionPositionWithLowVelocity")) {
            autochangerTrucks.moveToSlowMotionPositionWithLowVelocity();
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "go fast to slow motion position ")
    public void moveToSlowMotionPositionWithHighVelocity() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("moveToSlowMotionPositionWithHighVelocity")) {
            autochangerTrucks.moveToSlowMotionPositionWithHighVelocity();
        }
    }


    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "go slowly to approachOnlinePosition ")
    public void moveToApproachOnlinePositionWithLowVelocity() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("moveToApproachOnlinePositionWithLowVelocity")) {
            autochangerTrucks.moveToApproachOnlinePositionWithLowVelocity();
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "align follower and move empty from approachStandby to Handoff")
    public void alignFollowerAndMoveEmptyFromApproachToHandoff() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("alignFollowerAndMoveEmptyFromApproachToHandoff")) {
            autochangerTrucks.alignFollowerAndMoveEmptyFromApproachToHandoff();
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "move slowly to standby position")
    public void moveToStandbyWithLowVelocity() {
        if (carousel instanceof Carousel) {
            ((Carousel) carousel).checkDeltaPosition();
        }
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("moveToStandbyWithLowVelocity")) {
            autochangerTrucks.moveToStandbyWithLowVelocity();
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "move fast to handoff position")
    public void moveToHandoffWithHighVelocity() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("moveToHandoffWithHighVelocity")) {
            autochangerTrucks.moveToHandoffWithHighVelocity();
        }
    }

    /**
     * Move a filter to standby position step by step. A first step with a high
     * speed to the approachStandbyPosition. a second step to standby position with
     * low speed.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "move filter to approachPosition with high speed and move to STANDBY with lowSpeed.", autoAck = false, timeout = 30000)
    public void moveFilterToStandby() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("moveFilterToStandby")) {
            autochangerTrucks.moveFilterToStandby();
        }
    }

    /**
     * Move a filter to standby position plus deltaStandbyPosition step by
     * step.
     *
     * A first step with a high speed to the approachStandbyPosition. A second
     * step to standby position with low speed.
     *
     * @param deltaStandbyPosition
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "move filter to approachPosition with high speed and move to STANDBY + deltaStandbyPosition with lowSpeed.", autoAck = false, timeout = 30000)
    public void moveFilterToStandbyPlusDelta(int deltaStandbyPosition) {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("moveFilterToStandbyPlusDelta")) {
            autochangerTrucks.moveFilterToStandbyPlusDelta(deltaStandbyPosition);
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Move filter to HANDOFF for loader", autoAck = false, timeout = 30000)
    public void moveFilterToHandoff() {
        if ( carousel.isHoldingFilter() ) {
            throw new RejectedCommandException(name + " this method should never be used when the filter is held by the carousel");
        }
        if ( isAtHandoffForLoader() ) {
            return;
        } else if ( isAtHandoff() ) {
            // Go away and come back with precision rather than aligning in place
            autochangerTrucks.moveToApproachOnlinePositionWithHighVelocity();
            autochangerTrucks.slowProfile();
            autochangerTrucks.goToHandOffForLoader();
        } else if ( isAtOnline() ) {
            autochangerTrucks.unclampAndMoveFilterToHandoff();
        } else {
            autochangerTrucks.fastProfile();
            autochangerTrucks.goToHandOffForLoader();
        }
    }

    /**
     * move follower truck to masterPosition. When moving along the rails with follower
     * controller in MASTER_ENCODER mode, at the end of motion there is a small
     * misalignment. This consists in : setting follower Controller mode to
     * PROFILE_POSITION, go to absolute masterPosition pos, and then set
     * follower Controller mode back to MASTER_ENCODER mode.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Align follower controller position to driver controller position.", timeout = 20000)
    public void alignFollower() {
        autochangerTrucks.alignFollowerStrict();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Align follower controller position to driver controller position to prepare interaction with loader (see configuration parameter in category autochanger : deltaPositionForAlignStrictLoader).", timeout = 20000)
    public void alignFollowerForLoader() {
        autochangerTrucks.alignFollowerForLoader();
    }

    /**
     * change speed & acceleration profile to slow
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "change ProfileVelocity parameter to lowSpeed and slowProfileAcceleration and slowProfileDeceleration")
    public void slowTrucksProfile() {
        autochangerTrucks.slowProfile();
    }

    /**
     * change speed and acceleration profile to fast
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "change ProfileVelocity and ProfileAcceleration and ProfileDeceleration parameters to highSpeed")
    public void fastTrucksProfile() {
        autochangerTrucks.fastProfile();
    }

    /**
     * Do homing of both trucks controllers.
     *
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Do homing for both  controllers.")
    public void homingTrucks() {
        autochangerTrucks.homing();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "move to approach position with high velocity then move to STANDBY with low velocity")
    public void dockingAtStandbyPositionWithoutFilter() {
        try (ActionGuard g = autochangerTrucks.new ReleaseAutochangerBrakes()) {
            autochangerTrucks.moveToApproachStandbyPositionWithHighVelocity();
            autochangerTrucks.moveToStandbyWithLowVelocity();
        }
    }

    /****************************************************************************
     * END OF AutochangerTrucks ENGINEERING_ROUTINE commands
     ***************************************************************************
     */

    /****************************************************************************
     * AutochangerLatches ENGINEERING_ROUTINE ACTION commands
     ***************************************************************************
     */

    /**
     * Open latches
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Open latches.")
    public void openLatches() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("openLatches")) {
            latches.open();
        }
    }

    /**
     * Close latches
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Close latches.")
    public void closeLatches() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("closeLatches")) {
            latches.close();
        }
    }

    /****************************************************************************
     * END OF AutochangerLatches ENGINEERING_ROUTINE commands
     ***************************************************************************
     */

    /****************************************************************************
     * AutochangerOnlineClamps ENGINEERING_ROUTINE ACTION commands
     ***************************************************************************
     */

    /**
     * Opens the 3 clamps. initial state = CLOSED final state = OPENED. For final
     * products AC1 and AC2. For prototype see openClampsInCurrentMode.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Opens the 3 online clamps in mode PROFILE_POSITION.", timeout = AutochangerOnlineClamp.TIMEOUT_FOR_OPENING)
    public void openClamps() {
        onlineClamps.openClamps();
    }

    /**
     * close clamps in mode PROFILE_POSITION. for AC1 and AC2
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = " close clamps in mode PROFILE_POSITION for AC1 and AC2", timeout = AutochangerOnlineClamp.TIMEOUT_FOR_CLOSING)
    public void closeClamps() {
        onlineClamps.closeClamps();
    }

    /**
     * Locks clamps : closed with a strong pressure (high current). The clamps have
     * to be CLOSED. At the end of this action, the clamps are CLOSED but a strong
     * pressure to hold safely the clamps.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Locks clamps : closed with a strong pressure (high current).", timeout = AutochangerOnlineClamp.TIMEOUT_FOR_CLOSING)
    public void lockClamps() {
        onlineClamps.lockClamps();
    }

    /**
     * Unlocks clamps : slows down current sent to controller in order to decrease
     * pressure on the clamps. The clamps have to be LOCKED. At the end of this
     * action, the clamps are CLOSED with a small pressure of the clamp hardware on
     * the filter frame.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "unlocks clamps : decreases current in controller to decrease pressure.", timeout = AutochangerOnlineClamp.TIMEOUT_FOR_OPENING)
    public void unlockClamps() {
        onlineClamps.unlockClamps();
    }

    /**
     * do homing of the 3 clamps
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "do homing of the 3 ONLINE clamps : open in CURRENT mode and homing of controller", timeout = 6000)
    public void homingClamps() {
        onlineClamps.homing();
    }

    /****************************************************************************
     * END OF AutochangerOnlineClamps ENGINEERING_ROUTINE commands
     ***************************************************************************
     */

    /**
     * Return true if there is no filter in the autochanger. This command doesn't
     * read again the sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if there is no filter in the autochanger. "
            + "This command doesn't read again the sensors.")
    public boolean isEmpty() {
        return latches.isEmpty();
    }

    /**
     * Moves empty to standby position and close latches on filter at standby on
     * carousel.
     *
     * @throws RejectedCommandException if latches are not OPEN.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, alias = "grabFilter", description = "Move autochanger trucks to STANDBY position, close latches and stay at STANDBY")
    public void grabFilterAtStandby() {
        updateState();
        if (!latches.isOpened()) {
            throw new RejectedCommandException(name + " latches must be open to execute grabFilterAtStandby");
        }
        grabFilterAtStandbyCounter.increment();
        final long beginTime = System.currentTimeMillis();
        try (ActionGuard g = autochangerTrucks.new ReleaseAutochangerBrakes()) {
            FCSLOG.info(name + " === About to move empty to standby position ===");
            autochangerTrucks.moveToApproachStandbyPositionWithHighVelocity();
            // Tolerance is low for the latches, needs strict alignment
            autochangerTrucks.alignFollowerStrict();
            autochangerTrucks.moveEmptyToStandbyWithLowVelocity();
        }
        FCSLOG.info(name + " ===> filter on autochanger after moveToStandbyEmptyWithLowVelocity =" + filterOnTrucksID
                + " should be != 0");
        updateFilterOnTrucksID();
        latches.close();

        GeneralAction.GRAB_FILTER_AT_STANDBY.publishDurationTopLevel(subs, System.currentTimeMillis() - beginTime);

        FCSLOG.info(name + ": filter " + filterOnTrucksID + " is now on autochanger");
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "open latches then go to approachStandbyPosition with low speed "
            + "and move to HANDOFF with highSpeed.", timeout = 20000, autoAck = false)
    public void moveEmptyFromStandbyToHandoff() {
        moveEmptyFromStandbyToHandoff(false);
    }

    public void moveEmptyFromStandbyToHandoff(boolean withPrecision) {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("moveEmptyFromStandbyToHandoff")) {
            if (!this.isCarouselHoldingFilterAtStandby()) {
                throw new RejectedCommandException(name + "%s : can't open latches if carousel is not holding it.");
            }
            if (latches.isClosed()) {
                latches.open();
            }
            waitForProtectionSystemUpdate();
            try (ActionGuard g = autochangerTrucks.new ReleaseAutochangerBrakes()) {
                autochangerTrucks.moveToApproachStandbyPositionWithLowVelocity();
                autochangerTrucks.alignFollowerAndMoveEmptyFromApproachToHandoff(withPrecision);
            }
            updateFilterOnTrucksID();
        }
    }

    /**
     * set a new value for the acquisition frequency of linear rail controllers
     * current.
     *
     * @param rate
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "set a new value for the acquisition "
            + "frequency of linear rail controllers current in milliseconds.")
    public void setFastAcqRateLinearRails(int rate) {
        configurationService.change("acTruckXminus-monitorCurrent", "taskPeriodMillis", rate);
        configurationService.change("acTruckXplus-monitorCurrent", "taskPeriodMillis", rate);
        configurationService.saveChangesForCategories("timers:fastRails");
    }

    /**
     * set a new value for the acquisition frequency of ONLINE clamps controllers
     * current.
     *
     * @param rate
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "set a new value for the acquisition "
            + "frequency of ONLINE clamps controllers current, in milliseconds.")
    public void setFastAcqRateOnlineClamps(int rate) {
        configurationService.change("onlineClampXminus-monitorCurrent", "taskPeriodMillis", rate);
        configurationService.change("onlineClampXplus-monitorCurrent", "taskPeriodMillis", rate);
        configurationService.change("onlineClampYminus-monitorCurrent", "taskPeriodMillis", rate);
        configurationService.saveChangesForCategories("timers:fast");
    }

    /**
     * speed up online clamps current monitoring for the 3 clamps
     */
    @Deprecated
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "speed up online clamps current monitoring for the 3 clamps")
    public void increaseCurrentMonitoring() {
        configurationService.loadCategories("timers:fast");
    }

    /**
     * slow down online clamps current monitoring for the 3 clamps
     */
    @Deprecated
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "slow down online clamps current monitoring for the 3 clamps")
    public void decreaseCurrentMonitoring() {
        configurationService.loadCategories("timers:slow");
    }

    /**
     * increase current monitoring for the 2 trucks
     */
    public void increaseLinearRailsCurrentMonitoring() {
        configurationService.loadCategories("timers:fastRails");
    }

    /**
     * slow down online clamps current monitoring for the 3 clamps
     */
    public void decreaseLinearRailsCurrentMonitoring() {
        configurationService.loadCategories("timers:slowRails");
    }

    /**
     * to be used in setFilter
     */
    public void lockFilterAtOnline() {
        if (!this.isAtOnline()) {
            throw new FcsHardwareException(name + "is not at ONLINE");
        }
        if (this.isEmpty()) {
            throw new FcsHardwareException(name + " must be loaded with a filter");
        }
        onlineClamps.lockFilterAtOnline();
    }

    /**
     * Read temperatures on both seneca ( 2x 4 values)
     * @return an array of 8 doubles representing temperatures.
     *         The first 4 values are from Seneca 1, and the last 4 are from Seneca 2.
     */
    private double[] readSenecaTemperatures() {
        double[] temp = new double[8];
        //
        for (int i= 0; i < 4; i++) {
            // Seneca 1
            temp[i] = tempSensorsDevice1.readChannel(i+1);
            // Seneca 2
            temp[i+4] = tempSensorsDevice2.readChannel(i+1);
            // Use a little sleep here to avoid sending too many SDO requests simultaneously
            FcsUtils.sleep(50, name);
        }

        return temp;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Print the eight temperatures of the autochanger")
    public String printTemperatures() {
        checkSenecaTemperatures();
        String fmt = "%-21s = %6.1fºC\n";
        StringBuilder sb = new StringBuilder();
        sb.append(String.format(fmt, "LinearRailMotorXplus", temperatures[0]));
        sb.append(String.format(fmt, "LinearRailMotorXminus", temperatures[1]));
        sb.append(String.format(fmt, "ClampMotorXplus", temperatures[2]));
        sb.append(String.format(fmt, "ClampMotorYminus", temperatures[3]));
        sb.append(String.format(fmt, "ClampMotorXminus", temperatures[4]));
        sb.append(String.format(fmt, "FrontBox", temperatures[5]));
        sb.append(String.format(fmt, "RearBox", temperatures[6]));
        sb.append(String.format(fmt, "CellXminus", temperatures[7]));
        return sb.toString();
    }

    /**
     * Check autochanger temperatures and publish the values
     */
    public void checkSenecaTemperatures() {
        try {
            this.temperatures = readSenecaTemperatures();
        } catch (Exception ex) {
            raiseWarning(
                FcsAlert.SDO_ERROR, "Autochanger temperature controller unreachable on the Canbus. " +
                "When powered on for a long time, the autochanger temperature sensor controllers (Seneca) may " +
                "shutdown the communication on the canbus, making them unable to receive a SDO request. " +
                "Since these modules are powered from the safety power (same as PLC) they cannot be reset and the solution " +
                "is to send an NMT Reset Communication signal on the Canbus so it starts responding again.",
                ex
            );
            tcpProxy.resetCommunication(tempSensorsDevice1.getNodeID());
            tcpProxy.resetCommunication(tempSensorsDevice2.getNodeID());
        }
    }

    private void publishSenecaTemperatures() {
        subs.publishSubsystemDataOnStatusBus(new KeyValueData(name + "/temperatures", createStatusDataPublishedBySeneca()));
    }

    /**
     * Create Autochanger temperatures data to be published at regular intervals
     */
    public StatusDataPublishedByAutochangerSeneca createStatusDataPublishedBySeneca() {
        checkSenecaTemperatures();
        return new StatusDataPublishedByAutochangerSeneca(this.temperatures);
    }
}
