package org.lsst.ccs.subsystems.fcs.ui;

import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByAutochangerTwoLatches;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByCarouselSocket;
import org.lsst.ccs.subsystems.fcs.ui.commons.Tools;

import javax.swing.*;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.DefaultTableModel;
import javax.swing.table.TableRowSorter;
import java.awt.*;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * Displays the list of filters available in the carousel and the autochanger.
 * For each filter, its name, ID, weight, family and location is displayed.
 */
public class FiltersViewPanel extends JPanel {
    private static final long serialVersionUID = 2654131875576434802L;
    // private final InterfaceGeneralGUI subs;
    private final String panelTitle = "Filters Overview";
    private JTable allFiltersTable; // list all filters
    private JTable onlineTable;
    private ConfigurationInfo configInfo;

    public FiltersViewPanel() {
        initComponents();
    }

    /**
     * Update allFiltersTable
     *
     * @param configInfo
     * @param filterNames
     */
    public void initializeGui(ConfigurationInfo configInfo, List<String> filterNames) {
        this.configInfo = configInfo;
        SwingUtilities.invokeLater(new GuiInitialization(filterNames));
    }

    public void updateFromStatusData(KeyValueData kvd) {
        SwingUtilities.invokeLater(new UpdateInstalledFilter(kvd));
    }

    @Override
    public String toString() {
        return panelTitle;
    }

    /**
     * Delete filter table rows.
     * Used when GUI is disconnecting to reset filter panel to empty values.
     */
    public void resetPanel() {
        Tools.resetTable(allFiltersTable);
        Tools.resetTable(onlineTable);
    }

    private void initComponents() {
        setBorder(Tools.getGeneralPanelTitle(panelTitle));

        initOnlineFiltersTable();
        JScrollPane scrollPane1 = new JScrollPane(onlineTable);
        scrollPane1.setBorder(BorderFactory.createTitledBorder("Filters installed in the camera"));
        scrollPane1.setPreferredSize(new Dimension(600, 165)); // Adjust dimensions as needed

        initAllFiltersTable();
        JScrollPane scrollPane2 = new JScrollPane(allFiltersTable);
        scrollPane2.setBorder(BorderFactory.createTitledBorder("View of all filters"));
        scrollPane2.setPreferredSize(new Dimension(600, 300));

        JPanel panel = new JPanel(new BorderLayout());
        panel.add(scrollPane1, BorderLayout.NORTH); // Add first table to the top

        add(panel);
        panel.add(scrollPane2, BorderLayout.SOUTH); // Add second table to fill remaining space
    }

    private void initOnlineFiltersTable() {
        // 5 sockets + autochanger = 6
        onlineTable = new JTable();
        onlineTable.setModel(new DefaultTableModel(
            new String[] {"Slot", "Name", "ID", "Obs. name", "Weight (kg)", "DeltaAutochangerStandbyPosition", "Available"}, 6) {
                private static final long serialVersionUID = 1L;
                private final Class<?>[] types = new Class[]{
                    String.class, String.class, Integer.class, String.class, Double.class, Integer.class, Boolean.class
                };
                private final boolean[] canEdit = new boolean[]{
                    false, false, false, false, false, false, false,
                };

                @Override
                public Class<?> getColumnClass(int columnIndex) {
                    return types[columnIndex];
                }

                @Override
                public boolean isCellEditable(int rowIndex, int columnIndex) {
                    return canEdit[columnIndex];
            }
        });

        // Initialise row names
        onlineTable.setValueAt("Autochanger", 0, 0);
        for (int i = 1; i < 6; i++) {
            onlineTable.setValueAt("Socket " + i, i, 0);
        }

        // Right-align text column
        DefaultTableCellRenderer rightRenderer = new DefaultTableCellRenderer();
        rightRenderer.setHorizontalAlignment(JLabel.RIGHT);
        onlineTable.getColumnModel().getColumn(1).setCellRenderer(rightRenderer);
        onlineTable.getColumnModel().getColumn(3).setCellRenderer(rightRenderer);

    }

    private void initAllFiltersTable() {
        allFiltersTable = new JTable();
        allFiltersTable.setModel(new DefaultTableModel(
            new String[] {"Filter family", "Name", "ID", "Obs. name", "Weight (kg)", "DeltaAutochangerStandbyPosition"}, 0) {
                private static final long serialVersionUID = 1L;
                private final Class<?>[] types = new Class[]{
                    String.class, String.class, Integer.class, String.class, Double.class, Integer.class
                };
                private final boolean[] canEdit = new boolean[]{
                    false, false, false, false, false, false,
                };

                @Override
                public Class<?> getColumnClass(int columnIndex) {
                    return types[columnIndex];
                }

                @Override
                public boolean isCellEditable(int rowIndex, int columnIndex) {
                    return canEdit[columnIndex];
            }
        });
        // Right-align text columns
        DefaultTableCellRenderer rightRenderer = new DefaultTableCellRenderer();
        rightRenderer.setHorizontalAlignment(JLabel.RIGHT);
        allFiltersTable.getColumnModel().getColumn(1).setCellRenderer(rightRenderer);
        allFiltersTable.getColumnModel().getColumn(3).setCellRenderer(rightRenderer);
        // Resize filter family column
        allFiltersTable.getColumnModel().getColumn(0).setPreferredWidth(200);
        // Add sorter on the filter family name column
        TableRowSorter<DefaultTableModel> sorter = new TableRowSorter<>((DefaultTableModel) allFiltersTable.getModel());
        sorter.setSortKeys(Collections.singletonList(new RowSorter.SortKey(0, SortOrder.ASCENDING)));
        allFiltersTable.setRowSorter(sorter);
    }

    /**
     * Initialization of the GUI from data stored in Configuration.
     */
    private class GuiInitialization implements Runnable {

        private final List<String> filterNamesList;

        public GuiInitialization(List<String> filterNames) {
            this.filterNamesList = filterNames;
        }

        public void updateAllFilters(DefaultTableModel model) {
            for (String filterName : filterNamesList) {
                Map<String, String> config = configInfo.getCurrentValuesFor("filterManager/" + filterName);
                Object[] filterInformation = {
                    FcsEnumerations.FilterFamily.valueOf(config.get("family")).getFamilyName(),
                    filterName,
                    Integer.parseInt(config.get("filterID")),
                    filterName + "_" + Integer.parseInt(config.get("filterID")),
                    Double.parseDouble(config.get("weight")),
                    Integer.parseInt(config.get("deltaAutochangerStandbyPosition"))
                };
                model.addRow(filterInformation);
            }
        }

        @Override
        public void run() {
            DefaultTableModel model = (DefaultTableModel) allFiltersTable.getModel();
            updateAllFilters(model);
            TableRowSorter<?> sorter = (TableRowSorter<?>) allFiltersTable.getRowSorter();
            sorter.sort();
        }
    }

    /**
     * A Runnable class to update the Filter List when a Filter publishes on the
     * Status bus its new location.
     */
    private class UpdateInstalledFilter implements Runnable {
        Object data;

        UpdateInstalledFilter(KeyValueData kvd) {
            this.data = kvd.getValue();
        }

        private Double getWeight(String filterName) {
            Map<String, String> config = configInfo.getCurrentValuesFor("filterManager/" + filterName);
            return Double.parseDouble(config.get("weight"));
        }

        private void updateAutochangerInfo(DefaultTableModel model, StatusDataPublishedByAutochangerTwoLatches s) {
            String filterName = s.getFilterName();
            int filterId = s.getFilterId();

            model.setValueAt(filterId > 0 ? filterName : "-", 0, 1);
            model.setValueAt(filterId > 0 ? s.getFilterId() : "-", 0, 2);
            model.setValueAt(filterId > 0 ? s.getFilterObservatoryName() : "-", 0, 3);
            model.setValueAt(filterId > 0 ? getWeight(filterName) : Double.NaN, 0, 4);
            model.setValueAt(filterId > 0 ? s.getFilterDeltaAutochangerStandbyPosition() : "-", 0, 5);
            model.setValueAt(s.isAvailable(), 0, 6);
        }

        private void updateCarouselInfo(DefaultTableModel model, StatusDataPublishedByCarouselSocket s) {
            int socketId = s.getSocketID();
            int filterId = s.getFilterID();
            String filterName = s.getFilterName();
            String filterObservatoryName = s.getFilterObservatoryName();

            model.setValueAt(filterId > 0 ? filterName : "-", socketId, 1);
            model.setValueAt(filterId > 0 ? filterId : "-", socketId, 2);
            model.setValueAt(filterId > 0 ? filterObservatoryName : "-", socketId, 3);
            model.setValueAt(filterId > 0 ? getWeight(filterName) : Double.NaN, socketId, 4);
            model.setValueAt(filterId > 0 ? s.getFilterDeltaAutochangerStandbyPosition() : "-", socketId, 5);
            model.setValueAt(s.isAvailable(), socketId, 6);
        }

        @Override
        public void run() {
            DefaultTableModel model = (DefaultTableModel) onlineTable.getModel();
            if (data instanceof StatusDataPublishedByAutochangerTwoLatches) {
                updateAutochangerInfo(model, (StatusDataPublishedByAutochangerTwoLatches) data);
            } else if (data instanceof StatusDataPublishedByCarouselSocket) {
                updateCarouselInfo(model, (StatusDataPublishedByCarouselSocket) data);
            } else {
                // We should never end there
            }
        }
    }
}
