
package org.lsst.ccs.subsystems.fcs.loader.ui;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.util.Map;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByLoaderCarrier;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.greenColor;
import org.lsst.ccs.subsystems.fcs.ui.commons.DigitalSwitch;
import org.lsst.ccs.subsystems.fcs.ui.commons.Tools;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.BIG_FONT;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.ZERO_VALUE;
import static org.lsst.ccs.subsystems.fcs.ui.commons.Tools.insets_std;

/**
 * A panel for the loader carrier.
 * The carrier holds a filter and can go from STORAGE position to HANDOFF
 * position.
 * HANDOFF position is  where the autochanger can take the filter to
 * put it in the camera.
 * @author virieux
 */
public class LoaderCarrierPanel extends JPanel {
    private static final long serialVersionUID = 2323057246140272874L;
    private final String carrierName;

    private InterfaceLoaderGUI subs;

    private final DigitalSwitch storageDigitalSwitch = new DigitalSwitch();
    private final DigitalSwitch inTravelDigitalSwitch = new DigitalSwitch();
    private final DigitalSwitch handoffDigitalSwitch = new DigitalSwitch();
    private final DigitalSwitch engagedDigitalSwitch = new DigitalSwitch();
    private final DigitalSwitch inErrorDigitalSwitch = new DigitalSwitch();
    private final JLabel cajLabel = new JLabel("Carrier position");
    private final JLabel inErrorjLabel = new JLabel("In Error");
    private final JLabel handoffSensorsErrorjLabel = new JLabel();
    private final JLabel engagedSensorsErrorjLabel = new JLabel();
    private final JLabel storageSensorsErrorjLabel = new JLabel();
    //parameters Panel
    private final JLabel actualPositionjLabel = new JLabel("0");
    private final JLabel speedjLabel = new JLabel("0");
    private final JLabel profileVelocityjLabel = new JLabel("0");
    private final JLabel currentjLabel = new JLabel("0");
    private final JProgressBar positionProgressBar = new JProgressBar(1, 0, 500);
    private int handoffPosition;
    //general informations panel
    private final JLabel storagePositionjLabel = new JLabel(Integer.toString(0));
    private final JLabel engagedPositionjLabel = new JLabel(Integer.toString(0));
    private final JLabel handoffPositionjLabel = new JLabel(Integer.toString(0));
    private final JLabel speedMaxjLabel = new JLabel(Integer.toString(0));
    private final JLabel speedLowjLabel = new JLabel(Integer.toString(0));
    private final JLabel speedHighjLabel = new JLabel(Integer.toString(0));
    private final JLabel currentMaxjLabel = new JLabel(Integer.toString(0));

    //control panel
    private final JButton updateStateWithSensorsButton = new JButton("updateStateWithSensors");
    private final JButton updatePositionButton = new JButton("updatePosition");
    private final JButton goToStorageButton = new JButton("goToStorage");
    private final JButton goToHandoffButton = new JButton("goToHandOff");
    private final JButton goToEngagedButton = new JButton("goToEngaged");
    private final JButton quickStopButton = new JButton("QuickStop");
    private final JButton abortButton = new JButton("Abort");

    //Fault Panel
    private JPanel faultPanel;
    private final DigitalSwitch faultSwitch = new DigitalSwitch();
    // End of variables declaration

    /**
     * Creates new form LoaderCarrierPanel and init all the components.
     * @param carrierName
     */
    public LoaderCarrierPanel(String carrierName) {
        this.carrierName = carrierName;
        initComponents();
    }

    public void setSubsystem(InterfaceLoaderGUI subs) {
        this.subs = subs;
    }

    public void initializeGui(ConfigurationInfo config) {
        SwingUtilities.invokeLater(new GuiInitialization(config));
    }

    public void updateCarrier(StatusDataPublishedByLoaderCarrier status) {
        SwingUtilities.invokeLater(new UpdateCarrier(status));
    }


    /**
     * A Runnable to initialize this Panel from Configuration data.
     *
     */
    private class GuiInitialization implements Runnable {

        private final ConfigurationInfo configInfo;

        public GuiInitialization(ConfigurationInfo configInfo) {
            this.configInfo = configInfo;
        }

        @Override
        public void run() {
            //Retrieve the configuration data for my name.
            Map<String, String> configForCarrier = configInfo.getCurrentValuesFor(carrierName);

            //General Informations Panel
            handoffPosition = Tools.toInt(configForCarrier,"handoffPosition");
            int engagedPosition = Tools.toInt(configForCarrier,"engagedPosition");
            int storagePosition = Tools.toInt(configForCarrier,"storagePosition");

            storagePositionjLabel.setText(Integer.toString(storagePosition));
            handoffPositionjLabel.setText(Integer.toString(handoffPosition));
            engagedPositionjLabel.setText(Integer.toString(engagedPosition));
            speedMaxjLabel.setText(configForCarrier.get("maxSpeed"));
            speedLowjLabel.setText(configForCarrier.get("lowSpeed"));
            speedHighjLabel.setText(configForCarrier.get("highSpeed"));
            currentMaxjLabel.setText(configForCarrier.get("maxCurrent"));

            /**
             * position progress bar. In a JProgressBar the minimum is at the
             * bottom and the maximum is at top but in loader carrier storage
             * position = 0 is at top and handoffPosition (~1000000) is at
             * bottom so to give the illusion of the motion of loader carrier,
             * the position have to be their negative on the progressBar.
             */
            positionProgressBar.setMinimum(-handoffPosition);
            positionProgressBar.setMaximum(-storagePosition);
            positionProgressBar.setValue(0);

            //Fault Panel
            faultSwitch.setColor(Color.GRAY);

            //Initialization of the timeouts for buttons commands.
            subs.setLoGoToHandoffTimeout(Tools.toInt(configForCarrier,"timeoutForGoingToHandOff"));
            subs.setLoGoToStorageTimeout(Tools.toInt(configForCarrier,"timeoutForGoingToStorage"));
            subs.setLoGoToEngagedTimeout(Tools.toInt(configForCarrier,"timeoutForGoingToEngaged"));

        }
    }

    private class UpdateCarrier implements Runnable {

        private final StatusDataPublishedByLoaderCarrier s;

        public UpdateCarrier(StatusDataPublishedByLoaderCarrier status) {
            this.s = status;
        }

        @Override
        public void run() {
            //TODO put the values of the limit switch sensors.
            storageDigitalSwitch.updateColor(s.isStorageInError(), s.isAtStorage());
            handoffDigitalSwitch.updateColor(s.isHandoffInError(), s.isAtHandoff());
            engagedDigitalSwitch.updateColor(s.isEngagedInError(), s.isAtEngaged());
            inTravelDigitalSwitch.setColor((!s.isAtHandoff()
                    && !s.isAtStorage() && !s.isAtEngaged()) ? Color.YELLOW : Color.gray);

            inErrorDigitalSwitch.setColor((s.isHandoffInError()
                    || s.isStorageInError()) ? Color.RED : greenColor);
            inErrorjLabel.setText((s.isHandoffInError()
                    || s.isStorageInError()) ? "IN ERROR" : "NO ERROR");
            handoffSensorsErrorjLabel.setText(s.isHandoffInError() ? "<HTML>Handoff sensors<BR>in error</HTML>" : null);
            engagedSensorsErrorjLabel.setText(s.isEngagedInError() ? "<HTML>Engaged sensors<BR>in error</HTML>" : null);
            storageSensorsErrorjLabel.setText(s.isStorageInError() ? "<HTML>Storage sensors<BR>in error</HTML>" : null);

            /* - : to give the illusion of the real motion of carrier*/
            positionProgressBar.setValue(- s.getPosition());
            positionProgressBar.setForeground(Color.PINK);

            actualPositionjLabel.setText(Integer.toString(s.getPosition()));
            speedjLabel.setText(Integer.toString(s.getSpeed()));
            currentjLabel.setText(Integer.toString(s.getCurrent()));
            profileVelocityjLabel.setText(Long.toString(s.getProfileVelocity()));

            //controller errors - fault panel
            faultSwitch.setColor(s.isControllerInError() ? Color.RED : greenColor);

        }
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    private void initComponents() {
        /*
         **  Add graphical components to the main panel
         */

        updateStateWithSensorsButton.addActionListener((ActionEvent evt) -> {
            updateStateWithSensorsActionPerformed(evt);
        });

        updatePositionButton.addActionListener((ActionEvent evt) -> {
            updatePositionActionPerformed(evt);
        });

        goToHandoffButton.addActionListener((ActionEvent evt) -> {
            goToHandoffActionPerformed(evt);
        });

        goToEngagedButton.addActionListener((ActionEvent evt) -> {
            goToEngagedActionPerformed(evt);
        });

        goToStorageButton.addActionListener((ActionEvent evt) -> {
            goToStorageActionPerformed(evt);
        });

        quickStopButton.addActionListener((ActionEvent evt) -> {
            quickStopActionPerformed(evt);
        });

        abortButton.addActionListener((ActionEvent evt) -> {
            abortActionPerformed(evt);
        });

        //sensor in Error Panel
        //creation Error Panel
        JPanel errorjPanel = new JPanel();
        errorjPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc1 = new GridBagConstraints();
        gbc1.insets = insets_std;
        inErrorDigitalSwitch.setSize(60);
        gbc1.gridx = 0;
        gbc1.gridy = 0;
        errorjPanel.add(inErrorDigitalSwitch, gbc1);
        gbc1.gridy++;
        errorjPanel.add(inErrorjLabel, gbc1);
        gbc1.gridy++;
        errorjPanel.add(storageSensorsErrorjLabel, gbc1);
        gbc1.gridy++;
        errorjPanel.add(handoffSensorsErrorjLabel, gbc1);

        /**
         ******************************************************************************
         ** Parameters Panel
         * *****************************************************************************
         */
        JPanel paramPanel = new JPanel();
        //paramPanel.setPreferredSize(new Dimension(200,50));
        paramPanel.setBorder(javax.swing.BorderFactory.createLineBorder(Color.BLACK));
        paramPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc2 = new GridBagConstraints();
        gbc2.insets = insets_std;
        //first colomn
        gbc2.gridx = 0;
        gbc2.gridy = 0;
        gbc2.anchor = GridBagConstraints.LINE_START;
        paramPanel.add(new JLabel("Actual Position: "), gbc2);
        gbc2.gridy++;
        paramPanel.add(new JLabel("Speed (rpm): "), gbc2);
        gbc2.gridy++;
        paramPanel.add(new JLabel("Current (mA): "), gbc2);
        gbc2.gridy++;
        paramPanel.add(new JLabel("ProfileVelocity: "), gbc2);
        //second colomn
        gbc2.gridx++;
        gbc2.gridy = 0;
        gbc2.anchor = GridBagConstraints.LINE_END;
        paramPanel.add(actualPositionjLabel, gbc2);
        gbc2.gridy++;
        paramPanel.add(speedjLabel, gbc2);
        gbc2.gridy++;
        paramPanel.add(currentjLabel, gbc2);
        gbc2.gridy++;
        paramPanel.add(profileVelocityjLabel, gbc2);
        /**
         ******************************************************************************
         ** End of parameters Panel
         * *****************************************************************************
         */

        /**
         ******************************************************************************
         ** General Informations Panel
         * *****************************************************************************
         */
        JPanel generalInfoPanel = new JPanel();
        generalInfoPanel.setBorder(BorderFactory.createTitledBorder(
                BorderFactory.createLineBorder(Color.BLUE,3),"Configurable parameters"));
        generalInfoPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc3 = new GridBagConstraints();
        gbc3.insets = insets_std;
        //first colomn
        gbc3.gridx = 0;
        gbc3.gridy = 0;
        gbc3.anchor = GridBagConstraints.LINE_START;
        generalInfoPanel.add(new JLabel("Storage Position: "), gbc3);
        gbc3.gridy++;
        generalInfoPanel.add(new JLabel("Engaged Position: "), gbc3);
        gbc3.gridy++;
        generalInfoPanel.add(new JLabel("Hand-Off Position: "), gbc3);
        gbc3.gridy++;
        generalInfoPanel.add(new JLabel("Speed max (rpm): "), gbc3);
        gbc3.gridy++;
        generalInfoPanel.add(new JLabel("Speed low (rpm): "), gbc3);
        gbc3.gridy++;
        generalInfoPanel.add(new JLabel("Speed high (rpm): "), gbc3);
        gbc3.gridy++;
        generalInfoPanel.add(new JLabel("Current max (mA): "), gbc3);
        //second colomn
        gbc3.gridx++;
        gbc3.gridy = 0;
        gbc3.anchor = GridBagConstraints.LINE_END;
        generalInfoPanel.add(storagePositionjLabel, gbc3);
        gbc3.gridy++;
        generalInfoPanel.add(engagedPositionjLabel, gbc3);
        gbc3.gridy++;
        generalInfoPanel.add(handoffPositionjLabel, gbc3);
        gbc3.gridy++;
        generalInfoPanel.add(speedMaxjLabel, gbc3);
        gbc3.gridy++;
        generalInfoPanel.add(speedLowjLabel, gbc3);
        gbc3.gridy++;
        generalInfoPanel.add(speedHighjLabel, gbc3);
        gbc3.gridy++;
        generalInfoPanel.add(currentMaxjLabel, gbc3);
         /**
         ******************************************************************************
         ** End of General Information Panel
         * *****************************************************************************
         */
        /**
         ******************************************************************************
         ** Fault Panel
         * *****************************************************************************
         */

        faultPanel = new JPanel();
        faultPanel.setBorder(BorderFactory.createTitledBorder("Controller"));
        faultPanel.setLayout(new GridBagLayout());
        faultPanel.setPreferredSize(new Dimension(100, 100));

        faultSwitch.setSize(50);

        GridBagConstraints gbcf = new GridBagConstraints();
        gbcf.insets = insets_std;
        gbcf.gridx = 0;
        gbcf.gridy = 0;
        gbcf.anchor = GridBagConstraints.CENTER;
        faultPanel.add(faultSwitch, gbcf);
        /**
         ******************************************************************************
         ** end of Fault Panel
         * *****************************************************************************
         */
        /**
         ******************************************************************************
         ** Control Panel
         * *****************************************************************************
         */
        JPanel controlPanel = new JPanel();
        controlPanel.setLayout(new java.awt.GridBagLayout());
        GridBagConstraints gbc4 = new java.awt.GridBagConstraints();
        gbc4.anchor = GridBagConstraints.FIRST_LINE_START;
        gbc4.fill = GridBagConstraints.HORIZONTAL;
        gbc4.insets = insets_std;
        gbc4.gridx = 0;
        gbc4.gridy = 0;
        controlPanel.add(updateStateWithSensorsButton, gbc4);
        gbc4.gridy++;
        controlPanel.add(updatePositionButton, gbc4);
        gbc4.gridy++;
        controlPanel.add(goToHandoffButton, gbc4);
        gbc4.gridy++;
        controlPanel.add(goToEngagedButton, gbc4);
        gbc4.gridy++;
        controlPanel.add(goToStorageButton, gbc4);
        gbc4.gridy++;
        controlPanel.add(abortButton, gbc4);
        /**
         ******************************************************************************
         ** End of Control Panel
         ******************************************************************************
         */

        /**
         *******************************************************************************
         * Whole panel
         ********************************************************************************
         */
        setBorder(javax.swing.BorderFactory.createTitledBorder("LOADER CARRIER"));
        setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = insets_std;
        gbc.anchor = GridBagConstraints.NORTH;
        gbc.gridx = 2; //(2,0)
        gbc.gridy = 0;
        cajLabel.setFont(BIG_FONT);
        add(cajLabel, gbc);

        //sensors in the 2 first columns
        gbc.gridx = 0;
        gbc.gridy = 1; //(0,1)
        gbc.anchor = GridBagConstraints.NORTHEAST;
        add(new JLabel("Storage"), gbc);
        gbc.gridx++;//(1,1)
        add(storageDigitalSwitch, gbc);

        gbc.anchor = GridBagConstraints.EAST;
        gbc.gridx = 0;
        gbc.gridy++; //(0,2)
        add(new JLabel("In Travel"), gbc);
        gbc.gridx++; //(1,2)
        add(inTravelDigitalSwitch, gbc);

        gbc.gridx = 0;
        gbc.gridy++; //(0,3)
        add(new JLabel("Engaged"), gbc);
        gbc.gridx++; //(1,3)
        add(engagedDigitalSwitch, gbc);

        gbc.gridx = 0;
        gbc.gridy++; //(0,4)
        add(new JLabel("Hand-Off"), gbc);
        gbc.gridx++; //(1,4)
        add(handoffDigitalSwitch, gbc);

        //carrier position progress bar in the column 2
        gbc.gridx = 2;
        gbc.gridy = 1;//(2,1)
        gbc.anchor = GridBagConstraints.CENTER;
        gbc.gridheight = 4;
        gbc.fill = GridBagConstraints.VERTICAL;
        positionProgressBar.setBorderPainted(true);
        positionProgressBar.setBackground(greenColor);
        positionProgressBar.setStringPainted(true);
        add(positionProgressBar, gbc);
        gbc.gridheight = 1;

        //parameters panel on the grid
        gbc.gridx = 3;
        gbc.gridy = 1;
        add(paramPanel, gbc);

        //error panel on the gri
        gbc.gridx = 3;//(3,3)
        gbc.gridy = 3;
        gbc.anchor = GridBagConstraints.CENTER;
        add(errorjPanel, gbc);

        //General Informations panel on the grid
        gbc.gridx = 3;//(3,4)
        gbc.gridy = 4;
        gbc.gridheight = 2;
        add(generalInfoPanel, gbc);

        //Fault panel on the grid
        gbc.gridx = 4;//(4,1)
        gbc.gridy = 1;
        add(faultPanel, gbc);

        //Control panel on the grid
        gbc.gridx = 4;//(4,4)
        gbc.gridy = 4;
        gbc.gridheight = 2;
        add(controlPanel, gbc);

        setDefaultValues();
    }

    private void setDefaultValues() {
        storageDigitalSwitch.setColor(Color.gray);
        inTravelDigitalSwitch.setColor(Color.gray);
        handoffDigitalSwitch.setColor(Color.gray);
        engagedDigitalSwitch.setColor(Color.gray);
        inErrorDigitalSwitch.setColor(Color.gray);
        actualPositionjLabel.setText(ZERO_VALUE);
        speedjLabel.setText(ZERO_VALUE);
        profileVelocityjLabel.setText(ZERO_VALUE);
        currentjLabel.setText(ZERO_VALUE);
        faultSwitch.setColor(Color.GRAY);
    }

    /**
     * Reset components to default values.
     */
    public void resetPanel() {
        setDefaultValues();
    }

    private void updateStateWithSensorsActionPerformed(java.awt.event.ActionEvent evt) {
        subs.updateStateWithSensors();
    }

    private void updatePositionActionPerformed(java.awt.event.ActionEvent evt) {
        subs.loUpdateCarrierPosition();
    }

    private void goToHandoffActionPerformed(java.awt.event.ActionEvent evt) {
        subs.loGoToHandOff();
    }

    private void goToEngagedActionPerformed(java.awt.event.ActionEvent evt) {
        subs.loGoToEngaged();
    }

    private void goToStorageActionPerformed(java.awt.event.ActionEvent evt) {
        subs.loGoToStorage();
    }

    private void quickStopActionPerformed(java.awt.event.ActionEvent evt) {
        subs.stopMyDestinationSubsystem();
    }

    private void abortActionPerformed(java.awt.event.ActionEvent evt) {
        subs.abortAction();
    }

    @Override
    public String toString() {
        return "Carrier Panel";
    }

    /**
     * Just for tests.
     * @param argv
     */
    public static void main(String[] argv) {
        LoaderCarrierPanel d = new LoaderCarrierPanel("my_carrier");
        JFrame frame = new JFrame("Loader Carrier Panel");
        frame.setContentPane(d);
        frame.pack();
        frame.setVisible(true);
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    }
}
