package org.lsst.ccs.subsystems.fcs.autochanger.ui;

import java.util.List;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import static org.lsst.ccs.subsystems.fcs.FCSCst.AC_PLUTO_GATEWAY_NAME;
import static org.lsst.ccs.subsystems.fcs.FCSCst.AC_TRUCKS_NAME;
import static org.lsst.ccs.subsystems.fcs.FCSCst.CHANGER_TCPPROXY_NAME;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByAutoChanger;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByAutoChangerTrucks;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByAutochangerLatch;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByAutochangerOnlineClamp;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByAutochangerThreeClamps;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByAutochangerTruck;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByAutochangerTwoLatches;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByEPOSController;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByHardware;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByPlutoGateway;
import org.lsst.ccs.subsystems.fcs.ui.commons.EPOSControllerPanel;
import org.lsst.ccs.subsystems.fcs.ui.commons.GatherPanel;
import org.lsst.ccs.subsystems.fcs.ui.commons.PlutoGatewayPanel;

/**
 * This panel gathers all the panels needed for the autochanger GUI.
 * Except the subsystemPanel which is handled by the GeneralPanel.
 * It takes care of the initialization of the panels and the update of the panels.
 *
 * @author virieux
 */
public class AutochangerGatherPanel extends GatherPanel {
    private static final long serialVersionUID = -975949078768889254L;
    


    protected InterfaceAutochangerGUI subs;

    //CANopen devices panels
    protected EPOSControllerPanel linearRailMasterControllerPanel;
    protected EPOSControllerPanel linearRailSlaveControllerPanel;
    protected EPOSControllerPanel latchXminusControllerPanel;
    protected EPOSControllerPanel latchXplusControllerPanel;
    protected EPOSControllerPanel onlineClampXminusControllerPanel;
    protected EPOSControllerPanel onlineClampXplusControllerPanel;
    protected EPOSControllerPanel onlineClampYminusControllerPanel;
    protected PlutoGatewayPanel plutoGatewayPanel;
    
    private EPOSControllerPanel[] controllerPanelsList; 

    //Exchanger hardware panels
    protected AutochangerThreeOnlineClampsPanel onlineClampsPanel;
    
    
    private AutochangerTwoTrucksPanel acTrucksPanel;
    private AutochangerTwoLatchesPanel acLatchesPanel = new AutochangerTwoLatchesPanel();
    private AutochangerThreeOnlineClampsPanel acClampsPanel = new AutochangerThreeOnlineClampsPanel();
    private final AutochangerGeneralViewPanel acGeneralViewPanel = new AutochangerGeneralViewPanel();

    public AutochangerGatherPanel(String tcpProxyName) {
        super(tcpProxyName);
        initPanels();
        setName("AutochangerGatherPanel");
    }

    public AutochangerGatherPanel(InterfaceAutochangerGUI subs, String tcpProxyName) {
        super(tcpProxyName);
        initPanels();
        this.subs = subs;
        this.acGeneralViewPanel.setSubsystem(subs);
        this.acClampsPanel.setSubsystem(subs);
        this.acLatchesPanel.setSubsystem(subs);
        this.acTrucksPanel.setSubsystem(subs);
        this.linearRailMasterControllerPanel.setSubsystem(subs);
        this.linearRailSlaveControllerPanel.setSubsystem(subs);
        this.latchXminusControllerPanel.setSubsystem(subs);
        this.latchXplusControllerPanel.setSubsystem(subs);
        this.onlineClampXminusControllerPanel.setSubsystem(subs);
        this.onlineClampXplusControllerPanel.setSubsystem(subs);
        this.onlineClampYminusControllerPanel.setSubsystem(subs);
        setName("AutochangerGatherPanel");
    }



    public AutochangerTwoTrucksPanel getAcTrucksPanel() {
        return acTrucksPanel;
    }

    public AutochangerTwoLatchesPanel getAcLatchesPanel() {
        return acLatchesPanel;
    }

    public AutochangerThreeOnlineClampsPanel getAcClampsPanel() {
        return acClampsPanel;
    }

    public AutochangerGeneralViewPanel getAcGeneralViewPanel() {
        return acGeneralViewPanel;
    }

    public EPOSControllerPanel getLinearRailMasterControllerPanel() {
        return linearRailMasterControllerPanel;
    }

    public EPOSControllerPanel getLinearRailSlaveControllerPanel() {
        return linearRailSlaveControllerPanel;
    }

    public EPOSControllerPanel getLatchXminusControllerPanel() {
        return latchXminusControllerPanel;
    }

    public EPOSControllerPanel getLatchXplusControllerPanel() {
        return latchXplusControllerPanel;
    }

    public EPOSControllerPanel getOnlineClampXminusControllerPanel() {
        return onlineClampXminusControllerPanel;
    }

    public EPOSControllerPanel getOnlineClampXplusControllerPanel() {
        return onlineClampXplusControllerPanel;
    }

    public EPOSControllerPanel getOnlineClampYminusControllerPanel() {
        return onlineClampYminusControllerPanel;
    }

    public PlutoGatewayPanel getPlutoGatewayPanel() {
        return plutoGatewayPanel;
    }

    @Override
    public void initializeGUIWithConfigInfo(ConfigurationInfo configInfo,
            List<String> hardwareNames) {
        super.initializeGUIWithConfigInfo(configInfo, hardwareNames);
        SwingUtilities.invokeLater(new GuiInitialization(configInfo));
    }
    
    /**
     * Update GUI when a StatusConfigurationInfo message is received on the status bus. 
     * @param configInfo
     */
    @Override
    public void updateFromConfigurationInfo(ConfigurationInfo configInfo) {
        SwingUtilities.invokeLater(new GuiInitialization(configInfo));
    }
    
    @Override
    public void updateFromStatusData(KeyValueData data) {
        
        if (CHANGER_TCPPROXY_NAME.equals(data.getKey())) {
            updateCanOpenHardwarePanel((int)data.getValue());
            
        } else if (data.getValue() instanceof StatusDataPublishedByEPOSController) {
            updateCanOpenHardwarePanel(data.getKey(),(StatusDataPublishedByHardware)data.getValue()); 
            updateController(data.getKey(),(StatusDataPublishedByEPOSController) data.getValue());
            
        } else if (data.getValue() instanceof StatusDataPublishedByPlutoGateway) {
            updateCanOpenHardwarePanel(data.getKey(),(StatusDataPublishedByHardware)data.getValue()); 
            this.plutoGatewayPanel.updatePlutoGateway(data.getKey(),(StatusDataPublishedByPlutoGateway)data.getValue());
            
        } else if (data.getValue() instanceof StatusDataPublishedByHardware) { 
            updateCanOpenHardwarePanel(data.getKey(),(StatusDataPublishedByHardware) data.getValue());    
            
        } else if (data.getValue() instanceof StatusDataPublishedByAutoChanger) {
            updateAutochanger((StatusDataPublishedByAutoChanger) data.getValue());
            
        } else if (data.getValue() instanceof StatusDataPublishedByAutochangerTruck) {
            updateTruck(data.getKey(),(StatusDataPublishedByAutochangerTruck) data.getValue());
            
        } else if (data.getValue() instanceof StatusDataPublishedByAutoChangerTrucks) {
            updateTrucks((StatusDataPublishedByAutoChangerTrucks) data.getValue());

        } else if (data.getValue() instanceof StatusDataPublishedByAutochangerLatch) {
            updateLatch(data.getKey(), (StatusDataPublishedByAutochangerLatch) data.getValue());
            
        } else if (data.getValue() instanceof StatusDataPublishedByAutochangerTwoLatches) {
            updateLatches((StatusDataPublishedByAutochangerTwoLatches) data.getValue());

        } else if (data.getValue() instanceof StatusDataPublishedByAutochangerOnlineClamp) {
            updateClamp(data.getKey(), (StatusDataPublishedByAutochangerOnlineClamp) data.getValue());
            
        } else if (data.getValue() instanceof StatusDataPublishedByAutochangerThreeClamps) {
            updateClamps((StatusDataPublishedByAutochangerThreeClamps) data.getValue());
        }
    }

    private void updateController(String name, StatusDataPublishedByEPOSController controllerStatus) {
        for (EPOSControllerPanel ctlPanel: controllerPanelsList) {
            ctlPanel.updateController(name,controllerStatus);
        }
    }

    /**
     * Initialization of the GUI for data stored in tcpConfiguration. *
     */
    private class GuiInitialization implements Runnable {

        private final ConfigurationInfo configInfo;

        public GuiInitialization(ConfigurationInfo configInfo) {
            this.configInfo = configInfo;
        }

        @Override
        public void run() {
            //** Autochanger General View Panel **/
            acGeneralViewPanel.initializeGui(configInfo);

            /**
             * Autochanger Clamps Panel *
             */
            acClampsPanel.initializeGui(configInfo);

            /**
             * Autochanger Latches Panel *
             */
            acLatchesPanel.initializeGui(configInfo);

            /**
             * Autochanger Trucks Panel *
             */
            
            acTrucksPanel.initializeGui(configInfo);

            /**
             * Linear Rail Master Controller Panel *
             */
            linearRailMasterControllerPanel.initializeGui(configInfo);

            /**
             * Linear Rail Slave Controller Panel*
             */
            linearRailSlaveControllerPanel.initializeGui(configInfo);

            /**
             * Latch Xminus Controller Panel*
             */
            latchXminusControllerPanel.initializeGui(configInfo);

            /**
             * Latch Xplus Controller Panel*
             */
            latchXplusControllerPanel.initializeGui(configInfo);

            /**
             * Online Clamp Xminus Controller Panel*
             */
            onlineClampXminusControllerPanel.initializeGui(configInfo);

            /**
             * Online Clamp Xplus Controller Panel*
             */
            onlineClampXplusControllerPanel.initializeGui(configInfo);

            /**
             * Online Clamp Z Controller Panel*
             */
            onlineClampYminusControllerPanel.initializeGui(configInfo);

            /**
             * PlutoGateway Panel*
             */
            plutoGatewayPanel.initializeGui(configInfo);

        }
    }
    
    protected void updateClamp(String clampName, StatusDataPublishedByAutochangerOnlineClamp s) {
        this.acClampsPanel.updateClamp(clampName, s);
    }

    protected void updateClamps(StatusDataPublishedByAutochangerThreeClamps s) {
        this.acClampsPanel.updateClamps(s);
        this.acGeneralViewPanel.updateClamps(s);
    }

    protected void updateAutochanger(StatusDataPublishedByAutoChanger s) {
        this.acGeneralViewPanel.updateAutochanger(s);
    }

    protected void updateLatch(String latchName, StatusDataPublishedByAutochangerLatch s) {
        this.acLatchesPanel.updateLatch(latchName,s);
    }

    protected void updateLatches(StatusDataPublishedByAutochangerTwoLatches s) {
        this.acLatchesPanel.updateLatches(s);
        this.acGeneralViewPanel.updateLatches(s);
    }
    
    protected void updateTruck(String truckName,StatusDataPublishedByAutochangerTruck s) {
        this.acTrucksPanel.updateTruck(truckName,s);
    }
    
    protected void updateTrucks(StatusDataPublishedByAutoChangerTrucks s) {
        this.acTrucksPanel.updateTrucks(s);
        this.acGeneralViewPanel.updateTrucks(s);
    }

    private void initPanels() {
        //controllers Panels
        linearRailMasterControllerPanel = new EPOSControllerPanel("linearRailMasterController");
        linearRailSlaveControllerPanel = new EPOSControllerPanel("linearRailSlaveController");
        latchXminusControllerPanel = new EPOSControllerPanel("latchXminusController");
        latchXplusControllerPanel = new EPOSControllerPanel("latchXplusController");
        onlineClampXminusControllerPanel = new EPOSControllerPanel("onlineClampXminusController");
        onlineClampXplusControllerPanel = new EPOSControllerPanel("onlineClampXplusController");
        onlineClampYminusControllerPanel = new EPOSControllerPanel("onlineClampYminusController");
        controllerPanelsList = new EPOSControllerPanel[] {linearRailMasterControllerPanel, linearRailSlaveControllerPanel, 
            latchXminusControllerPanel,latchXplusControllerPanel, 
            onlineClampXminusControllerPanel, onlineClampXplusControllerPanel, onlineClampYminusControllerPanel};
        //plutoGatewayPanel
        plutoGatewayPanel = new PlutoGatewayPanel(AC_PLUTO_GATEWAY_NAME);

        acClampsPanel = new AutochangerThreeOnlineClampsPanel();
        acLatchesPanel = new AutochangerTwoLatchesPanel();
        acTrucksPanel = new AutochangerTwoTrucksPanel(AC_TRUCKS_NAME);
    }
    
    @Override
    public void resetPanel() {
        FCSLOG.info(getName() + " resetting panels");
        super.resetPanel();
        linearRailMasterControllerPanel.resetPanel();
        linearRailSlaveControllerPanel.resetPanel();
        latchXminusControllerPanel.resetPanel();
        latchXplusControllerPanel.resetPanel();
        onlineClampXminusControllerPanel.resetPanel();
        onlineClampXplusControllerPanel.resetPanel();
        onlineClampYminusControllerPanel.resetPanel();
        plutoGatewayPanel.resetPanel();
        acGeneralViewPanel.resetPanel();
        acLatchesPanel.resetPanel();
        acTrucksPanel.resetPanel();
        acClampsPanel.resetPanel();
    }

}
