package org.lsst.ccs.subsystem.demo.gui.configurationview;

import java.awt.Color;
import java.awt.Component;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.swing.JTable;
import javax.swing.SwingConstants;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.DefaultTableModel;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;

/**
 *
 * @author LSST CCS Team
 */
public class ConfigurationParameterTable extends JTable {
    private static final long serialVersionUID = 3525133163059126232L;

    // maps the parameter name to the row number
    private final Map<String, Integer> indexMap = new HashMap<>();
    
    private List<ConfigurationParameterInfo> data;
    
    public ConfigurationParameterTable() {
        super (new ConfigurationParameterTableModel());
        getColumnModel().getColumn(0).setPreferredWidth(300);
        getColumnModel().getColumn(0).setMaxWidth(600);
        setTableHeader(null);
    }
    
    public void initialize(List<ConfigurationParameterInfo> parmList) {
        Collections.sort(parmList, (c1,c2) -> c1.getParameterName().compareTo(c2.getParameterName()));
        data = parmList;
        DefaultTableModel model = (DefaultTableModel)getModel();
        for (ConfigurationParameterInfo parmInfo : parmList){
            model.addRow(new Object[]{parmInfo.getParameterName(), parmInfo.getCurrentValue()});
            indexMap.put(parmInfo.getParameterName(), getRowCount()-1);
        }
        setDefaultRenderer(String.class, new ValueCellRenderer());
        setPreferredScrollableViewportSize(getPreferredSize());
    }
    
    public void updateValues(List<ConfigurationParameterInfo> delta) {
        for (ConfigurationParameterInfo parmInfo : delta) {
            int index = indexMap.get(parmInfo.getParameterName());
            setValueAt(parmInfo.getCurrentValue(), index, 1);
        }
    }
    
    public String getValue(String parmName) {
        return (String)getValueAt(indexMap.get(parmName), 1);
    }
    
    static class ConfigurationParameterTableModel extends DefaultTableModel {
        private static final long serialVersionUID = 7019866943751097254L;
        
        ConfigurationParameterTableModel() {
            super(new Object[]{"name", "value"}, 0);
        }
        
        @Override
        public Class getColumnClass(int col) {
            return String.class;
        }
        
        @Override
        public boolean isCellEditable(int row, int column) {
            return column > 0 ;
        }
    }
    
    class ValueCellRenderer extends DefaultTableCellRenderer {
        private static final long serialVersionUID = -6165053430970716475L;
        
        ValueCellRenderer() {
            setHorizontalAlignment(SwingConstants.LEFT);
        }
        
        @Override
        public Component getTableCellRendererComponent(JTable table, Object value, 
                boolean isSelected, boolean hasFocus, int row, int column) {
            if (value == null) {
                return super.getTableCellRendererComponent(table, "", false, false, row, column);
            }
            Component c = super.getTableCellRendererComponent(table, value, false, hasFocus, row, column);
            if (column == 1 && data.get(row).isDirty()) {
                c.setForeground(Color.blue);
            }
            else {
                c.setForeground(Color.black);
            }
//            ((JLabel)c).setHorizontalAlignment(SwingConstants.RIGHT);

            return c;
        }
        
    }
    
}
