package org.lsst.ccs.subsystem.demo.gui;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Container;
import java.awt.FlowLayout;
import java.awt.Font;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.Future;
import javax.swing.BorderFactory;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import javax.swing.SwingWorker;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.messaging.ConcurrentMessagingUtils;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Generic view of the parameters set of DemoConfigurableSubsystem.
 * This Panel has no knowledge of the structure of the worker subsystem.
 * @author The LSST CCS Team
 */
public class DemoConfigurableSubsystemParameterGenericGUI extends JPanel {
    
    private final Map<String, ComponentPanel> componentPanelMap = new TreeMap<>();

    private final JPanel componentsPanel;
    private final JPanel buttonPanel = new JPanel(new FlowLayout(FlowLayout.CENTER));
    private final JButton submitAllButton = new JButton("Submit All");
    private final JButton commitButton = new JButton("Commit All");
    private final ConcurrentMessagingUtils cmu;
    private final String destination;
    
    public DemoConfigurableSubsystemParameterGenericGUI(ConcurrentMessagingUtils cmu, String dest){
        super(new BorderLayout());
        this.cmu = cmu;
        this.destination = dest;
        // panel gathering the panels for each component
        componentsPanel = new JPanel();
        componentsPanel.setLayout(new BoxLayout(componentsPanel, BoxLayout.Y_AXIS));
        add(new JScrollPane(componentsPanel), BorderLayout.CENTER);

        buttonPanel.add(submitAllButton);
        add(buttonPanel, BorderLayout.SOUTH);
        
        submitAllButton.addActionListener(e -> {
            for (ComponentPanel cp : componentPanelMap.values()) {
                cp.submitChanges();
            }
        });
        
        commitButton.addActionListener(e -> new CommandWorker(this, new CommandRequest(destination, "commitBulkChange")).execute());
        buttonPanel.add(commitButton);
    }

    public void initializeWithConfigInfo(ConfigurationInfo configInfo) {
        // grouping components by the component they belong to
        Map<String, List<ConfigurationParameterInfo>> mapByComponentName = 
                new TreeMap<String, List<ConfigurationParameterInfo>>(
                        ConfigurationInfo.getParameterInfoGroupByComponent(configInfo.getAllParameterInfo()));
        
        for (Map.Entry<String, List<ConfigurationParameterInfo>> entry : mapByComponentName.entrySet()){
            ComponentPanel componentPanel = componentPanelMap.get(entry.getKey());
            if (componentPanel == null){
                componentPanel = new ComponentPanel(entry.getKey());
                componentPanelMap.put(entry.getKey(), componentPanel);
                componentsPanel.add(componentPanel);
            }
            componentPanel.initializeWithConfigInfo(entry.getValue());
        }
    }

    public void updateWithConfigInfo(List<ConfigurationParameterInfo> latestChanges) {
         // grouping components by the component they belong to
        Map<String, List<ConfigurationParameterInfo>> mapByComponentName = 
                ConfigurationInfo.getParameterInfoGroupByComponent(latestChanges);
        
        for(Map.Entry<String, List<ConfigurationParameterInfo>> entry : mapByComponentName.entrySet()){
            String componentName = entry.getKey();
            componentPanelMap.get(componentName).updateWithConfigInfo(entry.getValue());
        }
    }

    void reset() {
        SwingUtilities.invokeLater(new Runnable() {
            @Override
            public void run() {
                componentsPanel.removeAll();
                componentPanelMap.clear();
                revalidate();
                repaint();
            }
        });
    }
    
    private class ParameterWidgets extends JPanel {
        
        private final Font myFont = new Font("Helvetica", Font.PLAIN, 12);
        private final Font changeFont = new Font("Helvetica", Font.BOLD, 12);
        
        private final String parameterName;
        private final JTextField textField;
        private boolean dirty;
        private final String componentName;

        private ParameterWidgets(ConfigurationParameterInfo parmInfo, String componentName){
            this(parmInfo.getParameterName(), parmInfo.getCurrentValue(), componentName);
            SwingUtilities.invokeLater(()->updateParameterWidget(parmInfo));
        }
        private ParameterWidgets(String parameterName, String currentValue, String componentName){
            super(new GridBagLayout() );
            this.parameterName = parameterName;
            this.componentName = componentName;
            
            GridBagConstraints gbc = new GridBagConstraints(0, 0, 1, 1, 0, 0, GridBagConstraints.NORTHWEST, GridBagConstraints.NONE, new Insets(0, 0, 0, 0), 10, 2);
            gbc.gridx = 0; gbc.gridy = 0; gbc.weightx = 0;
            add(new JLabel(componentName+"/"+parameterName), gbc);
            gbc.gridx = 1; 
            textField = new JTextField(currentValue);
            textField.setFont(myFont);
            add(textField, gbc);
            
            textField.addActionListener(e -> changeParameterValue());
        }
        
        private void changeParameterValue() {
            new CommandWorker(this, new CommandRequest(destination, "change", componentName,parameterName, textField.getText())).execute();
        }
        
        public final void updateParameterWidget(ConfigurationParameterInfo parmInfo) {
            if ( parmInfo.isDirty() ) {
                textField.setFont(changeFont);            
                textField.setForeground(Color.blue);
            } else {
                textField.setFont(myFont);            
                textField.setForeground(Color.black);                
            }
            
            if ( ! parmInfo.getCurrentValue().equals(textField.getText()) ) {
                textField.setText(parmInfo.getCurrentValue());
            }
            
        }
        
        public String getParameterName() {
            return parameterName;
        }

        public JTextField getTextField() {
            return textField;
        }

        public boolean isDirty() {
            return dirty;
        }        
    }
    
    private class ComponentPanel extends JPanel {
        private final String componentName;
        private final Map<String, ParameterWidgets> parametersMap = new TreeMap<>();
        private final JPanel parametersPanel = new JPanel(new GridBagLayout());
        private final JButton submitChangesButton = new JButton("submit changes for component");
        
        private ComponentPanel(String componentName){
            super(new BorderLayout());
            setBorder(BorderFactory.createTitledBorder(BorderFactory.createLineBorder(Color.BLACK), componentName));
            this.componentName = componentName;
            add(parametersPanel, BorderLayout.CENTER);
            JPanel buttonPanel = new JPanel(new FlowLayout(FlowLayout.LEADING));
            buttonPanel.add(submitChangesButton);
            add(buttonPanel, BorderLayout.SOUTH);
        }

        private void initializeWithConfigInfo(List<ConfigurationParameterInfo> parmList) {
            for (ConfigurationParameterInfo parmInfo : parmList){
                parametersMap.put(parmInfo.getParameterName(), 
                        new ParameterWidgets(parmInfo, componentName));
            }
            
            SwingUtilities.invokeLater(new Runnable() {
                @Override
                public void run() {
                    int parmCount = 0;
                    GridBagConstraints gbc = new GridBagConstraints(0, 0, 1, 1, 0, 0, GridBagConstraints.NORTHWEST, GridBagConstraints.NONE, new Insets(0, 0, 0, 0), 10, 2);
                    for(ParameterWidgets parmWidgets : parametersMap.values()){
                        gbc.gridx = 0; gbc.gridy = parmCount; gbc.weightx = 0;
                        parametersPanel.add(parmWidgets, gbc);
                        parmCount +=1;
                    }
                }
            });
            // activate buttons
            submitChangesButton.addActionListener(e -> submitChanges());
            
        }
        
        private void updateWithConfigInfo(List<ConfigurationParameterInfo> recentChanges) {
            SwingUtilities.invokeLater(new Runnable(){
                @Override
                public void run() {
                    for (ConfigurationParameterInfo parmInfo : recentChanges){
                        parametersMap.get(parmInfo.getParameterName()).updateParameterWidget(parmInfo);
                    }
                }
            });
        }
        
        private void submitChanges() {
            Map<String, Object> res = new HashMap<>();
            for (ParameterWidgets parmWidgets : parametersMap.values()){
                String text = parmWidgets.getTextField().getText();
                if (!text.isEmpty()){
                    res.put(parmWidgets.getParameterName(), text);
                }
            }
            new CommandWorker(this,
                    new CommandRequest(destination, "submitChanges", componentName, res)
            ).execute();
        }
        
    }
    
    /**
     * A SwingWorker that executes a command and displays the result in a 
     * simple dialog.
     */
    private class CommandWorker extends SwingWorker {

        private final Container parent;
        private final CommandRequest commandRequest;
        
        private CommandWorker(Container parent, CommandRequest commandRequest){
            this.parent = parent;
            this.commandRequest = commandRequest;
        }
        
        @Override
        protected Object doInBackground() throws Exception {
            Future<Object> future = cmu.sendAsynchronousCommand(commandRequest);
            return future.get();
        }
        
        protected void done(){
            try {
                Object res = get();
                String textRes = "";
                if (res == null) textRes = "OK : VOID";
                else if (res instanceof Exception) {
                    StringWriter stackTraceWriter = new StringWriter();
                    ((Throwable)res).printStackTrace(new PrintWriter(stackTraceWriter));
                    textRes = stackTraceWriter.toString();
                }
                else textRes = res.toString();
                if ( res instanceof Exception ) {
                    JOptionPane.showMessageDialog(parent, textRes);
                }
            } catch (Exception ex) {
                Logger.getLogger("org.lsst.ccs.subsystem.demo.gui").error(ex);
            }
        }
        
    }
}
