package org.lsst.ccs.subsystem.common.switches;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.utilities.conv.InputConversionEngine;

/**
 * A class to define custom levels for turning switches on/off as a final ConfigurationParameter.
 * To use this class annotate a field for this class as a @ConfigurationParameter
 * and define the switches custom levels as a List of strings like: [switchName]-[On|Off]:[customLevel]
 * 
 * For example, this is a valid configuration
 * 
 * [HV-On:3, water-Off:2, gas-Off:3, something-On:5]
 *
 * Validate the custom levels in the desired command by using the preconditions
 * of the CommandHelper class. For example:
 * 
 * <pre><code>
 * 
 * @ConfigurationParameter(isFinal = true)
 * SwitchesOnOffCustomLevels switchesCustomLevels = SwitchesOnOffCustomLevels();
 * 
 * {@literal @}Command(type = Command.CommandType.ACTION, description = "Turn SwitchOn", level=Command.NORMAL, autoAck=false)
 * public void setSwitchOn(String switchName, boolean isOn) {
 *     agent.helper()
 *       .precondition(switchesCustomLevels.isCommandRequestCompatibleWithCustomLevel(agent,switchName,isOn),
 *              "Insufficient lock level for turning the switch %s. Required custom level is: %d",
 *               isOn?"On":"Off",
 *               switchesCustomLevels.getCustomLevelForSwitch(switchName,isOn))
 *       .action(()-> { 
 *            //CODE TO TURN THE SWITCH ON/OFF
 *        });
 *      }
 * </code></pre>
 * 
 * 
 * @author The LSST CCS Team
 */
public class SwitchesOnOffCustomLevels {
    
    private final List<SwitchCustomLevel> customLevels = new ArrayList();
    private final Map<String,SwitchCustomLevel> customLevelsMap = new HashMap<>();
    
    /**
     * Constructor to be used by InputConversionEngine to build and
     * object from the input configuration parameter.
     * 
     * @param stringRepresentation 
     */
    public SwitchesOnOffCustomLevels(String stringRepresentation) {        
        List<String> customLevelsStrings = new ArrayList();
        customLevelsStrings.addAll((List)InputConversionEngine.convertArgToType(stringRepresentation, List.class));
        for ( String str : customLevelsStrings ) {            
            SwitchCustomLevel scl = (SwitchCustomLevel)InputConversionEngine.convertArgToType(str, SwitchCustomLevel.class);
            addSwitchCustomLevel(scl);
        }
    }    
    
    /**
     * Default constructor.
     */
    public SwitchesOnOffCustomLevels() {
        
    }

    /**
     * Check if the lock level embedded in the incoming command request is 
     * compatible with the specified custom level for the action to be
     * performed on the selected switch.
     * 
     * If no custom level has been defined true is returned.
     * 
     * @param agent      The top level Agent
     * @param switchName The name of the switch 
     * @param isOn       The operation being performed on the switch. true for On, false for Off.
     * @return           true if the operation can proceed. false otherwise.
     * 
     */
    public boolean isCommandRequestCompatibleWithCustomLevel(Agent agent, String switchName, boolean isOn) {
        SwitchCustomLevel scl = getCustomLevelForSwitchPosition(switchName, isOn);
        if ( scl == null ) {
            return true;
        }
        if ( agent.getCommandForThisThread() != null ) {
            CommandRequest commandRequest = agent.getCommandForThisThread().getCommandRequest();
            return commandRequest.getLevel() >= scl.customLevel;
        } else {
            //There is no command request associated with this method invocation.
            //This means that this method has been invoked programmatically, like
            //in an emergency command response. We then accept the request.
            return true;
        }
    }
    
    
    @Override
    public String toString() {
        return customLevels.toString();
    }

    @Override
    public boolean equals(Object obj) {
        if ( obj instanceof SwitchesOnOffCustomLevels in ) {
            return in.customLevels.equals(this.customLevels);
        }
        return false;
    }
    
    

    
    SwitchCustomLevel getCustomLevelForSwitchPosition(String switchName, boolean isOn) {
        return customLevelsMap.get(getKeyForSwitch(switchName, isOn));
    }

    /**
     * Get the custom command level for the specified switch and operation.
     * If no custom level is defined, -999 is returned.
     * 
     * @param switchName The name of the switch.
     * @param isOn       The operation to be performed: true for On, false for Off
     * @return           The specified custom level or -999.
     */
    public int getCustomLevelForSwitch(String switchName, boolean isOn) {
        SwitchCustomLevel scl = getCustomLevelForSwitchPosition(switchName, isOn);
        return scl != null ? scl.customLevel : -999;
    }
    
    final void addSwitchCustomLevel(SwitchCustomLevel scl) {
        customLevels.add( scl );
        addSwitchCustomLevelToMapAndValidate(scl);
    }

    final void addSwitchCustomLevel(String switchName, boolean isOn, int customLevel) {
        SwitchCustomLevel scl = new SwitchCustomLevel(switchName, isOn, customLevel);
        addSwitchCustomLevel(scl);
    }
    
    private void addSwitchCustomLevelToMapAndValidate(SwitchCustomLevel scl) {
        String key = getKeyForSwitch(scl.switchName, scl.isOn);
        if ( customLevelsMap.containsKey(key) ) {
            throw new IllegalArgumentException("Custom level already defined for switch "+key+" \""+customLevelsMap.get(key)+"\"");
        }
        customLevelsMap.put(key, scl);
    }
    
    
    private static String getKeyForSwitch(String switchName, boolean isOn) {
        String key = switchName + (isOn ? SwitchCustomLevel.ON_SEPARATOR : SwitchCustomLevel.OFF_SEPARATOR);
        return key.substring(0, key.length()-1);
    }
    
    public static class SwitchCustomLevel {
        private final String switchName;
        private final int customLevel;
        private final boolean isOn;
        
        private static final String ON_SEPARATOR = "-On:";
        private static final String OFF_SEPARATOR = "-Off:";
        
        
        SwitchCustomLevel(String switchName, boolean isOn, int customLevel) {
            this.switchName = switchName;
            this.isOn = isOn;
            this.customLevel = customLevel;
        }

        public SwitchCustomLevel(String stringRepresentation) {
            this.isOn = stringRepresentation.contains(ON_SEPARATOR);
            String separator = isOn ? ON_SEPARATOR : OFF_SEPARATOR;
            String[] split = stringRepresentation.split(separator);
            
            if ( split.length != 2 ) {
                throw new IllegalArgumentException("Illegal format for input string \""+stringRepresentation+"\"");
            } 
            
            this.switchName = split[0];
            try {
                this.customLevel = Integer.parseInt(split[1]);
            } catch (Exception e) {
                throw new IllegalArgumentException("Illegal format for input string \""+stringRepresentation+"\"",e);
            }
            
        }
                
        @Override
        public String toString() {
            return switchName+(isOn ? ON_SEPARATOR : OFF_SEPARATOR)+customLevel;                    
        }

        @Override
        public boolean equals(Object obj) {
            if ( obj instanceof SwitchCustomLevel in ) {
                return in.switchName.equals(switchName) && in.isOn == isOn && in.customLevel == customLevel;
            }
            return false;
        }
        
        
        
        int getCustomLevel() {
            return customLevel;
        }
        
    }
    
    
    
    
    
}
