package org.lsst.ccs.subsystem.common.devices.turbopump;

import java.util.HashMap;
import java.util.Arrays;
import java.util.Map;
import java.util.logging.Logger;
import java.util.logging.Level;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.Options;
import org.lsst.ccs.command.SupportedOption;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Option;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.config.ConfigurationParameterDescription;
import org.lsst.ccs.config.HasConfigurationParameterDescription;
import org.lsst.ccs.drivers.twistorr.TwisTorr;
import org.lsst.ccs.drivers.twistorr.TwisTorrException;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.common.alerts.TurboAlerts;
import org.lsst.ccs.subsystem.common.devices.vacuum.states.TurboState;

/* Pre-defined command levels */
import static org.lsst.ccs.command.annotations.Command.NORMAL;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING_ROUTINE;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING_ADVANCED;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING_EXPERT;

/**
 *  Interfaces with driver for Agilent TwisTorr turbo pump.
 *
 *  The Channels read are TurboPump driver/hardware quantities.
 *
 *  The groovy Channel type  identifies whether a quantity to be read 
 *  is of type logical ("Bool"), integer ("Numeric") or alphanumeric 
 *  ("Alpha").  The subtype identifies the specific quantity tto be read.
 *
 *  @author Al Eisner, CCS Team
 */
public class TwisTorrDevice extends Device implements HasConfigurationParameterDescription {

   /**
    *  Data fields.
    */

    @LookupPath
    protected String path;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService stateService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;

    private final TwisTorr turbo = new TwisTorr();  // Associated driver
    private Integer chanKey;
    private volatile TurboState lastState;

    private Thread openMoment;
    
    protected volatile TwisTorr.CntlrModels model;
    private static final Logger LOG = Logger.getLogger(TwisTorrDevice.class.getName());
    private static final String autotuneMsg = "autotuning: speed below nominal, Run-up timer (15m to FAIL) started";
    private Boolean OK_TO_OPERATE = true;   // should be volatile?

    private final SupportedOption forceOption = SupportedOption.getSupportedOption("force");

    @ConfigurationParameter(isFinal = true, units = "unitless", description = "serial port")
    protected volatile String devcId;       // Device connection identification

    /* Configuration parameters related to hardware settings */

    @ConfigurationParameter(isFinal = true, description = "Active-Stop mode <true|false>, req pump stopped", units = "unitless")
    protected volatile Boolean activeStopMode;

    @ConfigurationParameter(isFinal = true, description = "0 = freq, 1 = power", units = "Hz")
    protected volatile Integer analogOutput;

    @ConfigurationParameter(isFinal = true, description = "Baudrate, 4=9600 ", units = "unitless")
    protected volatile Integer baudRate;

    @ConfigurationParameter(isFinal = true, description = "0 = always on,1 = controlled", units = "unitless")
    protected volatile Integer extFanConfig;

    @ConfigurationParameter(isFinal = true, description = "0,1,2 for Ar/N2/He", units = "unitless")
    protected volatile Integer gasType;

    @ConfigurationParameter(isFinal = true, description = "true/false for Ar/N2", units = "unitless")
    protected volatile Boolean gasTypeArgon;
    
    @ConfigurationParameter(isFinal = true, description = "interlock type ... normally continuous", units = "unitless")
    protected volatile Boolean interlockType;

    @ConfigurationParameter(isFinal = true, units = "unitless", description = "true/false for low/normal speed")
    protected volatile Boolean lowSpeedMode;

    @ConfigurationParameter(isFinal = true, description = "max power cutoff (read-only)", units = "Watts")
    protected volatile Integer maxpower;

    @ConfigurationParameter(isFinal = true, description = "Rotation Freq Low limit", units = "Hz")
    protected volatile Integer rotfreqLow;

    @ConfigurationParameter(isFinal = true, description = "Rotation Freq Max limit", units = "Hz")
    protected volatile Integer rotfreqMax;

    @ConfigurationParameter(isFinal = true, description = "Rotation Freq Set limit", units = "Hz")
    protected volatile Integer rotfreqSet;

    @ConfigurationParameter(isFinal = true, description = "0:freq, 1:current, ...", units = "Hz")
    protected volatile Integer setpointType;  // 0:freq, 1:current, ...

    @ConfigurationParameter(isFinal = true, description = "Threshold to activate digital outout", units = "Hz")
    protected volatile Integer setpointThresh; // Thresh we use ~0.1*maxrotspeed which is not default

    @ConfigurationParameter(isFinal = true, description = "delay to activate", units = "s")
    protected volatile Integer setpointDelay; // using default value of 0 (no delay)

    @ConfigurationParameter(isFinal = true, description = "Setpoint value for digital outout", units = "Hz")
    protected volatile Integer setpointHyst;  // using default value of 2%

    @ConfigurationParameter(isFinal = true, description = "0|1 = active(High|Low)", units = "unitless")
    protected volatile Boolean setpointActive;  // setpointActive 0:High or 1:Low

    @ConfigurationParameter(isFinal = true, description = "Soft-Start mode <true|false>, req pump stopped", units = "unitless")
    protected volatile Boolean softStartMode; 

    @ConfigurationParameter(isFinal = true, description = "Read speed after Stop", units = "unitless")
    protected volatile Boolean speedReadActivate; 

    @ConfigurationParameter(isFinal = true, description = "critical value per model", units = "unitless")
    protected volatile Boolean ventvalveType;      // certain models only

    @ConfigurationParameter(isFinal = true, description = "By-command mode", units = "unitless")
    protected volatile Boolean ventvalveByCmnd; 

    @ConfigurationParameter(isFinal = true, description = "opening delay count", units = "0.2s/count")
    protected volatile Integer ventDelay;
    
    @ConfigurationParameter(isFinal = true, description = "opening time, 74,84 models only", units = "0.2s/count")
    protected volatile Integer ventOpentime;
    
    @ConfigurationParameter(isFinal = true, description = "water cooling", units = "unitless")
    protected volatile Boolean waterCooling;

    private double pulseTime = 0.0;
    private int pulseCnt= 0;

  /**
    *  Performs basic initialization.  
    */
    @Override
    public void initDevice() {

        stateService.registerState(TurboState.class, "Turbo-pump State", this);
        alertService.registerAlert(TurboAlerts.IN_AUTOTUNING.newAlert(path));
        alertService.registerAlert(TurboAlerts.VERIFY_FAILED.newAlert(path));

        chanKey = 0;

        if (devcId.equals("")) {
            LOG.log(Level.SEVERE, "{0} configuration parameter devcId is missing", new Object[]{name});
            throw new RuntimeException("Fatal configuration error");
        }

    }
    
    public void verifyTurboSetting(TwisTorr.CmndBool cmnd, Object val) {
        boolean success = false;
        if ((model.getModelMask() & cmnd.getModels()) != 0) {  // the cmnd is supported by the model
            try {
                Boolean value = turbo.readBool(cmnd);
                success = (value == (Boolean)val);
                LOG.log(Level.FINE, "success:{0} for value:{1} and val:{2} for cmnd:{3} on model:{4}", new Object[]{success, value, val, cmnd, model});
                if (success) {
                    return;
                } else {
                    LOG.log(Level.SEVERE, "Check of {0}:{1} != {2} (config value) on model:{4}",
                            new Object[]{cmnd, val, value, model});
                    alertService.raiseAlert(TurboAlerts.VERIFY_FAILED.newAlert(path), AlertState.ALARM,
                            String.format("Check of %s:%s != %s (config value) on model:%s", cmnd, val, value, model));
                    OK_TO_OPERATE = false;
                }
            } catch (DriverException e) {
                LOG.log(Level.SEVERE,"Read of val:{0} for register:{1} of model:{2} failed",new Object[]{val, cmnd, model});
                alertService.raiseAlert(TurboAlerts.VERIFY_FAILED.newAlert(path), AlertState.ALARM,
                        String.format("Read of value:%s for register:%s of model:%s failed",new Object[]{val, cmnd, model}));
                OK_TO_OPERATE = false;
                return;
            }
        } else {  // unsupported command
            LOG.log(Level.INFO,"skipping val:{0} check of {1} for model:{2} as unsupported",new Object[]{val, cmnd, model});
        }
    }

    public void verifyTurboSetting(TwisTorr.CmndNumeric cmnd, Object val) {
        boolean success = false;
        if ((model.getModelMask() & cmnd.getModels())!=0) {
            try {
                int value = turbo.readNumeric(cmnd);
                success = (value == (Integer)val);
                LOG.log(Level.FINE, "success:{0} for value:{1} and val:{2} for cmnd:{3} on model:{4}", new Object[]{success, value, val, cmnd, model});
                if (success) {
                    return;
                } else {
                    LOG.log(Level.SEVERE, "Check of {0}:{1} != {2} (config value) on model:{4}",
                            new Object[]{cmnd, val, value, model});
                    alertService.raiseAlert(TurboAlerts.VERIFY_FAILED.newAlert(path), AlertState.ALARM,
                            String.format("Check of %s:%s != %s (config value) on model:%s", cmnd, val, value, model));
                    OK_TO_OPERATE = false;
                }
            } catch (DriverException e) {
                LOG.log(Level.SEVERE,"Read of value:{0} for register:{1} of model:{2} failed",new Object[]{val, cmnd, model});
                alertService.raiseAlert(TurboAlerts.VERIFY_FAILED.newAlert(path), AlertState.ALARM,
                        String.format("Read of value:%s for register:%s of model:%s failed",new Object[]{val, cmnd, model}));
                OK_TO_OPERATE = false;
                return;
            }
        } else {
            LOG.log(Level.INFO,"skipping val:{0} check of {1} for model:{2} as unsupported",new Object[]{val, cmnd, model});
        }
    }

   /**
    *  Performs full initialization.
    */
    @Override
    public void initialize() {

        try {
            turbo.open(devcId);
            int status = turbo.readNumeric(TwisTorr.CmndNumeric.STATUS);
            TurboState state = TurboState.getState(status);
            LOG.log(Level.INFO, "Connected to {0} Turbo state = {1}, starting hardware settings", new Object[]{fullName, state});
            closeVentValve(); // initialize to a known safe state

            /* Hardware initialization */
            // settings to be written
            // we may need to write to SOFT_START_MODE but wait and see
            // critical settings to be verified
            verifyTurboSetting(TwisTorr.CmndBool.ACTIVE_STOP_MODE, activeStopMode);         // c
            verifyTurboSetting(TwisTorr.CmndNumeric.GAS_TYPE, gasType);                     // c, set to N2
            verifyTurboSetting(TwisTorr.CmndBool.GAS_TYPE_ARGON, gasTypeArgon);             // c, set to N2
            verifyTurboSetting(TwisTorr.CmndBool.INTERLOCK_TYPE, interlockType);            // c, critical, so check
            verifyTurboSetting(TwisTorr.CmndBool.SPEED_READ_ACTIVATE, speedReadActivate);   // c, critical, so check
            verifyTurboSetting(TwisTorr.CmndNumeric.SETPOINT_TYPE, setpointType);           // d, default but critical
            verifyTurboSetting(TwisTorr.CmndNumeric.SETPOINT_THRESH, setpointThresh);         // c, super critical
            verifyTurboSetting(TwisTorr.CmndBool.SETPOINT_ACTIVE, setpointActive);            // d, but critical so check
            verifyTurboSetting(TwisTorr.CmndBool.SOFT_START_MODE, softStartMode);           // c
            verifyTurboSetting(TwisTorr.CmndBool.VENTVALVE_TYPE, ventvalveType);            // c
            verifyTurboSetting(TwisTorr.CmndBool.VENTVALVE_BY_CMND, ventvalveByCmnd);       // c
            verifyTurboSetting(TwisTorr.CmndNumeric.VENT_DELAY, ventDelay);                 // c,
            verifyTurboSetting(TwisTorr.CmndNumeric.VENT_OPENTIME, ventOpentime);           // c, only 74,84

            // remaining setings also verified
            verifyTurboSetting(TwisTorr.CmndBool.LOW_SPEED_MODE, lowSpeedMode);
            verifyTurboSetting(TwisTorr.CmndNumeric.SETPOINT_DELAY, setpointDelay);
            verifyTurboSetting(TwisTorr.CmndNumeric.SETPOINT_HYST, setpointHyst);
            verifyTurboSetting(TwisTorr.CmndBool.WATER_COOLING, waterCooling);
            verifyTurboSetting(TwisTorr.CmndNumeric.BAUD_RATE, baudRate);
            verifyTurboSetting(TwisTorr.CmndNumeric.ANALOG_OUTPUT, analogOutput);
            verifyTurboSetting(TwisTorr.CmndNumeric.ROTFREQ_LOW, rotfreqLow);
            verifyTurboSetting(TwisTorr.CmndNumeric.ROTFREQ_SET, rotfreqSet);
            verifyTurboSetting(TwisTorr.CmndNumeric.ROTFREQ_MAX, rotfreqMax);
            verifyTurboSetting(TwisTorr.CmndNumeric.VENT_DELAY, ventDelay);
            verifyTurboSetting(TwisTorr.CmndNumeric.EXT_FAN_CONFIG, extFanConfig);
            verifyTurboSetting(TwisTorr.CmndNumeric.MAXPOWER, maxpower);

            initSensors();
            setOnline(true);
            LOG.log(Level.INFO, "{0} Hardware initialization succeeded", new Object[]{name});

        } catch (DriverException e) {
            if (!inited) {
                LOG.log(Level.SEVERE, "Error connecting/initializing {0}: {1}, closing", new Object[]{fullName, e});
            }
            close();
        }

        inited = true;
    }


   /**
    *  Closes the connection.
    */
    @Override
    public void close() {
        try {
            turbo.close();
        }
        catch (DriverException e) {
        }
    }

    /**
     * Pump commands and settings
     */

   /**
    *  Start pump
    *
    *  @throws  DriverException
    */
    @Command(type=Command.CommandType.ACTION, name="startTurboPump", 
             description="Start turbo pump", level=100)
    public void startTurboPump() throws DriverException
    {
        if (!OK_TO_OPERATE) {
            throw new RuntimeException("Must clear error state to operate turbo.");
        }
        turbo.writeBool(TwisTorr.CmndBool.START_STOP, true);
    }

   /**
    *  Stop pump
    *
    *  @throws  DriverException
    */
    @Command(type=Command.CommandType.ACTION, name="stopTurboPump", 
             description="Stop turbo pump")
    public void stopTurboPump() throws DriverException 
    {
        turbo.writeBool(TwisTorr.CmndBool.START_STOP, false);
    }

   /**
    *  Timed open vent valve - open vent valve for a predefined peiod of time
    *
    *  @throws  DriverException
    */
    @Command(type=Command.CommandType.ACTION, name="timedPulseVentValve",
             description="pulse open vent valve (certain models only)")
    public void timedPulseVentValve() throws DriverException 
    {
        if (!OK_TO_OPERATE) {
            throw new RuntimeException("Must clear error state to operate turbo.");
        }
        if (!(pulseTime >= 0.0) || pulseTime > 0.5) {
            throw new RuntimeException("Period for vent must satisfy 0.0 =< pulseTime < 0.5");
        }
        openMoment = new Thread("openMoment") {
                @Override
                public void run() {
                    try {
                        openVentValve();
                    } catch (DriverException ex) {
                        try {
                            closeVentValve();
                            LOG.log(Level.SEVERE, "Closed VentValve due to Driver exception");
                        } catch (DriverException ec) { }
                        throw new RuntimeException("Driver exception occurred when trying to open the turbo vent valve",ex);
                    }
                    try{
                        Thread.sleep((long)(pulseTime*1000));
                    } catch (InterruptedException ex) {
                        try {
                            closeVentValve();
                            LOG.log(Level.SEVERE, "Closed VentValve due to exception during sleep()");
                        } catch  (DriverException ec) { }
                        throw new RuntimeException("Unexpected interrupt while waiting for turbo vent valve openning period to end",ex);
                    }
                    try {
                        closeVentValve();
                    } catch (DriverException ex) {
                        throw new RuntimeException("Driver exception occurred when trying to close the turbo vent valve",ex);
                    }
                }
            };
        openMoment.setDaemon(true);
        openMoment.start();
    }

   /**
    *  Open vent valve
    *
    *  @throws  DriverException
    *
    *  Constraints:
    *              VENTVALVE_TYPE (140), if supported, must have value of 1 for model 305FS
    *  Question:
    *              should this be a command?  Maybe just an internal method?
    */
    @Command(type=Command.CommandType.ACTION, name="openVentValve",
             description="open vent valve (certain models only)",level=100)
    public void openVentValve() throws DriverException 
    {
        if (!OK_TO_OPERATE) {
            throw new RuntimeException("Must clear error state to operate turbo.");
        }
        if ((model.getModelMask() & TwisTorr.CmndBool.VENTVALVE_TYPE.getModels()) != 0) { // model supports type
            Boolean type = turbo.readBool(TwisTorr.CmndBool.VENTVALVE_TYPE);  // null object will cast to false
            if (!type) {
                throw new RuntimeException("vent valve type (window 140) must be 1");
            }
        }
        if ((model.getModelMask() & TwisTorr.CmndBool.VENTVALVE_OPEN.getModels()) != 0) {
            turbo.writeBool(TwisTorr.CmndBool.VENTVALVE_OPEN, true);
        }
    }

   /**
    *  Close vent valve
    *
    *  @throws  DriverException
    *
    *  Constraints:
    *              VENTVALVE_TYPE (140) must have value of 1 for model 305FS (no others are tested yet)
    *  Question:
    *              Is this being handled right?  We don't want to pevent closing
    *              Worst case on fail is to power off the controller for NC valve
    */
    @Command(type=Command.CommandType.ACTION, name="closeVentValve",
             description="close vent valve (certain models only)")
    public void closeVentValve() throws DriverException 
    {
        if ((model.getModelMask() & TwisTorr.CmndBool.VENTVALVE_TYPE.getModels()) != 0) { // model supports type
            Boolean type = turbo.readBool(TwisTorr.CmndBool.VENTVALVE_TYPE);  // null object will cast to false
            if (!type) {
                throw new RuntimeException("vent valve type (window 140) must be 1");
            }
        }
        if ((model.getModelMask() & TwisTorr.CmndBool.VENTVALVE_OPEN.getModels()) != 0) {
            turbo.writeBool(TwisTorr.CmndBool.VENTVALVE_OPEN, false);
        }
    }

   /**
    *  Show vent valve settings and status
    *
    *  @return  String
    *  @throws  DriverException
    */
    @Command(type=Command.CommandType.QUERY, name="showVentValveStatus",
             description="show vent valve settings and status (certain models only)")
    public String showVentValveStatus() throws DriverException 
    {
        String text = "";
        if ((model.getModelMask() & TwisTorr.CmndBool.VENTVALVE_TYPE.getModels()) != 0) {
            Boolean type = turbo.readBool(TwisTorr.CmndBool.VENTVALVE_TYPE);
            if (!type) {
                throw new RuntimeException("vent valve type (window 140) must be 1");
            }
        }
        if ((model.getModelMask() & TwisTorr.CmndBool.VENTVALVE_OPEN.getModels()) != 0) {
            Boolean set = turbo.readBool(TwisTorr.CmndBool.VENTVALVE_OPEN);
            text = String.format("VENTVALVE_SETTING (122) = %s", set);
        } else {
            text = "Not implemented for this model";
        }
        return text;
    }

   /**

    *  Check whether the vent valve is open or not
    *
    *  @return  Boolean (null (unsupported), false:closed, true:open | throws)
    *  @throws  DriverException
    *
    */
    @Command(type=Command.CommandType.QUERY, name="isVentValveOpen",
             description="return whether the vent valve is open (certain models only)")
    public Boolean isVentValveOpen() throws DriverException 
    {
        if ((model.getModelMask() & TwisTorr.CmndBool.VENTVALVE_TYPE.getModels()) != 0) {
            Boolean type = turbo.readBool(TwisTorr.CmndBool.VENTVALVE_TYPE);
            if (!type) {
                throw new RuntimeException("vent valve type (window 140) must be 1");
            }
        }
        if ((model.getModelMask() & TwisTorr.CmndBool.VENTVALVE_OPEN.getModels()) != 0) {
            return(turbo.readBool(TwisTorr.CmndBool.VENTVALVE_OPEN));
        } else {
            return null;
        }
    }

   /**
    *  Checks a channel's parameters for validity.
    *  The numeric type returned is just an ordinal Channel index;
    *  the numeric subtype is not used.
    *
    *  @param   Channel ch    The channel
    *  @return  A two-element array containing the encoded type [0] and
    *           subtype [1] values.
    *  @throws  Exception if any errors found in the parameters.
    */
    @Override
    public int[] checkChannel(Channel ch) throws Exception
    {
        String type = ch.getTypeStr();
        String subtype = ch.getSubTypeStr();
        String name = ch.getName();
        try {
            switch (type) {
                case "Bool":
                    TwisTorr.CmndBool.valueOf(subtype);
                    break;
                case "Numeric":
                    TwisTorr.CmndNumeric.valueOf(subtype);
                    break;
                case "Alpha":
                    TwisTorr.CmndAlpha.valueOf(subtype);
                    break;
                default:
                    LOG.log(Level.SEVERE, "Invalid type value ({0}) for {1} channel", new Object[]{type, name});
                    throw new Exception("Invalid channel parameter");
            }
        }
        catch (IllegalArgumentException e) {
            LOG.log(Level.SEVERE, "Invalid type value ({0}) for {1} channel", new Object[]{type, name});
            throw new Exception("Invalid channel parameter");
        }
        chanKey++;
        return new int[]{chanKey, 0};
    }


   /**
    *  Reads a channel.
    *
    *  @param  Channel ch
    *  @return  The read value or NaN
    */
    @Override
    public double readChannel(Channel ch)
    {
        double value = super.readChannel(ch);     //NaN
        String itemType = ch.getTypeStr();
        String item = ch.getSubTypeStr();
        // Note:  the framework assures that null "get" returns do not occur

        try {
            switch (itemType) {
                case "Bool":
                    if (turbo.readBool(TwisTorr.CmndBool.valueOf(item))) {
                        value = 1.0;
                    } else {
                        value = 0.0;
                    }
                    break;
                case "Numeric":
                    value = turbo.readNumeric(TwisTorr.CmndNumeric.valueOf(item));
                    if (item.equals("STATUS")) {  // this should be split out into separate method?
                        int status = (int) Math.round(value);
                        TurboState state = TurboState.getState(status);
                        if (state != lastState) {
                            stateService.updateAgentComponentState(this, state);
                            if (state == TurboState.AUTOTUNING) {
                                alertService.raiseAlert(TurboAlerts.IN_AUTOTUNING.newAlert(path), AlertState.WARNING, autotuneMsg);
                            } else if (lastState == TurboState.AUTOTUNING) {
                                alertService.raiseAlert(TurboAlerts.IN_AUTOTUNING.newAlert(path), AlertState.NOMINAL, "autotuning exit");
                            }
                            if (state == TurboState.FAIL) {
                                try {
                                    String cause = getTurboFailCode();
                                    LOG.log(Level.INFO, name + " TURBO FAILURE, cause(s) = " + cause);
                                }
                                catch (Exception ex) {
                                    LOG.log(Level.SEVERE, name + " failed to read error code, " + ex);
                                }
                            }
                            lastState = state;
                        }
                    }
                    break;
                case "Alpha":
                    turbo.readAlpha(TwisTorr.CmndAlpha.valueOf(item));
                    // Not implemented, relevant hardware not installed
                    break;
            }
        }        
        catch (DriverTimeoutException et) {
            LOG.log(Level.SEVERE, name + ": TurboPump timeout reading data " + item + ": " + et);
            setOnline(false);
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, name + ": TurboPump exception reading data " + item + ": " + e);
        }
        
        return value;
    }

   /**
    *  Get and decode error word (bits show reason for pump failure)
    *
    *  @return String
    *  @throws  DriverException
    */
    public String getTurboFailCode()  throws DriverException
    {
        int errcode = turbo.readNumeric(TwisTorr.CmndNumeric.ERRCODE);
        return Integer.toString(errcode, 16) + ", "+ TwisTorr.decodeError(errcode);
    }

    /* Read-commands for command-line usage */

   /** 
    *  Read and decode pump status
    *
    *  @return  String indicating pump state or failure
    *  @throws  DriverException
    */
    @Command(type=Command.CommandType.QUERY, level=NORMAL, name="readTurboStatus", 
             description="Get pump status, including operating modes and failure")
    public TwisTorr.PumpStatus readTurboStatus()  throws DriverException
    {
        int status = turbo.readNumeric(TwisTorr.CmndNumeric.STATUS);
        return TwisTorr.PumpStatus.decodeStatus(status);
    }

   /** 
    *  Read error code (bits show reason for pump failure)
    *
    *  @return  String giving error code in hexadecimal and decoding it
    *  @throws  DriverException
    */
    @Command(type=Command.CommandType.QUERY, name="readTurboFailCode", 
             description="Find reason(s) for failure", level=NORMAL)
    public String readTurboFailCode()  throws DriverException
    {
        return getTurboFailCode();
    }

    /**
     *  Set vent-valve pulse count
     *
     *  @param value   number of pulses
     *  @throws  DriverException
     */
    @Command(name="setVentPulseCnt", description="Vent valve-open pulse count")
    public void setVentPulseCnt(int value)
    {
        pulseCnt = value;
    }

    /**
     *  Set vent-valve open time in seconds
     *
     *  @param value   Time open in seconds
     *  @throws  DriverException
     */
    @Command(name="setVentPulseTime", description="Vent valve-open time with time in seconds")
    public void setVentPulseTime(double value)
    {
        pulseTime = value;
    }

   /** 
    *  Read TurboPump running-time in hours
    *
    *  @return  String giving active time
    *  @throws  DriverException 
    */
    @Command(type=Command.CommandType.QUERY,level=ENGINEERING_ROUTINE,
             name="readTurboLife",
             description="show TurboPump running-time in hours")
    public String readTurboLife() throws DriverException
    {
        int life = turbo.readNumeric(TwisTorr.CmndNumeric.PUMP_HOURS);
        return Integer.toString(life) + " hours";
    }

   /**
    * Read all settings and data from pump controller.
    *
    * Loops over all read commands and returns them in a table format.
    * All DriverExceptions are caught; if one occurs, the data field
    * is replaced by the text (String) associated with the exception.
    *  
    * @throws InterruptedException
    * @return  String reporting all data read and exceptions.
    */
    @Command(type=Command.CommandType.QUERY,level=ENGINEERING_ROUTINE, name="readAll", description="Read all supported controller settings and data") 
    @Option(name = "force", description = "Read all known registers, regardless of controller model")
    public String readAll(Options options) throws InterruptedException
    {
        long startTime = System.currentTimeMillis();
        String table = "Read all turbo-pump settings and data\n" +"\n";
        boolean forceRead = options.hasOption(forceOption);
        table += String.format("Controller Model Set to: %s", model.toString());

        /* Boolean commands */

        TwisTorr.CmndBool cmndB[] = TwisTorr.CmndBool.values();
        int nB = cmndB.length;
        for (int i = 0; i < nB; i++) {
            if (forceRead || (model.getModelMask() & cmndB[i].getModels()) != 0) {
                table += String.format("\n%3s    %-22s", cmndB[i].getWindow(), cmndB[i]);
                try {
                    boolean respB = turbo.readBool(cmndB[i]);
                    table += Boolean.toString(respB);
                } catch (DriverException ex) {
                    table += ex.getMessage();
                }
            }
        }
        table += "\n";

        /* Numeric commands */

        TwisTorr.CmndNumeric cmndN[] = TwisTorr.CmndNumeric.values();
        int nN = cmndN.length;
        for (int i = 0; i < nN; i++) {
            if (forceRead || (model.getModelMask() & cmndN[i].getModels()) != 0) {
                table += String.format("\n%3s    %-22s", cmndN[i].getWindow(), cmndN[i]);
                try {
                    int respN = turbo.readNumeric(cmndN[i]);
                    table += Integer.toString(respN);
                } catch (DriverException ex) {
                    table += ex.getMessage();
                }
            }
        }
        table += "\n";

        TwisTorr.CmndAlpha cmndA[] = TwisTorr.CmndAlpha.values();
        int nA = cmndA.length;
        for (int i = 0; i < nA; i++) {
            if (forceRead || (model.getModelMask() & cmndA[i].getModels()) != 0) {
                table += String.format("\n%3s    %-22s", cmndA[i].getWindow(), cmndA[i]);
                try {
                    String respA = turbo.readAlpha(cmndA[i]);
                    table += respA;
                } catch (DriverException ex) {
                    table += ex.getMessage();
                }
            }
        }
        table += "\n";
        long dt = System.currentTimeMillis() - startTime;
        table += String.format("\nreadAll() time: %d\n", dt);

        return table;
    }

   /**
    *  Get the current TurboState
    *
    *  @return TurboState
    */
    public TurboState getTurboState() {
        return lastState;
    }

   /** 
    *  Return the OK_TO_OPERATE flag state
    *
    *  @return  boolean
    */
    @Command(type=Command.CommandType.QUERY,
             name="getOperationalState",
             description="returns the OK_TO_OPERATE flag state")
    public Boolean getOperationalState()
    {
        return OK_TO_OPERATE;
    }

   /** 
    *  Resets the OK_TO_OPERATE flag
    *
    *  @return  void
    */
    @Command(type=Command.CommandType.ACTION,level=ENGINEERING_ROUTINE,
             name="resetOperationalState",
             description="resets the OK_TO_OPERATE flag")
    public void resetOperationalState()
    {
        OK_TO_OPERATE=true;
    }


    /**
     *  Set the current turbo model
     *
     *  @return turbo mode
     */

    //    @ConfigurationParameterChanger(propertyName = "model")
    public void setModel(TwisTorr.CntlrModels model) throws DriverException {
        this.model = model;

        LOG.log(Level.SEVERE, "Model set to " + model);
        
    }

}
