package org.lsst.ccs.subsystem.common;

import java.security.InvalidParameterException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.description.ComponentNode;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.subsystem.common.data.MonitorTask;

/**
 *  Controls special monitoring tasks.
 * 
 *  @author saxton
 */
public class MonitorTaskControl implements HasLifecycle {
    
    /**
     *  Constants.
     */
    private static final String
        MON_UPDATE_TASK = "monitor-update",
        MON_PUBLISH_TASK = "monitor-publish";

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;

    private final Map<String, MonitorTask> monitorTaskMap = new LinkedHashMap<>();
    private boolean periodChanged = false;
    

    /**
     *  Create a new object and add it to the tree. 
     * 
     *  @param  subsys  The subsystem object
     *  @param  name  The name of this node
     *  @return  The created object
     */
    public static MonitorTaskControl createNode(Subsystem subsys, String name)
    {
        ComponentLookup lookupService = subsys.getComponentLookup();
        ComponentNode topNode = lookupService.getComponentNodeForObject(subsys);
        MonitorTaskControl control = new MonitorTaskControl();
        ComponentNode controlNode = new ComponentNode(name, control);
        lookupService.addComponentNodeToLookup(topNode, controlNode);
        return control;
    }


    /**
     *  Post-init phase.
     */
    @Override
    public void postInit()
    {
        // Get the names of the special monitor update tasks
        List<String> allTasks = periodicTaskService.getAgentPeriodicTaskNames();
        Set<String> monTasks = new TreeSet<>();  // Sorts the names
        for (String taskPath : allTasks) {
            if (taskPath.startsWith(MON_UPDATE_TASK + "/")) {
                monTasks.add(taskPath.substring(taskPath.indexOf('/') + 1));
            }
        }
        int index = 0;
        for (String taskName : monTasks) {
            monitorTaskMap.put(taskName, new MonitorTask(taskName, index++, getTaskUpdatePeriod(taskName)));
        }
    }


    /**
     *  Command to set the monitor publishing period.
     *
     *  @param  period  The period (milliseconds) to set.
     */
    @Command(type=Command.CommandType.ACTION, description="Set the monitor publish period")
    public synchronized void setPublishPeriod(@Argument(description="The publish period (ms)") int period)
    {
        periodicTaskService.setPeriodicTaskPeriod(MON_PUBLISH_TASK, Duration.ofMillis(period));
        for (MonitorTask task : monitorTaskMap.values()) {
            synchronized (task) {
                if (!task.isActive()) {
                    setTaskPublishPeriod(task.getName(), period);
                }
            }
        }
        periodChanged = true;
    }


    /**
     *  Command to get the list of known special monitor task pair names.
     * 
     *  @return  The list of names
     */
    @Command(type=Command.CommandType.QUERY, level=0, description="Get the special monitor task names")
    public List<String> getTaskNames()
    {
        List<String> monTasks = new ArrayList<>();
        monTasks.addAll(monitorTaskMap.keySet());
        return monTasks;
    }


    /**
     *  Command to set the period for a special monitor task pair.
     * 
     *  @param  taskName  The name of the task
     *  @param  period    The period (ms)
     */
    @Command(type=Command.CommandType.ACTION, description="Set the period for a special monitor task")
    public void setTaskPeriod(@Argument(description="The task name") String taskName,
                              @Argument(description="The task period (ms)") int period)
    {
        MonitorTask task = getTask(taskName);
        synchronized (task) {
            task.setPeriod(period);
            if (task.isActive()) {
                setTaskUpdatePeriod(taskName, period);
                setTaskPublishPeriod(taskName, period);
            }
        }
    }


    /**
     *  Command to set the active state for a special monitor task pair.
     * 
     *  @param  taskName  The name of the task
     *  @param  active    Whether to activate or not
     */
    @Command(type=Command.CommandType.ACTION, description="Set the active state for a special monitor task")
    public void setTaskActive(@Argument(description="The task name") String taskName,
                              @Argument(description="Whether to make active") boolean active)
    {
        MonitorTask task = getTask(taskName);
        synchronized (task) {
            task.setActive(active);
            if (active) {
                int period = task.getPeriod();
                setTaskUpdatePeriod(taskName, period);
                setTaskPublishPeriod(taskName, period);
            }
            else {
                setTaskUpdatePeriod(taskName, getUpdatePeriod());
                setTaskPublishPeriod(taskName, getPublishPeriod());
            }
        }
    }


    /**
     *  Gets whether update period has changed.
     * 
     *  @return  Whether changed
     */
    public synchronized boolean hasPeriodChanged()
    {
        boolean changed = periodChanged;
        periodChanged = false;
        return changed;
    }


    /**
     *  Gets the map of monitoring task pairs.
     * 
     *  @return  The map
     */
    public Map<String, MonitorTask> getMonitorTaskMap()
    {
        return monitorTaskMap;
    }


    /**
     *  Gets the monitoring publishing period.
     *
     *  @return  The publishing period (ms)
     */
    public int getPublishPeriod()
    {
        return (int)periodicTaskService.getPeriodicTaskPeriod(MON_PUBLISH_TASK).toMillis();
    }


    /**
     *  Gets the monitoring update period.
     *
     *  @return  The update period (ms)
     */
    private int getUpdatePeriod()
    {
        return (int)periodicTaskService.getPeriodicTaskPeriod(MON_UPDATE_TASK).toMillis();
    }


    /**
     *  Gets the update period for a monitoring task pair.
     *
     *  @param  taskName  The task name
     *  @return  The update period (ms)
     */
    private int getTaskUpdatePeriod(String taskName)
    {
        return (int)periodicTaskService.getPeriodicTaskPeriod(MON_UPDATE_TASK + "/" + taskName).toMillis();
    }
    

    /**
     *  Sets the update period for a monitoring task pair.
     *
     *  @param  taskName  The task name
     *  @param  period    The period (ms) to set.
     */
    private void setTaskUpdatePeriod(String taskName, int period)
    {
        periodicTaskService.setPeriodicTaskPeriod(MON_UPDATE_TASK + "/" + taskName, Duration.ofMillis(period));
    }
    

    /**
     *  Sets the publishing period for a monitoring task pair.
     *
     *  @param  taskName  The task name
     *  @param  period    The period (ms) to set.
     */
    private void setTaskPublishPeriod(String taskName, int period)
    {
        periodicTaskService.setPeriodicTaskPeriod(MON_PUBLISH_TASK + "/" + taskName, Duration.ofMillis(period));
    }
    

    /**
     *  Gets a named monitor task pair.
     * 
     *  @param  taskName    The task name
     */
    private MonitorTask getTask(String taskName)
    {
        MonitorTask task = monitorTaskMap.get(taskName);
        if (task == null) {
            throw new InvalidParameterException("Unknown monitor task name: " + taskName);
        }
        return task;
    }

}
