package org.lsst.ccs.subsystem.common.devices.power.distribution;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArraySet;
import org.lsst.ccs.Agent;
import org.lsst.ccs.StateChangeListener;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.subsystem.common.devices.power.distribution.state.PduOutletState;
import org.lsst.ccs.subsystem.common.devices.power.distribution.state.PduState;
import org.lsst.ccs.subsystem.common.devices.power.distribution.state.PduProperties;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.description.ComponentNode;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.apcpdu.APC7900;
import org.lsst.ccs.drivers.apcpdu.APC7900B;
import org.lsst.ccs.drivers.apcpdu.APC7900Series;
import org.lsst.ccs.drivers.apcpdu.APC7900Sim;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  APC7900 device class
 *
 *  @author: The LSST CCS Team
 */
public class APC7900Device extends Device implements StateChangeListener, ClearAlertHandler {

    public final static int
        TYPE_POWER = 1,
        CHAN_CURRENT = 0,
        CHAN_POWER = 1;

    final static Map<String, Integer> typeMap = new HashMap<>();
    static {
        typeMap.put("POWER", TYPE_POWER);
    }

    @ConfigurationParameter(isFinal = true)
    private String type = "APC7900";

    @ConfigurationParameter(isFinal = true)
    protected String node;

    @ConfigurationParameter(isFinal = true)
    protected int numberOfOutlets = 8;

    @ConfigurationParameter(isFinal = true)
    protected List<String> outlets = new ArrayList<>();

    @ConfigurationParameter(isFinal = true)
    protected boolean updateNamesInPDU = false;
    
    @ConfigurationParameter
    protected int maxReadFailures = 3;
    private int readFailures = 0;
    
    private final List<PduOutlet> listOfOutlets = new ArrayList();
    
    protected Map<String,PduOutlet> mapOfOutlets = new HashMap<>();
    private final Map<PduOutlet, PduOutletState> intendedOutletStates = new ConcurrentHashMap<>();
    private final Set<PduOutlet> outletsInAlertState = new CopyOnWriteArraySet<>();
    //A lock used to change/detect the intended state of an outlet. 
    private final Object intendedOutletStateLock = new Object();
    
    String user = "apc";
    String passwd = "apc";

    private static final Logger LOG = Logger.getLogger(APC7900Device.class.getName());
    @Deprecated
    protected final Logger sLog = LOG;
    private APC7900Series pdu;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private Agent a;
    
    private ComponentLookup componentLookup;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    protected AgentStateService stateService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    protected AlertService alertService;

    @LookupPath
    private String path;
    
    @Override
    public void build() {
        super.build(); //To change body of generated methods, choose Tools | Templates.
        
        if ( ! outlets.isEmpty() ) {
            if ( outlets.size() != numberOfOutlets ) {
                throw new IllegalArgumentException("The size of the provided outlets ("+outlets.size()+") must match the provided numberOfOutlets "+numberOfOutlets);
            }
        }
        
        componentLookup = a.getComponentLookup();
        ComponentNode thisNode = componentLookup.getComponentNodeForObject(this);
        for ( int i = 1; i <= numberOfOutlets; i++ ) {
            PduOutlet outlet = new PduOutlet(i);
            listOfOutlets.add(outlet);
            //Add the Pdu Outlet node and register its state.
            componentLookup.addComponentNodeToLookup(thisNode, new ComponentNode(outlet.getName(), outlet));
        }

        a.setAgentProperty(PduProperties.HAS_PDU, "true");
        String components = a.getAgentService(AgentPropertiesService.class).getAgentProperty(PduProperties.PDU_COMPONENTS);
        if (components == null) {
            components = "";
        }
        if ( !components.isEmpty() ) {
            components += ",";
        }
        components += path;
        a.setAgentProperty(PduProperties.PDU_COMPONENTS, components);        
    }

    @Override
    public void init() {

        if ( null != type ) switch (type) {
            case "APC7900":
                pdu = new APC7900();
                break;
            case "APC7900B":
                pdu = new APC7900B();
                break;
            case "APC7900Sim":
                pdu = new APC7900Sim(outlets);
                break;
            default:
                break;
        }

        if ( outlets.isEmpty() || outlets.size() != numberOfOutlets ) {
            throw new RuntimeException("Configuration parameter \"outlets\" must be specified");
        } 
        
        
        //At this point the outlets names must be available and we can set 
        //the node names correctly.        
        for ( int i = 0; i < numberOfOutlets; i++ ) {
            PduOutlet outlet = listOfOutlets.get(i);
            outlet.setName(outlets.get(i));
            mapOfOutlets.put(outlet.getName(), outlet);
            ComponentNode n = componentLookup.getComponentNodeForObject(outlet);
            componentLookup.setComponentNodeName(n, outlet.getName());
        }
        
        //Register the Pdu State for this component
        stateService.registerState(PduState.class, "The state of the Pdu", this);        
        //And set its initial state
        stateService.updateAgentComponentState(this, PduState.NOTCONFIGURED);
                
        for ( PduOutlet outlet : mapOfOutlets.values() ) {
            stateService.registerState(PduOutletState.class, "The state of a Pdu Outlet: ON/OFF", outlet);            
        }
        
        super.init();
    }

    @Override
    public void start() {
        //Add state change listener to raise alerts if/when outlets are turned off.
        stateService.addStateChangeListener(this, PduOutletState.class, PduState.class);
    }

    @Override
    public ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
        if ( alert.getAlertId().equals(PduProperties.PDU_ALERT) || alert.getAlertId().equals(PduProperties.PDU_OUTLET_ALERT) ) {
            return alertState != AlertState.NOMINAL ? ClearAlertCode.DONT_CLEAR_ALERT : ClearAlertCode.CLEAR_ALERT;
        }
        return ClearAlertCode.UNKNOWN_ALERT;
    }
    

    
    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice()
    {
        if (node == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "node", "is not specified");
        }
        fullName = "APC7900 PDU (" + node + ")";
    }


    /**
     *  Initializes the device.
     */
    @Override
    protected void initialize()
    {
        try {
            pdu.open(APC7900.ConnType.TELNET, node, user, passwd);
            Map<String,Integer> outNumMap = pdu.getOutletNumberMap();

            if ( outNumMap.size() != numberOfOutlets ) {
                throw new RuntimeException("Unexpected number of outlets: "+outNumMap.size()+" it should have been "+numberOfOutlets+".");
            }
            
            /**
             * Check if there is a mismatch between the outlet names on the PDU
             * and the ones provided by CCS.
             */
            StringBuilder sb = new StringBuilder("Name mismatch detected for outlets: ");
            boolean nameMismatchDetected = false;
            for (String outletName : outNumMap.keySet()) {
                int index = outNumMap.get(outletName);
                PduOutlet outlet = listOfOutlets.get(index-1);

                String providedOutletName = outlet.getName();
                if (!outletName.equals(providedOutletName)) {
                    if ( updateNamesInPDU ) {
                        LOG.warning("Updated the name of outlet "+index+" to "+providedOutletName+" (was "+outletName+")");
                        pdu.setOutletName(index, providedOutletName);
                    } else {
                        nameMismatchDetected = true;
                        sb.append(index).append(") PDU name: ").append(outletName).append(" from configuration: ").append(providedOutletName);
                    }
                }
            }
            if ( nameMismatchDetected ) {
                throw new RuntimeException(sb.toString());
            }

            initSensors();
            setOnline(true);
            stateService.updateAgentComponentState(this, PduState.OK);
            LOG.info("Connected to " + fullName);
        }
        catch (DriverException e) {
            if (!inited) {
                LOG.error("Error connecting to " + fullName + ": " + e);
            }
            close();
        }
        inited = true;
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
        try {
            pdu.close();
        }
        catch (DriverException e) {
        }
        stateService.updateAgentComponentState(this, PduState.NOTCONFIGURED);
    }

    @Override
    public void stateChanged(Object changedObj, Enum newState, Enum oldState) {
        if ( changedObj == this ) {
            alertService.raiseAlert(new Alert(PduProperties.PDU_ALERT,"PDU problem"), 
                    newState != PduState.OK ? AlertState.ALARM : AlertState.NOMINAL, "PDU "+getName()+" is in state "+newState);            
        }
    }

   /**
    *  Checks a channel's parameters for validity.
    *
    *  @param  name     The channel name
    *  @param  hwChan   The hardware channel
    *  @param  type     The channel type string
    *  @param  subtype  The channel subtype string
    *  @return  Two-element array containing the encoded type [0] and subtype [1] 
    *  @throws  Exception  If parameters are invalid
    */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception
    {
        Integer iType = typeMap.get(type.toUpperCase());
        int index = 0;
        if (iType == null) {
            MonitorLogUtils.reportError(LOG, name, "channel type", type);
        }
        if (hwChan != CHAN_CURRENT && hwChan != CHAN_POWER) {
            MonitorLogUtils.reportError(LOG, name, "hardware channel", hwChan);
        }
        
        return new int[]{iType | (index << 16), 0};
    }


    /**
     *  Reads a channel.
     *
     *  @param  hwChan  The hardware channel number
     *  @param  type    The encoded channel type
     *  @return  The read value
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = 0;
        switch (type & 0xffff) {
            case TYPE_POWER:
                try {
                    value = hwChan == CHAN_POWER ? pdu.readPower() : pdu.readCurrent()[0];
                    readFailures = 0;
                } catch (DriverException e) {
                    readFailures++;
                    if (readFailures >= maxReadFailures) {
                        LOG.error("Error reading power/current value: " + e);
                        setOnline(false);
                    } else {
                        LOG.fine("Failed to read APC Device ("+readFailures+")");
                    }
                }
        }
        return value;
    }

    @Override
    protected void readChannelGroup() {
        synchronized(intendedOutletStateLock) {            
            try {                
                //Issue the "status" command to get all the states in one go
                //and check the states here.
                verifyOutletsIntendedStates(pdu.getOutletOnStateMap());
            } catch (DriverException e) {
                LOG.warning("Exception in readChannelGroup", e);
            }
        }
    }
    
    /**
     * Read the state of the outlet from the driver and verify that this
     * state is similar to the intended state of the outlet (as set from a command).
     * If the outlet state is inconsistent, then an alert is raised.
     * @param outlet
     * @return
     * @throws DriverException 
     */
    private void verifyOutletsIntendedStates(Map<String,Boolean> states) throws DriverException {
        boolean needsToRaiseAlert = false;
        StateBundle sb = new StateBundle();
        synchronized(intendedOutletStateLock) {                
            for (Entry<String,Boolean> e : states.entrySet()) {
                String outletName = e.getKey();
                PduOutlet outlet = mapOfOutlets.get(outletName);
                PduOutletState state = e.getValue().booleanValue() ? PduOutletState.ON : PduOutletState.OFF;

                if (state != getIntendedOutletState(outlet, (PduOutletState) state)) {
                    if (outletsInAlertState.add(outlet)) {
                        needsToRaiseAlert = true;
                    }
                } else {
                    if (outletsInAlertState.remove(outlet)) {
                        needsToRaiseAlert = true;
                    }
                }
                sb.setComponentState(componentLookup.getComponentNodeForObject(outlet).getPath(), state);
            }
        }
        stateService.updateAgentState(sb);
        if (needsToRaiseAlert) {            
            String cause = null;
            AlertState alertState = null;
            if (outletsInAlertState.size() > 0) {
                cause = "Outlets in inconsistent states: ";
                for (PduOutlet o : outletsInAlertState) {
                    cause += o.getName() + " ";
                }
                alertState = AlertState.ALARM;
            } else {
                cause = "All outlets are in the intended state";
                alertState = AlertState.NOMINAL;
            }

            alertService.raiseAlert(new Alert(PduProperties.PDU_OUTLET_ALERT, "Outlet inconsistent state"),
                    alertState, cause);
        }            
    }
    
    /**
     *  Disables the PDU.
     *
     *  Needed to relinquish control to the web or another telnet client
     */
    @Override
    @Command(name="disable", description="Disable the connection to the PDU")
    public void disable()
    {
        super.disable();
        setOnline(false);
    }


    /**
     *  Enables the PDU.
     */
    @Override
    @Command(name="enable", description="Enable the connection to the PDU")
    public void enable()
    {
        super.enable();
    }


    /**
     *  Gets the list of outlet names.
     *
     *  @return  The list of names
     */
    @Command(name="getOutletNames", description="Get the list of outlet names")
    public List<String> getOutletNames()
    {
        List<String> result = new ArrayList<>();
        for ( PduOutlet outlet: listOfOutlets ) {
            result.add(outlet.getName());
        }
        return result;
    }


    /**
     *  Gets the map of outlet on states.
     *
     *  @return  The map of outlet names to on states (true or false)
     *  @throws  DriverException
     */
    @Command(name="getOutletOnStateMap", description="Get the map of outlet on states")
    public Map<String, Boolean> getOutletOnStateMap() throws DriverException
    {
        return pdu.getOutletOnStateMap();
    }


    /**
     *  Turns an outlet on.
     *
     *  @param  name  The outlet name
     *  @throws  DriverException
     */
    @Command(name="outletOn", description="Turn outlet on")
    public void outletOn(@Argument(name="name", description="Outlet name")
                         String name) throws DriverException
    {
        PduOutlet outlet = mapOfOutlets.get(name);
        if ( outlet == null ) {
            throw new IllegalArgumentException("Outlet name: "+name+" is illegal. The chosen outlet does not exist.");
        }
        synchronized(intendedOutletStateLock) {
            pdu.delayedOutletOn(getOutletNumber(name));        
            setIntendedOutletState(outlet, PduOutletState.ON);
        }
    }


    /**
     *  Turns an outlet off.
     *
     *  @param  name  The outlet name
     *  @throws  DriverException
     */
    @Command(name="outletOff", description="Turn outlet off")
    public void outletOff(@Argument(name="name", description="Outlet name")
                          String name) throws DriverException
    {
        PduOutlet outlet = mapOfOutlets.get(name);
        if ( outlet == null ) {
            throw new IllegalArgumentException("Outlet name: "+name+" is illegal. The chosen outlet does not exist.");
        }
        synchronized(intendedOutletStateLock) {
            pdu.delayedOutletOff(getOutletNumber(name));
            setIntendedOutletState(outlet, PduOutletState.OFF);
        }
    }


    /**
     *  Forces an outlet to turn on.
     *
     *  @param  name  The outlet name
     *  @throws  DriverException
     */
    @Command(name="forceOutletOn", description="Force outlet to turn on")
    public void forceOutletOn(@Argument(name="name", description="Outlet name")
                              String name) throws DriverException
    {
        PduOutlet outlet = mapOfOutlets.get(name);
        if ( outlet == null ) {
            throw new IllegalArgumentException("Outlet name: "+name+" is illegal. The chosen outlet does not exist.");
        }
        synchronized(intendedOutletStateLock) {       
            pdu.setOutletOn(getOutletNumber(name));
            setIntendedOutletState(outlet, PduOutletState.ON);
        }
    }


    /**
     *  Forces an outlet to turn off.
     *
     *  @param  name  The outlet name
     *  @throws  DriverException
     */
    @Command(name="forceOutletOff", description="Force outlet to turn off")
    public void forceOutletOff(@Argument(name="name", description="Outlet name")
                               String name) throws DriverException
    {
        PduOutlet outlet = mapOfOutlets.get(name);
        if ( outlet == null ) {
            throw new IllegalArgumentException("Outlet name: "+name+" is illegal. The chosen outlet does not exist.");
        }
        synchronized(intendedOutletStateLock) {
            pdu.setOutletOff(getOutletNumber(name));
            setIntendedOutletState(outlet, PduOutletState.OFF);
        }
    }

    @Command(name="causeUnintendedOutletFlipOfState", description="Cause an unintended state transition which raises an alert.", simulation = true )
    public void causeUnintendedOutletFlipOfState(@Argument(name="name", description="Outlet name")
                               String name) throws DriverException
    {
        PduOutlet outlet = mapOfOutlets.get(name);
        if ( outlet == null ) {
            throw new IllegalArgumentException("Outlet name: "+name+" is illegal. The chosen outlet does not exist.");
        }
        if ( isOutletOn(name) ) {
            pdu.setOutletOff(outlet.getIndex());
        } else {
            pdu.setOutletOn(outlet.getIndex());            
        }
    }
    
    @Command(name="causePDUFailure", description="Cause a PDU failure.", simulation = true )
    public void causePDUFailure() throws DriverException {
        setOnline(false);
    }

    /**
     *  Gets whether an outlet is on.
     *
     *  @param  name  The outlet name
     *  @return  Whether outlet is turned on
     *  @throws  DriverException
     */
    @Command(name="isOutletOn", description="Get whether outlet is on")
    public boolean isOutletOn(@Argument(name="name", description="Outlet name")
                              String name) throws DriverException
    {
        return pdu.isOutletOn(getOutletNumber(name));
    }


    /**
     *  Change the state of an outlet
     *
     *  @param name  The outlet name
     *  @param state The desired PduOutletState for the outlet.
     *  @param force true if the state change is to be forced.
     *  @throws  DriverException
     */
    @Command(name="setOutletState", description="Set the outlet state to a given OutletState")
    public void setOutletState(
            @Argument(allowedValueProvider = "getOutletNames", name="name", description="Outlet name") String name, 
            @Argument(name="state", description="The desired PduOutletState") PduOutletState state, 
            @Argument(name="force", description="true to force the state change", defaultValue = "false") boolean force 
            ) throws DriverException
    {
        PduOutlet outlet = mapOfOutlets.get(name);
        if ( outlet == null ) {
            throw new IllegalArgumentException("Outlet name: "+name+" is illegal. The chosen outlet does not exist.");
        }
        
        synchronized(intendedOutletStateLock) {
            if (force) {
                if (state == PduOutletState.OFF) {
                    pdu.setOutletOff(name);
                } else {
                    pdu.setOutletOn(name);
                }
            } else {
                if (state == PduOutletState.OFF) {
                    pdu.delayedOutletOff(name);
                } else {
                    pdu.delayedOutletOn(name);
                }
            }
            setIntendedOutletState(outlet, state);
        }
    }

    
    
    /**
     *  Gets an outlet number from its name.
     *
     *  This routine should be unnecessary since the PDU handles names directly,
     *  but there's a bug in the PDU parser which causes it to report an error
     *  if the name is hyphenated.
     *
     *  @param  name  The outlet name
     *  @return  The outlet number
     *  @throws  DriverException
     */
    int getOutletNumber(String name) throws DriverException
    {
        for ( PduOutlet outlet : listOfOutlets) {
            if ( outlet.getName().equals(name) ) {
                return outlet.getIndex();
            }
        }
        throw new DriverException("Invalid outlet name "+name);
    }    
    
    private void setIntendedOutletState(PduOutlet outlet, PduOutletState intendedState) throws DriverException {
        synchronized(intendedOutletStateLock) {
            intendedOutletStates.put(outlet,intendedState);
            stateService.updateAgentComponentState(outlet, intendedState);
        }
    }
    
    //If there isn't an intended outlet state, then the intended state becomes
    //the current state of the outlet
    private PduOutletState getIntendedOutletState(PduOutlet outlet, PduOutletState currentState) {
        synchronized(intendedOutletStateLock) {
            PduOutletState intendedState = intendedOutletStates.getOrDefault(outlet, currentState);
            intendedOutletStates.put(outlet, intendedState);
            return intendedState;
        }
    }
}
