package org.lsst.ccs.subsystem.common.ui.focalplane.view;

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.GridBagLayout;
import java.awt.Point;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.*;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.SwingUtilities;
import javax.swing.table.AbstractTableModel;
import org.lsst.ccs.gconsole.annotations.services.persist.Create;
import org.lsst.ccs.gconsole.base.Const;
import org.lsst.ccs.gconsole.base.filter.AgentChannelsFilter;
import org.lsst.ccs.gconsole.plugins.monitor.AbstractMonitorView;
import org.lsst.ccs.gconsole.plugins.monitor.DisplayChannel;
import org.lsst.ccs.gconsole.plugins.monitor.DisplayChannelSingle;
import org.lsst.ccs.gconsole.plugins.monitor.FormattedValue;
import org.lsst.ccs.gconsole.plugins.monitor.MonitorCell;
import org.lsst.ccs.gconsole.plugins.monitor.MonitorField;
import org.lsst.ccs.gconsole.plugins.monitor.MonitorFormat;
import org.lsst.ccs.gconsole.plugins.monitor.MonitorTable;
import org.lsst.ccs.gconsole.plugins.monitor.MonitorTableCellRenderer;
import org.lsst.ccs.gconsole.plugins.monitor.MonitorView;
import org.lsst.ccs.gconsole.plugins.monitor.SectionedTable;
import org.lsst.ccs.gconsole.plugins.monitor.Updatable;
import org.lsst.ccs.gconsole.services.aggregator.AgentChannel;
import org.lsst.ccs.gconsole.util.ThreadUtil;
import org.lsst.ccs.gconsole.util.swing.SComboBox;
import org.lsst.ccs.subsystem.common.ui.focalplane.Segment;
import static org.lsst.ccs.subsystem.common.ui.focalplane.Segment.*;
import org.lsst.ccs.subsystem.common.ui.focalplane.filter.FocalPlaneFilter;

/**
 * {@link MonitorView} that displays focal plane schematics as a hierarchy of tables.
 *
 * @author onoprien
 */
public class DiagramView extends AbstractMonitorView implements FocalPlaneView {

// -- Fields : -----------------------------------------------------------------
    
    static private MonitorField DEF = new MonitorField(null, "Default", null);
    
    private final Descriptor descriptor;
    
    private List<MonitorField> filterGroupFields; // fields for groups, as requested by filter
    private List<MonitorField> filterPlainFields; // fields to display for each channel in plain table, as requested by filter
    private List<String> groups; // groups
    
    private JPanel rootPanel;
    private JPanel centerPanel;
    private JButton backButton;
    private JComboBox<String> groupCombo;
    private JComboBox<MonitorField> fieldCombo;
    private Table table;
    private SectionedTable plainTable;
    private JScrollPane plainTableScrollPane;
    private JLabel statusLabel;
    
    private boolean armed = true; // set to false while resetting channels
    
    private Segment level = RAFT; // current table granularity level
    private MonitorField field = MonitorField.AVERAGE_VALUE; // currently displayed field
    private int[] indices; // currently sepected segment (parent of current table)
    private String statusPrefix; // path prefix the currently selected segment, for us in status string
    private String group; // currently displayed group
    private List<DisplayChannel> plainTableChannels;
    
// -- Life cycle : -------------------------------------------------------------
    
    @Create(category = FocalPlaneView.CATEGORY,
            name = "Focal Plane Diagram",
            path = "Built-In/Diagram",
            description = "Focal plane hierarchical diagram.")
    public DiagramView() {
        descriptor = new Descriptor();
    }
    
    public DiagramView(Descriptor descriptor) {
        this.descriptor = descriptor.clone();
        indices = decodeSegment(this.descriptor.getSegment());
        group = this.descriptor.getGroup();
    }
    
    @Override
    public JComponent getPanel() {
        if (rootPanel == null) {
            constructGUI();
        }
        return rootPanel;
    }

    @Override
    public FocalPlaneFilter getFilter() {
        return (FocalPlaneFilter) super.getFilter();
    }

    @Override
    public void setFilter(AgentChannelsFilter filter) {
        if (!(filter instanceof FocalPlaneFilter)) throw new IllegalArgumentException(getClass().getName() +" only accepts FocalPlaneFilter filters");
        super.setFilter(filter);
    }

    @Override
    public void install() {
        ThreadUtil.invokeLater(this::start);
        super.install();
    }

    @Override
    public void uninstall() {
        super.uninstall();
        ThreadUtil.invokeLater(this::stop);
    }
    
    private void start() {
        if (filter == null) throw new IllegalStateException("Trying to initialize FocalPlaneDiagramView without a filter.");
        List<String> attributes = filter.getFields(false);
        filterPlainFields = attributes == null ? MonitorTable.DEFAULT_FIELDS : new ArrayList<>(MonitorField.getDefaultFields(attributes));
    }
    
    private void stop() {
        if (table != null) {
            table.getModel().destroy();
            table = null;
        }
    }
    
    private void constructGUI() {
        
        // Top panel :
        
        rootPanel = new JPanel(new BorderLayout());
        
        // Controls :
        
        Box top = Box.createHorizontalBox();
        rootPanel.add(top, BorderLayout.NORTH);
        top.setBorder(BorderFactory.createEtchedBorder());
        backButton = new JButton("Back");
        top.add(backButton);
        backButton.addActionListener(e -> upLevel());
        top.add(Box.createRigidArea(Const.HDIM));
        top.add(new JLabel("Channel group: "));
        groupCombo = new JComboBox<>();
        groupCombo.addActionListener(e -> {
            int i = groupCombo.getSelectedIndex();
            if (armed || i >= 0) {
                group = groups.get(i);
                descriptor.setGroup(group);
                if (filterGroupFields != null && filterGroupFields.get(i) != null) {
                    field = filterGroupFields.get(i);
                    armed = false;
                    fieldCombo.setSelectedItem(field);
                    armed = true;
                    descriptor.setField(null);
                }
                rebuildTable();
            }
        });
        top.add(groupCombo);
        top.add(Box.createRigidArea(Const.HDIM));
        top.add(new JLabel("Field: "));
        fieldCombo = new JComboBox<>();
        fieldCombo.setToolTipText("What to display for this channel");
        fieldCombo.addActionListener(e -> {
            if (armed) {
                field = (MonitorField) fieldCombo.getSelectedItem();
                if (field == DEF || field == null) {
                    descriptor.setField(null);
                    if (filterGroupFields != null) {
                        int i = groupCombo.getSelectedIndex();
                        field = filterGroupFields.get(i);
                        if (field == null) field = MonitorField.DEFAULT_GROUP_FIELD;
                    } else {
                        field = MonitorField.DEFAULT_GROUP_FIELD;
                    }
                    armed = false;
                    fieldCombo.setSelectedItem(field);
                    armed = true;
                } else {
                    descriptor.setField(field.getTitle());
                }
                table.getModel().clear();
            }
        });
        top.add(fieldCombo);
        top.add(Box.createRigidArea(Const.HDIM));
        top.add(Box.createHorizontalGlue());
        
        // Diagram table :

        centerPanel = new JPanel(new GridBagLayout());
        rootPanel.add(centerPanel, BorderLayout.CENTER);
        centerPanel.setMinimumSize(new Dimension(15,15));
        centerPanel.setOpaque(true);
        table = new Table();
        centerPanel.add(table);
        
        // Plain table:
        
        plainTableScrollPane = new JScrollPane();
        
        // Status string :
        
        statusLabel = new JLabel();
        rootPanel.add(statusLabel, BorderLayout.SOUTH);
        statusLabel.setBorder(BorderFactory.createEtchedBorder());
    }
    
    
// -- Updates : ----------------------------------------------------------------

    /**
     * Called in response to changes in the list of display channels.
     * Completely resets the view, respecting the current descriptor.
     */
    @Override
    protected void resetChannels() {
        armed = false;
        
        // Compile the list of groups and update groupCombo accordingly:
        
        groups = getGroups();
        groupCombo.setModel(new DefaultComboBoxModel<>(groups.toArray(new String[0])));
        if (groups.isEmpty()) {
            groupCombo.setEnabled(false);
        } else {
            groupCombo.setEnabled(true);
            group = descriptor.getGroup();
            if (group == null || !groups.contains(group)) {
                group = groups.get(0);
                descriptor.setGroup(group);
                descriptor.setField(null);
            }
            groupCombo.setSelectedItem(group);
        }
        SComboBox.setPrototypeDisplayValue(groupCombo);
        
        // Compile the list of fields and update fieldCombo accordingly:
        
        if (groups.isEmpty()) {
            
            fieldCombo.setModel(new DefaultComboBoxModel<>(new MonitorField[0]));
            fieldCombo.setEnabled(false);
            
        } else {
            
            filterGroupFields = getGroupFields();
            if (filterGroupFields != null && filterGroupFields.size() != groups.size()) {
                filterGroupFields = null;
            }
            
            if (filterGroupFields == null) {
                fieldCombo.setModel(new DefaultComboBoxModel<>(MonitorField.GROUP_FIELDS));
            } else {
                MonitorField[] ff = new MonitorField[MonitorField.GROUP_FIELDS.length + 1];
                ff[0] = DEF;
                for (int i=0; i<MonitorField.GROUP_FIELDS.length; i++) {
                    ff[i+1] = MonitorField.GROUP_FIELDS[i];
                }
                fieldCombo.setModel(new DefaultComboBoxModel<>(ff));
            }
            SComboBox.setPrototypeDisplayValue(fieldCombo);
            
            String descField = descriptor.getField();
            if (descField == null) {
                if (filterGroupFields == null) {
                    field = MonitorField.DEFAULT_GROUP_FIELD;
                } else {
                    int i = groupCombo.getSelectedIndex();
                    field = filterGroupFields.get(i);
                    if (field == null) field = MonitorField.DEFAULT_GROUP_FIELD;
                }
            } else {
                field = MonitorField.stringToField(descField);
            }
            fieldCombo.setSelectedItem(field);
        }
        
        // Set currently selected segment:
        
        if (groups.isEmpty()) {
            indices = decodeSegment(null);
        } else {
            indices = decodeSegment(descriptor.getSegment());
        }
        
        // Replace model:
        
        rebuildTable();
        armed = true;
    }

    /**
     * Called in response to monitored data changes.
     * Notifies the table that the model has changed.
     */
    @Override
    protected void update() {
        if (plainTableChannels == null) {
            if (table != null) {
                table.getModel().fireTableDataChanged();
            }
        } else {
            for (DisplayChannel dc : plainTableChannels) {
                dc.update(null);
            }
        }
    }
    
    /**
     * Called in response to changes in selected segment or group.
     * Replaces model based on current settings.
     */
    private void rebuildTable() {
        
        // destroy previous table models
        
        table.getModel().destroy();
        if (plainTable != null) {
            plainTable.destroy();
            plainTable = null;
            plainTableChannels = null;
        }
        
        // try to create diagram model

        Model model;
        if (indices[0] == -1) {
            level = RAFT;
            model = buildModel();
            if (model == null) {
                level = REB;
                model = buildModel();
            }
        } else if (indices[3] == -1) {
            if (indices[0]%4 == 0 && indices[1]%4 == 0 && indices[2] != -1) {
                level = null;
                model = null;
            } else {
                level = CCD;
                model = buildModel();
            }
        } else if (indices[5] == -1) {
             level = AMP;
             model = buildModel();
        } else {
            level = null;
            model = null;
        }
        
        // display diagram or plain table
        
        if (model == null) {
            if (!displayPlainTable()) {
                upLevel();
                rebuildTable();
            }
        } else {
            table.setModel(model);
            if (centerPanel.getParent() == null) {
                rootPanel.remove(plainTableScrollPane);
                rootPanel.add(centerPanel, BorderLayout.CENTER);
                rootPanel.revalidate();
                rootPanel.repaint();
            }
        }
        
        // Enable back button if necessary
        
        backButton.setEnabled(indices[0] != -1);
        fieldCombo.setEnabled(plainTable == null);
        
        // Update status label
        
        String pref = Segment.getPathPrefix(indices);
        statusPrefix = "<html><b><font color=blue> "+ (pref.isEmpty() ? "/" : pref) +"</font></b>";
        statusLabel.setText(statusPrefix);
    }
    
    private void upLevel() {
        if (indices[5] != -1) {
            indices[4] = -1;
            indices[5] = -1;
        } else if (indices[2] != -1) {
            indices[2] = -1;
            indices[3] = -1;
        } else {
            indices[0] = -1;
            indices[1] = -1;
        }
        descriptor.setSegment(encodeSegment(indices));
        rebuildTable();
    }
    
    private boolean displayPlainTable() {
        String prefix = Segment.getPathPrefix(indices);
        int i = prefix.length();
        ArrayList<AgentChannel> channels = new ArrayList<>();
        for (Map.Entry<String, DisplayChannel> e : data.entrySet()) {
            String path = e.getKey();
            if (path.startsWith(prefix) && group.equals(getGroup(path))) {
                channels.addAll(e.getValue().getChannels());
            }
        }
        if (channels.isEmpty()) {
            return false;
        } else {
            TreeMap<String, DisplayChannel> tableData = new TreeMap<>();
            for (AgentChannel ch : channels) {
                tableData.put(ch.getPath(), new DisplayChannelSingle(ch.getPath(), ch, null));
            }
            plainTableChannels = new ArrayList<>(channels.size());
            for (Map.Entry<String,DisplayChannel> e : tableData.entrySet()) {
                plainTableChannels.add(e.getValue());
            }
            plainTable = SectionedTable.getInstance(tableData.entrySet(), filterPlainFields, null);
            plainTableScrollPane.setViewportView(plainTable.getTable());
            if (plainTableScrollPane.getParent() == null) {
                rootPanel.remove(centerPanel);
                rootPanel.add(plainTableScrollPane, BorderLayout.CENTER);
            }
            return true;
        }
    }

    @Override
    public List<String> getGroups() {
        List<String> out = FocalPlaneView.super.getGroups();
        if (out == null) out = super.getGroups();
        return out;
    }

    
// -- Local classes and related methods : --------------------------------------
    
    private final class Table extends JTable {
        
        private final Dimension prefSize = new Dimension(0,0);
        
        Table() {
            super(new Model(0, 0));
            setDefaultRenderer(FormattedValue.class, new MonitorTableCellRenderer());
            setCellSelectionEnabled(true);
            setShowGrid(true);
            setFillsViewportHeight(true);
            setBorder(BorderFactory.createLineBorder(gridColor));
            MouseAdapter mouseListener = new MouseAdapter() {
                @Override
                public void mouseClicked(MouseEvent e) {
                    mouseClick(e);
                }
                @Override
                public void mouseMoved(MouseEvent e) {
                    setSegmentString(e);
                }

                @Override
                public void mouseExited(MouseEvent e) {
                    statusLabel.setText(statusPrefix);
                }
            };
            addMouseMotionListener(mouseListener);
            addMouseListener(mouseListener);
        }

        @Override
        public Model getModel() {
            return (Model) super.getModel();
        }

        @Override
        public Dimension getPreferredSize() {
            Container parent = this.getParent();
            if (parent == null) {
                return new Dimension(0, 0);
            }
            int nRows = getRowCount();
            int w,h;
            if (level == AMP) {
                Dimension p = super.getPreferredSize();
                w = Math.min(p.width, parent.getWidth());
                h = Math.min(parent.getHeight(), w);
            } else {
                w = Math.min(parent.getWidth(), parent.getHeight());
                h = w;
            }
            if (nRows > 0) {
                int rowH = h / nRows;
                h = rowH * nRows;
                setRowHeight(rowH);
            }
            prefSize.height = h;
            prefSize.width = w;
            return prefSize;
        }
        
        private void mouseClick(MouseEvent e) {
            int nClick = e.getClickCount();
            Point point = e.getPoint();
            Table table = (Table) e.getSource();
            int x = table.rowAtPoint(point);
            int y = table.columnAtPoint(point);
            Cell cell = table.getModel().cells[x][y];
            if (cell.getChannels().isEmpty()) return;
            x = table.getRowCount() - x - 1;
            if (nClick == 2) {
                switch (level) {
                    case RAFT:
                        indices[0] = x;
                        indices[1] = y;
                        break;
                    case REB:
                        indices[0] = x / 3;
                        indices[1] = y;
                        if (indices[0] == 0 && (indices[1] == 0 || indices[1] == 4)) {
                            indices[2] = 2 - x;
                        } else {
                            indices[2] = x % 3;
                        }
                        break;
                    case CCD:
                        if (indices[0] == 0 && indices[1] == 0) { // raft 00
                            if (x == 2 && y == 2) {
                                indices[2] = 0;
                            } else {
                                indices[2] = 1;
                                indices[3] = x == 2 ? 1 : 0;
                            }
                        } else if (indices[0] == 0 && indices[1] == 4) { // raft 04
                            if (x == 2 && y == 0) {
                                indices[2] = 0;
                            } else {
                                indices[2] = 1;
                                indices[3] = x == 2 ? 0 : 1;
                            }
                        } else if (indices[0] == 4 && indices[1] == 0) { // raft 40
                            if (x == 0 && y == 2) {
                                indices[2] = 0;
                            } else {
                                indices[2] = 1;
                                indices[3] = x == 0 ? 0 : 1;
                            }
                        } else if (indices[0] == 4 && indices[1] == 4) { // raft 44
                            if (x == 0 && y == 0) {
                                indices[2] = 0;
                            } else {
                                indices[2] = 1;
                                indices[3] = x == 0 ? 1 : 0;
                            }
                        } else {
                            indices[2] = x;
                            indices[3] = y;
                        }
                        break;
                    case AMP:
                        // NO NEED TO SUPPORT CORNER RAFTS HERE - NO DATA
                        indices[4] = x;
                        indices[5] = y;
                        break;
                }
                descriptor.setSegment(encodeSegment(indices));
                rebuildTable();
            }
        }
        
        private void setSegmentString(MouseEvent e) {
            Point point = e.getPoint();
            Table table = (Table) e.getSource();
            int x = table.getRowCount() - table.rowAtPoint(point) - 1;
            int y = table.columnAtPoint(point);
            switch (level) {
                case RAFT:
                    statusLabel.setText(statusPrefix + RAFT.toString() + x + y);
                    break;
                case REB:
                    int raftX = x / 3;
                    x = x % 3;
                    int raftY = y;
                    String reb = null;
                    if (raftX == 0 && (raftY == 0 || raftY == 4)) {
                        if (x == 1) {
                            reb = "G";
                        } else if (x == 2) {
                            reb = "W";
                        }
                    } else if (raftX == 4 && (raftY == 0 || raftY == 4)) {
                        if (x == 1) {
                            reb = "G";
                        } else if (x == 0) {
                            reb = "W";
                        }
                    } else {
                        reb = Integer.toString(x);
                    }
                    StringBuilder sb = new StringBuilder(statusPrefix);
                    if (reb != null) {
                        sb.append(RAFT).append(raftX).append(raftY).append("/").append(REB).append(reb);
                    }
                    statusLabel.setText(sb.toString());
                    break;
                case CCD:
                    String s = null;
                    if (indices[0] == 0 && indices[1] == 0) { // raft 00
                        if (x == 2 && y == 2) {
                            s = REB +"W/"+ CCD +"W";
                        } else {
                            if (x == 2 && y == 1) {
                                s = REB +"G/"+ CCD +"G1";
                            } else if (x == 1 && y == 2) {
                                s = REB +"G/"+ CCD +"G0";
                            }
                        }
                    } else if (indices[0] == 0 && indices[1] == 4) { // raft 04
                        if (x == 2 && y == 0) {
                            s = REB.toString() + "W";
                        } else {
                            if (x == 1 && y == 0) {
                                s = REB +"G/"+ CCD +"G1";
                            } else if (x == 2 && y == 1) {
                                s = REB +"G/"+ CCD +"G0";
                            }
                        }
                    } else if (indices[0] == 4 && indices[1] == 0) { // raft 40
                        if (x == 0 && y == 2) {
                            s = REB.toString() + "W";
                        } else {
                            if (x == 1 && y == 2) {
                                s = REB +"G/"+ CCD +"G1";
                            } else if (x == 0 && y == 1) {
                                s = REB +"G/"+ CCD +"G0";
                            }
                        }
                    } else if (indices[0] == 4 && indices[1] == 4) { // raft 44
                        if (x == 0 && y == 0) {
                            s = REB +"W/"+ CCD +"W";
                        } else {
                            if (x == 0 && y == 1) {
                                s = REB +"G/"+ CCD +"G1";
                            } else if (x == 1 && y == 0) {
                                s = REB +"G/"+ CCD +"G0";
                            }
                        }
                    } else {
                        s = REB.toString() + x +"/"+ CCD + x + y;
                    }
                    statusLabel.setText(s == null ? statusPrefix : statusPrefix + s);
                    break;
                case AMP:
                    // NO NEED TO SUPPORT CORNER RAFTS HERE - NO DATA
                    statusLabel.setText(statusPrefix + AMP + x + y);
                    break;
            }
        }
    }
    
    private class Model extends AbstractTableModel {
        
        int nRows, nColumns; 
        Cell[][] cells;
        
        Model(int rows, int columns) {
            nRows = rows;
            nColumns = columns;
            cells = new Cell[nRows][nColumns];
            for (int row = 0; row < nRows; row++) {
                for (int col = 0; col < nColumns; col++) {
                    cells[row][col] = new Cell();
                }
            }
        }
        
        Model() {
            switch (level) {
                case RAFT:
                    nRows = 5;
                    nColumns = 5;
                    break;
                case REB:
                    nRows = 5*3;
                    nColumns = 5;
                    break;
                case CCD:
                    nRows = 3;
                    nColumns = 3;
                    break;
                case AMP:
                    boolean vertical = true;
                    if (indices[0] == 0) {
                        if (indices[1] == 0) {
                            if (indices[2] == 1 && indices[3] == 0) vertical = false;
                        } else if (indices[1] == 4) {
                            if (indices[2] == 0 || (indices[2] == 1 && indices[3] == 1)) vertical = false;
                        }
                    } else if (indices[0] == 4) {
                        if (indices[1] == 0) {
                            if (indices[2] == 0 || (indices[2] == 1 && indices[3] == 1)) vertical = false;
                        } else if (indices[1] == 4) {
                            if (indices[2] == 1 && indices[3] == 0) vertical = false;
                        }
                    }
                    nRows = vertical ? 2 : 8;
                    nColumns = vertical ? 8 : 2;
                    break;
            }
            cells = new Cell[nRows][nColumns];
            for (int row = 0; row < nRows; row++) {
                for (int col = 0; col < nColumns; col++) {
                    cells[row][col] = new Cell();
                }
            }
        }

        @Override
        public int getRowCount() {
            return nRows;
        }

        @Override
        public int getColumnCount() {
            return nColumns;
        }

        @Override
        public Object getValueAt(int rowIndex, int columnIndex) {
            MonitorCell cell = cells[rowIndex][columnIndex];
            FormattedValue fv = cell.getFormattedValue();
            if (fv == null) {
                MonitorFormat.DEFAULT.format(cell);
                fv = cell.getFormattedValue();
            }
            return fv;
        }

        @Override
        public Class<?> getColumnClass(int columnIndex) {
            return FormattedValue.class;
        }
        
        void destroy() {
            for (int row = 0; row < nRows; row++) {
                for (int col = 0; col < nColumns; col++) {
                    Cell cell = cells[row][col];
                    for (DisplayChannel dc : cell.getChannels()) {
                        dc.setTarget(null);
                    }
                }
            }
        }
        
        void clear() {
            for (int row = 0; row < nRows; row++) {
                for (int col = 0; col < nColumns; col++) {
                    cells[row][col].setFormattedValue(null);
                }
            }
            fireTableDataChanged();
        }
        
        void trimToSize() {
            for (int row = 0; row < nRows; row++) {
                for (int col = 0; col < nColumns; col++) {
                    cells[row][col].getChannels().trimToSize();
                }
            }
        }    
    }
    
    private class Cell implements MonitorCell, Updatable {
        
        private final ArrayList<DisplayChannel> channels;
        private FormattedValue formattedValue;
        
        Cell(ArrayList<DisplayChannel> channels) {
            this.channels = channels;
        }
        
        Cell() {
            this.channels = new ArrayList<>();
        }

        @Override
        public ArrayList<DisplayChannel> getChannels() {
            return channels;
        }

        @Override
        public MonitorField getField() {
            return field;
        }

        @Override
        public FormattedValue getFormattedValue() {
            return formattedValue;
        }

        @Override
        public void setFormattedValue(FormattedValue formattedValue) {
            if (formattedValue != null) {
                formattedValue.horizontalAlignment = SwingUtilities.CENTER;
            }
            this.formattedValue = formattedValue;
        }

        @Override
        public void update(DisplayChannel channelHandle) {
            formattedValue = null;
        }
        
    }
    
    private Model buildModel() {
        
        // Create model of appropriate size
        
        Model model = new Model();
        
        if (!data.isEmpty() && group != null) {
        
        // Create prefix string reflecting position of the parent of the current table
        
            int[] ind;
            if (indices[2] == -1) {
                ind = indices;
            } else {
                ind = Arrays.copyOf(indices, 6);
                ind[2] = -1;
            }
            int switchToReb = 0; // 0 - no REB or RAFT channels; 1 - has REB but no RAFT channels; 2 - has RAFT channels
            String pref = Segment.getPathPrefix(ind);
            
        // Fill the model
            
            for (Map.Entry<String, DisplayChannel> e : data.entrySet()) {
                String path = e.getKey();
                if (path.startsWith(pref) && group.equals(getGroup(path))) {
                    int[] ii = Segment.getIndices(path);
                    if (ii != null) {
                        try {
                            Cell cell = null;
                            switch (level) {
                                case RAFT:
                                    if (switchToReb < 2) {
                                        if (ii[2] == -1) {
                                            switchToReb = 2;
                                        } else if (ii[3] == -1) {
                                            switchToReb = 1;
                                        }
                                    }
                                    cell = model.cells[4 - ii[0]][ii[1]];
                                    break;
                                case REB:
                                    if (ii[2] == -1) {
                                        model.destroy();
                                        return null;
                                    }
                                    int row;
                                    if (ii[0] == 0 && (ii[1] == 0 || ii[1] == 4)) {
                                        row = 12 + ii[2];
                                    } else if (ii[0] == 4 && (ii[1] == 0 || ii[1] == 4)) {
                                        row = 2 - ii[2];
                                    } else {
                                        row = 14 - (ii[0] * 3 + ii[2]);
                                    }
                                    cell = model.cells[row][ii[1]];
                                    break;
                                case CCD:
                                    if (ii[3] == -1) {
                                        model.destroy();
                                        return null;
                                    }
                                    if (ii[0] == 0 && ii[1] == 0) { // Raft 00
                                        if (ii[2] == 0) {
                                            cell = model.cells[0][2];
                                        } else {
                                            cell = ii[3] == 0 ? model.cells[1][2] : model.cells[0][1];
                                        }
                                    } else if (ii[0] == 0 && ii[1] == 4) { // Raft 04
                                        if (ii[2] == 0) {
                                            cell = model.cells[0][0];
                                        } else {
                                            cell = ii[3] == 0 ? model.cells[0][1] : model.cells[1][0];
                                        }
                                    } else if (ii[0] == 4 && ii[1] == 0) { // Raft 40
                                        if (ii[2] == 0) {
                                            cell = model.cells[2][2];
                                        } else {
                                            cell = ii[3] == 0 ? model.cells[2][1] : model.cells[1][2];
                                        }
                                    } else if (ii[0] == 4 && ii[1] == 4) { // Raft 44
                                        if (ii[2] == 0) {
                                            cell = model.cells[2][0];
                                        } else {
                                            cell = ii[3] == 0 ? model.cells[1][0] : model.cells[2][1];
                                        }
                                    } else { // Science rafts
                                        cell = model.cells[2 - ii[2]][ii[3]];
                                    }
                                    break;
                                case AMP:
                                    if (ii[4] == -1) {
                                        model.destroy();
                                        return null;
                                    }
                                    if (ii[0] == 0 && ii[1] == 0) { // Raft 00
                                        if (ii[2] == 0) { // W
                                            cell = ii[3] == 0 ? model.cells[0][ii[5]] : model.cells[1][7-ii[5]];
                                        } else { // G
                                            cell = ii[3] == 0 ? model.cells[7-ii[5]][1-ii[4]] : model.cells[ii[4]][7-ii[5]];
                                        }
                                    } else if (ii[0] == 0 && ii[1] == 4) { // Raft 04
                                        if (ii[2] == 0) { // W
                                            cell = ii[3] == 0 ? model.cells[7-ii[5]][0] : model.cells[ii[5]][1];
                                        } else {
                                            cell = ii[3] == 0 ? model.cells[ii[4]][7-ii[5]] : model.cells[ii[5]][ii[4]];
                                        }
                                    } else if (ii[0] == 4 && ii[1] == 0) { // Raft 40
                                        if (ii[2] == 0) { // W
                                            cell = ii[3] == 0 ? model.cells[ii[5]][1] : model.cells[7-ii[5]][0];
                                        } else {
                                            cell = ii[3] == 0 ? model.cells[1-ii[4]][ii[5]] : model.cells[7-ii[5]][1-ii[4]];
                                        }
                                    } else if (ii[0] == 4 && ii[1] == 4) { // Raft 44
                                        if (ii[2] == 0) { // W
                                            cell = ii[3] == 0 ? model.cells[1][7-ii[5]] : model.cells[0][ii[5]];
                                        } else {
                                            cell = ii[3] == 0 ? model.cells[ii[5]][ii[4]] : model.cells[1-ii[4]][ii[5]];
                                        }
                                    } else { // Science rafts
                                        cell = model.cells[1 - ii[4]][ii[5]];
                                    }
                                    break;
                            }
                            DisplayChannel dc = e.getValue();
                            cell.getChannels().add(dc);
                            dc.setTarget(cell);
                        } catch (ArrayIndexOutOfBoundsException x) {
                        }
                    }
                }
            }
            
        // If switching to REB view is necessary, destroy the model and return null
            
            if (switchToReb == 1) {
                model.destroy();
                return null;
            }
        }

        model.trimToSize();
        return model;
    }
    
    
// -- Saving/restoring : -------------------------------------------------------
    
    static public class Descriptor extends FocalPlaneView.Descriptor {

        private String group; // currently selected group, if any
        private String field; // currently selected non-default field, if any
        private Integer segment; // encoded currently selected segment (parent of current table)

        public String getGroup() {
            return group;
        }

        public void setGroup(String group) {
            this.group = group;
        }

        public String getField() {
            return field;
        }

        public void setField(String field) {
            this.field = field;
        }

        public Integer getSegment() {
            return segment;
        }

        public void setSegment(Integer segment) {
            this.segment = segment;
        }

        @Override
        public Descriptor clone() {
            return (Descriptor) super.clone();
        }
        
    }

    @Override
    public Descriptor getDescriptor() {
        return descriptor;
    }

    @Override
    public Descriptor save() {
        descriptor.setSegment(indices == null ? null : encodeSegment(indices));
        descriptor.setGroup(group);
        return descriptor.clone();
    }
    
    private Integer encodeSegment(int[] indices) {
        int out = 0;
        int shift = 1;
        for (int i=0; i<4; i++) {
            int k = indices[i];
            if (k == -1) k = 9;
            out += k * shift;
            shift *= 10;
        }
        return out == 9999 ? null : out;
    }
    
    private int[] decodeSegment(Integer code) {
        int[] out = {-1, -1, -1, -1, -1, -1};
        if (code == null) return out;
        for (int i=0; i<4; i++) {
            int k = code % 10;
            if (k != 9) out[i] = k;
            code /= 10;
        }
        return out;
    }

}
