package org.lsst.ccs.subsystem.common.ui.focalplane.fpmap;

import org.lsst.ccs.gconsole.services.persist.Savable;

/**
 * A data model for {@link FocalPlaneMap}.
 * <p>
 * When {@link FocalPlaneMap} paints itself, it calls {@code getValue(...)} methods for each cell
 * in order of increasing granularity (RAFT - REB - CCD - Amplifier), stopping at the first method
 * that returns either a non-split value (which is displayed) or {@code null} (the default value is
 * displayed). The default value is the value for the closest ancestor cell that is not {@code NONE}.
 * <p>
 * The numbering scheme for RAFTs, REBs, CCDs, and amplifiers follows the standard described in 
 * <a href="https://docushare.lsst.org/docushare/dsweb/Get/LCA-13381">LCA-13381</a>.
 * {@link FocalPlaneMap} documentation describes the view orientation.
 * 
 * <p>
 * All methods are called on EDT.
 *
 * @author onoprien
 */
public interface FocalPlaneMapModel extends Savable {
    
    /**
     * Returns the title to be displayed by {@code FocalPlaneMap}.
     * @return Title, or {@code null} is this model does not provide a title.
     */
    default String getTitle() {
        return null;
    }
    
    /**
     * Adds a listener to be notified of changes in this model.
     * @param listener Listener to be registered.
     */
    void addFocalPlaneMapModelListener(FocalPlaneMapModelListener listener);
    
    /**
     * Removes a model listener.
     * @param listener Listener to be removed, or {@code null} if all listeners should be removed.
     */
    void removeFocalPlaneMapModelListener(FocalPlaneMapModelListener listener);
    
    /**
     * Returns the default value for cells for which the value is not otherwise specified.
     * This value is used unless one of the more specific {@code getValue(...)} methods overrides it for a specific display cell.
     * The default implementation returns {@code EMPTY}.
     * 
     * @return Value to be displayed, or {@code null} if the default value should be used for this raft.
     */
    default FocalPlaneMapValue getValue() {
        return FocalPlaneMapValue.EMPTY;
    }
    
    /**
     * Returns the value for the specified RAFT cell.
     * The default implementation returns {@code NONE}.
     * 
     * @param raftX RAFT X index. Range 0 to 4.
     * @param raftY RAFT Y index. Range 0 to 4.
     * @return Value to be displayed, or {@code null} if the default value should be used for this raft.
     */
    default FocalPlaneMapValue getValue(int raftX, int raftY) {
        return FocalPlaneMapValue.NONE;
    }
    
    /**
     * Returns the value for the specified REB cell.
     * The default implementation returns {@code NONE}.
     * 
     * @param raftX RAFT X index. Range 0 to 4.
     * @param raftY RAFT Y index. Range 0 to 4.
     * @param reb REB index. Range 0 to 2.
     * @return Value to be displayed, or {@code null} if the default value should be used for this REB.
     */
    default FocalPlaneMapValue getValue(int raftX, int raftY, int reb) {
        return FocalPlaneMapValue.NONE;
    }
    
    /**
     * Returns the value for the specified CCD cell.
     * The default implementation returns {@code NONE}.
     * 
     * @param raftX RAFT X index. Range 0 to 4.
     * @param raftY RAFT Y index. Range 0 to 4.
     * @param reb REB index. Range 0 to 2.
     * @param ccdY CCD Y index. Range 0 to 2.
     * @return Value to be displayed, or {@code null} if the default value should be used for this CCD.
     */
    default FocalPlaneMapValue getValue(int raftX, int raftY, int reb, int ccdY) {
        return FocalPlaneMapValue.NONE;
    }
    
    /**
     * Returns the value for the specified CCD cell.
     * The default implementation returns {@code null}.
     * 
     * @param raftX RAFT X index. Range 0 to 4.
     * @param raftY RAFT Y index. Range 0 to 4.
     * @param reb REB index. Range 0 to 2.
     * @param ccdY CCD Y index. Range 0 to 2.
     * @param ampX Amplifier X index. Range 0 to 7.
     * @param ampY Amplifier Y index. Range 0 to 1.
     * @return Value to be displayed, or {@code null} if the default value should be used for this CCD.
     */
    default FocalPlaneMapValue getValue(int raftX, int raftY, int reb, int ccdY, int ampX, int ampY) {
        return null;
    }
    
}
