package org.lsst.ccs.subsystem.common.ui.focalplane.view;

import java.awt.Component;
import java.awt.GridLayout;
import java.awt.Point;
import java.util.ArrayList;
import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import org.freehep.swing.popup.HasPopupItems;
import org.lsst.ccs.gconsole.annotations.services.persist.Create;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.Const;
import org.lsst.ccs.gconsole.base.filter.AgentChannelsFilter;
import org.lsst.ccs.gconsole.plugins.monitor.MonitorView;
import org.lsst.ccs.gconsole.services.persist.PersistenceService;
import org.lsst.ccs.subsystem.common.ui.focalplane.filter.FocalPlaneFilter;

/**
 * {@link MonitorView} that displays a grid of {@link MapView}.
 *
 * @author onoprien
 */
public class MultiMapView implements FocalPlaneView {

// -- Fields : -----------------------------------------------------------------
    
    private final Descriptor descriptor;
    
    private FocalPlaneFilter filter;
    private Page page;
    private ArrayList<MapView> views;
    
// -- Life cycle : -------------------------------------------------------------
    
     public MultiMapView() {
         this(1, 1);
     }
    
    @Create(category = FocalPlaneView.CATEGORY,
            name = "Focal plane multi-region map",
            path = "Built-In/Map/Multi-Region",
            description = "A view that contains a grid of focal plane maps.")
     public MultiMapView(int rows, int columns) {
         descriptor = new Descriptor();
         descriptor.setRows(rows);
         descriptor.setColumns(columns);
     }
    
    public MultiMapView(Descriptor descriptor) {
        this.descriptor = descriptor.clone();
    }

    @Override
    public FocalPlaneFilter getFilter() {
        return filter;
    }

    @Override
    public void setFilter(AgentChannelsFilter filter) {
        this.filter = (FocalPlaneFilter) filter;
    }

    @Override
    public JComponent getPanel() {
        if (page == null) {
            page = new Page();
        }
        return page;
    }

    @Override
    public void install() {
        getPanel();
        if (views != null) {
            for (MapView v : views) {
                if (v != null) {
                    v.install();
                }
            }
        }
    }

    @Override
    public void uninstall() {
        if (views != null) {
            for (MapView v : views) {
                if (v != null) {
                    v.uninstall();
                }
            }
        }
    }


// -- Graphical component : ----------------------------------------------------
    
    final class Page extends JPanel implements HasPopupItems {

        Page() {
            setLayout(new GridLayout(descriptor.getRows(), descriptor.getColumns(), Const.HSPACE, Const.VSPACE));
            PersistenceService service = Console.getConsole().getSingleton(PersistenceService.class);
            MapView.Descriptor[] regionDescriptors = descriptor.getViews();
            int n = descriptor.getRows() * descriptor.getColumns();
            views = new ArrayList<>(n);
            for (int i = 0; i < n; i++) {
                MapView.Descriptor regionDesc = regionDescriptors == null ? null : regionDescriptors[i];
                if (regionDesc == null) {
                    regionDesc = new MapView.Descriptor();
                }
                MapView view = (MapView) service.make(regionDesc);
                if (filter != null && view.getFilter() == null) {
                    view.setFilter(filter);
                }
                views.add(view);
                add(view.getPanel());
            }
        }
    
        /**
         * Forwards the call to a {@code FocalPlaneMapRegion}, if any, allowing it to modify the menu.
         *
         * @param menu Original menu
         * @param component Component.
         * @param point Position.
         * @return Modified menu.
         */
        @Override
        public JPopupMenu modifyPopupMenu(JPopupMenu menu, Component component, Point point) {
            Component c = getComponentAt(point);
            if (c instanceof HasPopupItems) {
                ((HasPopupItems)c).modifyPopupMenu(menu, component, point);
            }
            return menu;
        }

    }
    
    
// -- Saving/restoring : -------------------------------------------------------
    
    static public class Descriptor extends FocalPlaneView.Descriptor {

        private int rows = 1;
        private int columns = 1;
        private MapView.Descriptor[] views;

        public int getRows() {
            return rows;
        }

        public void setRows(int rows) {
            this.rows = rows;
        }

        public int getColumns() {
            return columns;
        }

        public void setColumns(int columns) {
            this.columns = columns;
        }

        public MapView.Descriptor[] getViews() {
            return views;
        }

        public void setViews(MapView.Descriptor[] views) {
            this.views = views;
        }

        @Override
        public Descriptor clone() {
            Descriptor desc = (Descriptor) super.clone();
            if (views != null) {
                int n = views.length;
                desc.views = new MapView.Descriptor[n];
                for (int i=0; i<n; i++) {
                    MapView.Descriptor d = views[i];
                    desc.views[i] = d == null ? null : d.clone();
                }
            }
            return desc;
        }
        
    }

    @Override
    public Descriptor getDescriptor() {
        return descriptor;
    }

    @Override
    public Descriptor save() {
        Descriptor desc = descriptor.clone();
        if (views != null) {
            int n = desc.getRows() * desc.getColumns();
            MapView.Descriptor[] vd = new MapView.Descriptor[n];
            for (int i = 0; i < n; i++) {
                MapView v = views.get(i);
                if (v != null) {
                    vd[i] = (MapView.Descriptor) v.save();
                }
            }
            desc.setViews(vd);
        }
        return desc;
    }

}
