package org.lsst.ccs.subsystem.common.ui.focalplane.filter;

import org.lsst.ccs.subsystem.common.ui.focalplane.Segment;
import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.util.*;
import java.util.concurrent.CancellationException;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.DefaultComboBoxModel;
import javax.swing.DefaultListCellRenderer;
import javax.swing.DefaultListModel;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.SwingUtilities;
import javax.swing.border.EtchedBorder;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.Const;
import org.lsst.ccs.gconsole.plugins.monitor.MonitorField;
import org.lsst.ccs.gconsole.services.aggregator.AgentChannel;
import org.lsst.ccs.gconsole.services.persist.PersistenceService;
import org.lsst.ccs.subsystem.common.ui.focalplane.fpmap.FocalPlaneMap;
import org.lsst.ccs.utilities.conv.TypeUtils;

/**
 * Dialog that configures filters for populating {@link FocalPlaneMap}.
 *
 * @author onoprien
 */
public final class GroupingTemplateFilterDialog extends JDialog {

// -- Fields : -----------------------------------------------------------------
    
    private GroupingTemplateFilter.Descriptor descriptor;
    private boolean other = false;
    
    private String[] agents;
    private ArrayList<Template> templates;
    
    private final JTextField nameField;
    private final JTextField agentsField;
    private final JComboBox<String> subsystemCombo;
    private final JComboBox<Segment> granularityComboIn;
    private final JList<Template> templatesIn;
    private JButton addButton, removeButton;
    private final JComboBox<String> groupCombo = new JComboBox<>();
    private final JButton upGroupButton, downGroupButton, addGroupButton, deleteGroupButton, renameGroupButton;
    private final JList<Template> templatesOut = new JList<>();
    private final HashMap<String,DefaultListModel<Template>> templatesOutModels = new HashMap<>();
    private final MonitorField[] fields = {MonitorField.AVERAGE_VALUE, MonitorField.MAX_VALUE, MonitorField.MIN_VALUE, MonitorField.MEDIAN_VALUE};
    private final JComboBox<MonitorField> fieldCombo = new JComboBox<>(fields);
    private final ArrayList<Integer> selectedFields = new ArrayList<>();
    
    private final String ANY = "ANY";

// -- Life cycle : -------------------------------------------------------------

    private GroupingTemplateFilterDialog(GroupingTemplateFilter.Descriptor before, Component parentComponent) {
        super(parentComponent == null ? null : SwingUtilities.getWindowAncestor(parentComponent), "Configure filter", ModalityType.APPLICATION_MODAL);
        descriptor = before == null ? new GroupingTemplateFilter.Descriptor() : before;
        
        List<AgentChannel> all = Console.getConsole().getStatusAggregator().getDictionaryChannels(null, Collections.singletonList("[^/]+/"+ Segment.RAFT +"[0-4]{2}/.+"));
        TreeSet<String> aa = new TreeSet<>();
        TreeSet<Template> tt = new TreeSet<>();
        all.forEach(channel -> {
            aa.add(channel.getAgentName());
            Template t = Template.valueOf(channel);
            if (t != null)  tt.add(t);
        });
        ArrayList<String> alist = new ArrayList(aa.size()+1);
        alist.add(ANY);
        alist.addAll(aa);
        agents = alist.toArray(new String[0]);
        templates = new ArrayList<>(tt);
        
        // Root panel:
        
        Box rootPanel = Box.createVerticalBox();
        add(rootPanel, BorderLayout.CENTER);
        rootPanel.setBorder(BorderFactory.createCompoundBorder(
                BorderFactory.createEtchedBorder(EtchedBorder.LOWERED), BorderFactory.createEmptyBorder(Const.VSPACE, Const.HSPACE, Const.VSPACE, Const.HSPACE)));

        // Name and agents fields:
        
        Box box = Box.createHorizontalBox();
        rootPanel.add(box);
        box.add(new JLabel("Name: "));
        nameField = new JTextField(20);
        box.add(nameField);
        nameField.setToolTipText("<html>Filter name will be used for panel titles<br> unless otherwise specified.");
        box.add(Box.createHorizontalGlue());

        rootPanel.add(Box.createRigidArea(Const.VDIM));

        box = Box.createHorizontalBox();
        rootPanel.add(box);
        box.add(new JLabel("Subsystems: "));
        agentsField = new JTextField(20);
        box.add(agentsField);
        agentsField.setToolTipText("<html>Subsystems to be considered when templates<br> do not explicitly specify a subsystem name.");
        agentsField.addActionListener(e -> updateAvailable());
        agentsField.addFocusListener(new FocusListener() {
            @Override
            public void focusGained(FocusEvent e) {}
            @Override
            public void focusLost(FocusEvent e) {updateAvailable();}
            
        });
        box.add(Box.createHorizontalGlue());

        rootPanel.add(Box.createRigidArea(Const.VDIM));
        
        // Available templates panel:
        
        Box availablePanel = Box.createVerticalBox();
        rootPanel.add(availablePanel);
        availablePanel.setBorder(BorderFactory.createTitledBorder("Available templates"));

        box = Box.createHorizontalBox();
        availablePanel.add(box);
        box.add(new JLabel("Subsystem: "));
        subsystemCombo = new JComboBox<>(agents);
        box.add(subsystemCombo);
        subsystemCombo.addActionListener(e -> {
            updateAvailable();
        });
        box.add(Box.createRigidArea(Const.HDIM));
        box.add(new JLabel("Granularity: "));
        granularityComboIn = new JComboBox<>(Segment.values());
        box.add(granularityComboIn);
        granularityComboIn.addActionListener(e -> {
            updateAvailable();
        });
        box.add(Box.createHorizontalGlue());

        availablePanel.add(Box.createRigidArea(Const.VDIM));

        templatesIn = new JList<>();
        availablePanel.add(new JScrollPane(templatesIn));
        templatesIn.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        templatesIn.setCellRenderer(new DefaultListCellRenderer() {
            @Override
            public Component getListCellRendererComponent(JList<?> list, Object value, int index, boolean isSelected, boolean cellHasFocus) {
                return super.getListCellRendererComponent(list, ((Template)value).code, index, isSelected, cellHasFocus);
            }
        });
        templatesIn.addListSelectionListener(e -> {
            addButton.setEnabled(templatesIn.getSelectedValue() != null && groupCombo.getSelectedItem() != null);
        });

        rootPanel.add(Box.createRigidArea(Const.VDIM));
        
        // Add and remove buttons:

        box = Box.createHorizontalBox();
        rootPanel.add(box);
        box.add(Box.createRigidArea(Const.HDIM));
        addButton = new JButton("  Add  ");
        addButton.setToolTipText("Add selected template to selected display path");
        box.add(addButton);
        addButton.addActionListener(e -> {
            Template template = templatesIn.getSelectedValue();
            if (template != null) {
                DefaultListModel<Template> out = (DefaultListModel) templatesOut.getModel();
                out.addElement(template);
                DefaultListModel<Template> in = (DefaultListModel) templatesIn.getModel();
                in.removeElement(template);
            }
        });
        addButton.setEnabled(false);
        box.add(Box.createRigidArea(Const.HDIM));
        removeButton = new JButton("Remove");
        removeButton.setToolTipText("Remove selected template from selected display path");
        box.add(removeButton);
        removeButton.addActionListener(e -> {
            Template template = templatesOut.getSelectedValue();
            if (template != null) {
                DefaultListModel<Template> out = (DefaultListModel) templatesOut.getModel();
                out.removeElement(template);
                updateAvailable();
            }
        });
        removeButton.setEnabled(false);
        box.add(Box.createRigidArea(Const.HDIM));
        box.add(Box.createHorizontalGlue());

        rootPanel.add(Box.createRigidArea(Const.VDIM));
        
        // Selected templates panel:

        Box selectedPanel = Box.createVerticalBox();
        rootPanel.add(selectedPanel);
        selectedPanel.setBorder(BorderFactory.createTitledBorder("Display channels"));

        box = Box.createHorizontalBox();
        selectedPanel.add(box);
        JLabel label = new JLabel("Channel: ");
        box.add(label);
        label.setToolTipText("<html>Display channel names.<br>Used as column names in summary tables.");
        box.add(groupCombo);
        groupCombo.setPrototypeDisplayValue("12345678901234567890");
        groupCombo.addActionListener(e -> {
            updateSelected();
        });
        box.add(Box.createRigidArea(Const.HDIM));
        box.add(Box.createHorizontalGlue());
        upGroupButton = new JButton(" Up ");
        box.add(upGroupButton);
        upGroupButton.setToolTipText("Move selected channel up in the list.");
        upGroupButton.addActionListener(e -> {
            int i = groupCombo.getSelectedIndex();
            if (i > 0) {
                Integer field = selectedFields.get(i);
                selectedFields.set(i, selectedFields.get(i-1));
                selectedFields.set(i-1, field);
                String s = groupCombo.getItemAt(i);
                groupCombo.removeItemAt(i);
                groupCombo.insertItemAt(s, i-1);
                groupCombo.setSelectedIndex(i-1);
            }
        });
        downGroupButton = new JButton("Down");
        box.add(downGroupButton);
        downGroupButton.setToolTipText("Move selected channel down in the list.");
        downGroupButton.addActionListener(e -> {
            int i = groupCombo.getSelectedIndex();
            if (i != -1 && i < groupCombo.getItemCount()-1) {
                Integer field = selectedFields.get(i);
                selectedFields.set(i, selectedFields.get(i+1));
                selectedFields.set(i+1, field);
                String s = groupCombo.getItemAt(i);
                groupCombo.removeItemAt(i);
                groupCombo.insertItemAt(s, i+1);
                groupCombo.setSelectedIndex(i+1);
            }
        });
        addGroupButton = new JButton("New...");
        box.add(addGroupButton);
        addGroupButton.setToolTipText("Add a new channel after the selected one.");
        addGroupButton.addActionListener(e -> {
            String s = JOptionPane.showInputDialog(addGroupButton, "Enter channel name");
            if (s != null) {
                templatesOutModels.put(s, new DefaultListModel<>());
                int i = groupCombo.getSelectedIndex() + 1;
                selectedFields.add(i, 0);
                groupCombo.insertItemAt(s, i);
                groupCombo.setSelectedIndex(i);
            }
        });
        deleteGroupButton = new JButton("Delete");
        box.add(deleteGroupButton);
        deleteGroupButton.setToolTipText("Delete selected channel.");
        deleteGroupButton.addActionListener(e -> {
            int i = groupCombo.getSelectedIndex();
            if (i != -1) {
                groupCombo.removeItemAt(i);
                selectedFields.remove(i);
            }
        });
        renameGroupButton = new JButton("Rename");
        box.add(renameGroupButton);
        renameGroupButton.setToolTipText("Rename selected channel.");
        renameGroupButton.addActionListener(e -> {
            String old = (String) groupCombo.getSelectedItem();
            if (old != null) {
                String s = JOptionPane.showInputDialog(renameGroupButton, "Enter channel name", old);
                if (s != null && !s.equals(old)) {
                    DefaultListModel<Template> model = templatesOutModels.remove(old);
                    templatesOutModels.put(s, model);
                    int i = groupCombo.getSelectedIndex();
                    groupCombo.removeItemAt(i);
                    groupCombo.insertItemAt(s, i);
                    groupCombo.setSelectedIndex(i);
                }
            }
        });
        box.add(fieldCombo);
        fieldCombo.setToolTipText("What to display for this channel");
        fieldCombo.addActionListener(e -> {
            int i = groupCombo.getSelectedIndex();
            if (i != -1) {
                selectedFields.set(i, fieldCombo.getSelectedIndex());
            }
        });
        box.add(Box.createRigidArea(Const.HDIM));

        selectedPanel.add(Box.createRigidArea(Const.VDIM));

        selectedPanel.add(new JScrollPane(templatesOut));
        templatesOut.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        templatesOut.addListSelectionListener(e -> {
            removeButton.setEnabled(templatesOut.getSelectedValue() != null);
        });
        
        // Save, OK, and cancel buttons:

        box = Box.createHorizontalBox();
        add(box, BorderLayout.SOUTH);
        
        box.add(Box.createRigidArea(Const.HDIM));

        JButton saveButton = new JButton("Save...");
        box.add(saveButton);
        saveButton.setToolTipText("Save this filter for future use");
        saveButton.addActionListener(e -> {
            writeDescriptor();
            Console.getConsole().getSingleton(PersistenceService.class).saveAs(descriptor, "Saving...", GroupingTemplateFilterDialog.this);
        });

        box.add(Box.createRigidArea(Const.HDIM));

        JButton otherButton = new JButton("Other...");
        box.add(otherButton);
        otherButton.setToolTipText("Create a different type of filter");
        otherButton.addActionListener(e -> {
            other = true;
            dispose();
        });

        box.add(Box.createRigidArea(Const.HDIM));
        box.add(Box.createHorizontalGlue());
        
        JButton cancelButton = new JButton("Cancel");
        box.add(cancelButton);
        cancelButton.addActionListener(e -> {
            descriptor = null;
            dispose();
        });
        
        box.add(Box.createRigidArea(Const.HDIM));
        
        JButton okButton = new JButton(" OK ");
        box.add(okButton);
        okButton.addActionListener(e -> {
            writeDescriptor();
            dispose();
        });
        
        box.add(Box.createRigidArea(Const.HDIM));
        
        // Initialize all widgets:

        readDescriptor();
    }

    static public GroupingTemplateFilter edit(GroupingTemplateFilter before, Component parentComponent) {
        if (parentComponent == null) parentComponent = Console.getConsole().getWindow();
        GroupingTemplateFilterDialog dialog = new GroupingTemplateFilterDialog(before == null ? null : before.save(), parentComponent);
        dialog.pack();
        dialog.setLocationRelativeTo(parentComponent);
        dialog.setVisible(true);
        if (dialog.other) return null;
        if (dialog.descriptor == null) throw new CancellationException();
        return new GroupingTemplateFilter(dialog.descriptor);
    }
    
// -- Local methods : ----------------------------------------------------------
    
    private void updateAvailable() {
        String agent = (String) subsystemCombo.getSelectedItem();
        Segment segment = (Segment) granularityComboIn.getSelectedItem();
        DefaultListModel<Template> inModel = new DefaultListModel<>();
        DefaultListModel outModel = (DefaultListModel) templatesOut.getModel();
        if (agent.equals(ANY)) {
            ArrayList<String> selectedAgents = getSelectedAgents();
            for (Template template : templates) {
                if (template.segment.equals(segment) && (selectedAgents == null || selectedAgents.contains(template.getAgent()))) {
                    template = template.stripAgent();
                    if (!outModel.contains(template)) {
                        inModel.addElement(template);
                    }
                }
            }
        } else {
            for (Template template : templates) {
                if (template.segment.equals(segment) && template.getAgent().equals(agent) && !outModel.contains(template)) {
                    inModel.addElement(template);
                }
            }
        }
        templatesIn.setModel(inModel);
    }
    
    private void updateSelected() {
        String group = (String) groupCombo.getSelectedItem();
        DefaultListModel<Template> model = templatesOutModels.get(group);
        templatesOut.setModel(model);
        int i = groupCombo.getSelectedIndex();
        upGroupButton.setEnabled(i>0);
        downGroupButton.setEnabled(i != -1 && i < groupCombo.getItemCount()-1);
        deleteGroupButton.setEnabled(i != -1);
        renameGroupButton.setEnabled(i != -1);
        if (i != -1) {
            fieldCombo.setSelectedIndex(selectedFields.get(i));
        }
        updateAvailable();
    }
    
    private void readDescriptor() {
        
        String name = descriptor.getName();
        if (name == null) {
            nameField.setText("");
        } else {
            nameField.setText(name);
        }
        
        String[] aa = descriptor.getAgents();
        if (aa == null) {
            agentsField.setText("");
        } else {
            agentsField.setText(TypeUtils.stringify(aa));
        }
        
        LinkedHashMap<String, ArrayList<String>> groups = descriptor.getGroups();
        templatesOutModels.clear();
        templatesOutModels.put(null, new DefaultListModel<>());
        DefaultComboBoxModel<String> comboModel = new DefaultComboBoxModel<>();
        if (!(groups == null || groups.isEmpty())) {
            groups.forEach((group, templates) -> {
                comboModel.addElement(group);
                DefaultListModel<Template> model = new DefaultListModel<>();
                if (templates != null) {
                    templates.forEach(t -> model.addElement(Template.fromString(t)));
                }
                templatesOutModels.put(group, model);
            });
            String[] ff = descriptor.getFields();
            if (ff == null || ff.length != groups.size()) {
                for (int i=groups.size(); --i>=0; ) {
                    selectedFields.add(0);
                }
            } else {
                for (String f : ff) {
                    int i=fields.length;
                    for (; --i>=0; ) {
                        MonitorField mf = fields[i];
                        if (mf.getTitle().equals(f)) {
                            selectedFields.add(i);
                            break;
                        }
                    }
                    if (i < 0 ) selectedFields.add(0);
                }
            }
        }
        groupCombo.setModel(comboModel);
        
        updateSelected();
    }
    
    private void writeDescriptor() {
        
        String name = nameField.getText();
        if (name.trim().isEmpty()) {
            descriptor.setName(null);
        } else {
            descriptor.setName(name);
        }
        
        ArrayList<String> aa = getSelectedAgents();
        if (aa == null) {
            descriptor.setAgents(null);
        } else {
            descriptor.setAgents(aa.toArray(new String[0]));
        }
        
        int nGroups = groupCombo.getItemCount();
        if (nGroups == 0) {
            descriptor.setGroups(null);
            descriptor.setFields(null);
        } else {
            LinkedHashMap<String, ArrayList<String>> groups = new LinkedHashMap<>();
            for (int i=0; i < nGroups; i++) {
                String group = groupCombo.getItemAt(i);
                DefaultListModel<Template> model = templatesOutModels.get(group);
                if (model != null) {
                    ArrayList<String> templates = new ArrayList<>(model.getSize());
                    Enumeration<Template> en = model.elements();
                    while (en.hasMoreElements()) {
                        templates.add(en.nextElement().toString());
                    }
                    groups.put(group, templates);
                }
            }
            descriptor.setGroups(groups);
            boolean saveFields = false;
            for (int f : selectedFields) {
                if (f != 0) {
                    saveFields = true;
                    break;
                }
            }
            if (saveFields) {
                String[] ff = new String[selectedFields.size()];
                for (int i = ff.length; --i>=0; ) {
                    ff[i] = fields[selectedFields.get(i)].getTitle();
                }
                descriptor.setFields(ff);
            } else {
                descriptor.setFields(null);
            }
        }   
    }
    
    private ArrayList<String> getSelectedAgents() {
        String text = agentsField.getText().trim();
        if (text.isEmpty() || text.equals(TypeUtils.NULL_STR)) {
            return null;
        } else {
            if (text.startsWith("[") && text.endsWith("]")) {
                text = text.substring(1, text.length()-1);
            }
            String[] ss = text.split("[ ,]+");
            ArrayList<String> out = new ArrayList<>(ss.length);
            for (String s : ss) {
                out.add(s.trim());
            }
            return out;
        }
    }
        
}
