package org.lsst.ccs.subsystem.common.ui.focalplane;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import java.util.concurrent.CancellationException;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.gconsole.annotations.Plugin;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.ConsolePlugin;
import org.lsst.ccs.gconsole.base.filter.PersistableAgentChannelsFilter;
import org.lsst.ccs.gconsole.services.optpage.OptionalPage;
import org.lsst.ccs.gconsole.plugins.monitor.LsstMonitorPlugin;
import org.lsst.ccs.gconsole.plugins.monitor.MonitorPage;
import org.lsst.ccs.gconsole.plugins.monitor.PersistableMonitorView;
import org.lsst.ccs.gconsole.plugins.monitor.SectionedTableView;
import org.lsst.ccs.gconsole.services.persist.Creator;
import org.lsst.ccs.gconsole.services.persist.Persistable;
import org.lsst.ccs.gconsole.services.persist.PersistenceService;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.subsystem.common.focalplane.data.HasFocalPlaneData;
import org.lsst.ccs.subsystem.common.ui.focalplane.filter.DataGroupFilter;
import org.lsst.ccs.subsystem.common.ui.focalplane.filter.FocalPlaneFilter;
import org.lsst.ccs.subsystem.common.ui.focalplane.filter.GroupAttributeFilter;
import org.lsst.ccs.subsystem.common.ui.focalplane.view.DiagramView;
import org.lsst.ccs.subsystem.common.ui.focalplane.view.FocalPlaneView;
import org.lsst.ccs.subsystem.common.ui.focalplane.view.MultiMapView;
import org.lsst.ccs.subsystem.common.ui.focalplane.view.SummaryView;

/**
 * Focal plane GUI plugin.
 *
 * @author onoprien
 */
@Plugin(name="LSST CCS Commons Focal Plane",
        id="commons-focal-plane",
        description="LSST CCS support for focal plane views")
public class FocalPlanePlugin extends ConsolePlugin {

// -- Fields : -----------------------------------------------------------------
    
    static public final String MONITOR_PAGE_CATEGORY = "FocalPlaneMonitorPage";
    
    private AgentPresenceListener agentPresenceListener;

// -- Life cycle : -------------------------------------------------------------
    
    public FocalPlanePlugin() {
    }
    
    @Override
    public void initialize() {
        // Pre-defined monitoring page (selectable channel group) :
        OptionalPage.getService().add(new FocalPlaneMonitoringOptionalPage());
    }

    @Override
    public void start() {
        
        // Initialize machinery for focal plane maps
        
        initFocalPlaneMaps();
        
        // Menu for opening focal plane monitor pages
        
        LsstMonitorPlugin monitorPlugin = getConsole().getSingleton(LsstMonitorPlugin.class);
        if (monitorPlugin != null) {
            Action act = new AbstractAction("New...") {
                @Override
                public void actionPerformed(ActionEvent e) {
                    openNewMonitorPage(monitorPlugin);
                }
            };
            monitorPlugin.getServices().addMenu(act, "400: CCS Tools :-1:15", LsstMonitorPlugin.MENU_NAME +":2:1", "Focal Plane:1");
            act = new AbstractAction("Load...") {
                @Override
                public void actionPerformed(ActionEvent e) {
                PersistenceService service = getConsole().getSingleton(PersistenceService.class);
                try {
                    Persistable.Descriptor desc = service.load(MONITOR_PAGE_CATEGORY, "Load monitoring page", getConsole().getWindow());
                    if (desc instanceof MonitorPage.Descriptor) {
                        MonitorPage page = (MonitorPage) service.make(desc);
                        monitorPlugin.openManagedPage(page);
                    }
                } catch (RuntimeException x) {
                }
                }
            };
            monitorPlugin.getServices().addMenu(act, "400: CCS Tools :-1:15", LsstMonitorPlugin.MENU_NAME +":2:1", "Focal Plane:2");
        }
        
    }

    @Override
    public void stop() {
        // Shut down machinery for focal plane maps
        getConsole().getMessagingAccess().getAgentPresenceManager().removeAgentPresenceListener(agentPresenceListener);
        agentPresenceListener = null;
    }
    
    
// -- Focal plane specific monitor views : -------------------------------------
    
    private void openNewMonitorPage(LsstMonitorPlugin monitorPlugin) {
        try {
            MonitorPage monPage = new MonitorPage();
            monPage.getDescriptor().setCategory(MONITOR_PAGE_CATEGORY);
            monPage.getDescriptor().setName("FP Mon");
            PersistableMonitorView.Descriptor viewDesc = new PersistableMonitorView.Descriptor();
            Creator.Descriptor creDesc = new Creator.Descriptor(FocalPlaneView.CATEGORY, SummaryView.PATH);
            viewDesc.setCreator(creDesc);
            monPage.getDescriptor().setView(viewDesc);
            PersistableAgentChannelsFilter.Descriptor filterDesc = new PersistableAgentChannelsFilter.Descriptor();
            filterDesc.setCategory(FocalPlaneFilter.CATEGORY);
            monPage.getDescriptor().setFilter(filterDesc);
            monPage = monPage.edit("Open Focal Plane Monitor Page", null);
            monitorPlugin.openManagedPage(monPage);
        } catch (CancellationException x) {
        } catch (RuntimeException x) {
            getConsole().error("Error opening monitor page", x);
        }
    }
    
    
// -- Handling focal plane maps : ----------------------------------------------
    
    private void initFocalPlaneMaps() {
        agentPresenceListener = new AgentPresenceListener() {
            @Override
            public void connected(AgentInfo... agents) { // FIXME: should this be done listening to AgentPresenceManager
//                if (!agent.getAgentProperty(HasFocalPlaneData.AGENT_PROPERTY, "").trim().isEmpty()) {
                SwingUtilities.invokeLater(() -> {
                    for (AgentInfo agent : agents) {
                        if (!agent.getAgentProperty(HasFocalPlaneData.AGENT_PROPERTY, "").trim().isEmpty()) {
                            String agentName = agent.getName();
                            Action act = new AbstractAction("Custom...") {
                                @Override
                                public void actionPerformed(ActionEvent e) {
                                    FocalPlaneMapPageDialog d = FocalPlaneMapPageDialog.show(null, agentName + " map");
                                    if (d != null) {
                                        openFocalPlaneMapPage(agentName, d.getRows(), d.getColumns(), d.getPageTitle());
                                    }
                                }
                            };
                            getServices().addMenu(act, "CCS Subsystems", agentName, "Focal plane", "Map");
                            addFocalPlaneMapMenu(agentName, 1, 1);
                            addFocalPlaneMapMenu(agentName, 1, 2);
                            addFocalPlaneMapMenu(agentName, 2, 1);
                            addFocalPlaneMapMenu(agentName, 2, 2);
                            addFocalPlaneMapMenu(agentName, 1, 3);
                            addFocalPlaneMapMenu(agentName, 2, 3);
                            addFocalPlaneMapMenu(agentName, 3, 3);
                            addFocalPlaneMapMenu(agentName, 4, 4);
                            act = new AbstractAction("Diagram") {
                                @Override
                                public void actionPerformed(ActionEvent e) {
                                    openFocalPlaneDiagramPage(agentName);
                                }
                            };
                            getServices().addMenu(act, "CCS Subsystems", agentName, "Focal plane");
                            act = new AbstractAction("Summary") {
                                @Override
                                public void actionPerformed(ActionEvent e) {
                                    openFocalPlaneSummaryPage(agentName);
                                }
                            };
                            getServices().addMenu(act, "CCS Subsystems", agentName, "Focal plane");
                        }
                    }
                });
            }
        };
        getConsole().getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(agentPresenceListener);
    }
        
    private void addFocalPlaneMapMenu(String agent, int rows, int columns) {
        Action act = new AbstractAction(columns +" x "+ rows) {
            @Override
            public void actionPerformed(ActionEvent e) {
                openFocalPlaneMapPage(agent, rows, columns, null);
            }
        };
        getServices().addMenu(act, "CCS Subsystems", agent, "Focal plane", "Map");
    }
    
    private void openFocalPlaneMapPage(String agent, int rows, int columns, String title) {
        if (title == null || title.trim().isEmpty()) title = agent +" map";
        MonitorPage.Descriptor pageDesc = new MonitorPage.Descriptor();
        pageDesc.setCategory(MONITOR_PAGE_CATEGORY);
        pageDesc.setName(title);
        MultiMapView.Descriptor viewDesc = new MultiMapView.Descriptor();
        Creator.Descriptor cd = new Creator.Descriptor();
        cd.setParameters(new String[] {Integer.toString(rows), Integer.toString(columns)});
        cd.setPath("Built-In/Map/Multi-Region");
        viewDesc.setCreator(cd);
        viewDesc.setRows(rows);
        viewDesc.setColumns(columns);
        pageDesc.setView(viewDesc);
        GroupAttributeFilter.Descriptor filterDesc = new GroupAttributeFilter.Descriptor();
        cd = new Creator.Descriptor();
        cd.setPath("Built-In/Standard Groups/One Subsystem");
        cd.setParameters(new String[] {agent});
        filterDesc.setCreator(cd);
        filterDesc.setAgent(agent);
        pageDesc.setFilter(filterDesc);
        LsstMonitorPlugin monPlugin = Console.getConsole().getSingleton(LsstMonitorPlugin.class);
        monPlugin.openManagedPage(pageDesc);
    }
    
    private void openFocalPlaneDiagramPage(String agent) {
        MonitorPage.Descriptor pageDesc = new MonitorPage.Descriptor();
        pageDesc.setCategory(MONITOR_PAGE_CATEGORY);
        pageDesc.setName(agent +" diagram");
        DiagramView.Descriptor viewDesc = new DiagramView.Descriptor();
        Creator.Descriptor cd = new Creator.Descriptor();
        cd.setParameters(new String[] {});
        cd.setPath("Built-In/Diagram");
        viewDesc.setCreator(cd);
        viewDesc.setName("Focal Plane Diagram");
        pageDesc.setView(viewDesc);
        GroupAttributeFilter.Descriptor filterDesc = new GroupAttributeFilter.Descriptor();
        cd = new Creator.Descriptor();
        cd.setPath("Built-In/Standard Groups/One Subsystem");
        cd.setParameters(new String[] {agent});
        filterDesc.setCreator(cd);
        filterDesc.setName("Standard data groups");
        filterDesc.setAgent(agent);
        pageDesc.setFilter(filterDesc);
        LsstMonitorPlugin monPlugin = Console.getConsole().getSingleton(LsstMonitorPlugin.class);
        monPlugin.openManagedPage(pageDesc);
    }
    
    private void openFocalPlaneSummaryPage(String agent) {
        PersistenceService service = Console.getConsole().getSingleton(PersistenceService.class);
        SummaryView view = (SummaryView) service.make(FocalPlaneView.CATEGORY, SummaryView.PATH);
        GroupAttributeFilter filter = (GroupAttributeFilter) service.make(FocalPlaneFilter.CATEGORY, "Built-In/Standard Groups/One Subsystem", agent);
        MonitorPage page = new MonitorPage(view, filter, agent +" summary");
        page.getDescriptor().setCategory(MONITOR_PAGE_CATEGORY);
        LsstMonitorPlugin monPlugin = Console.getConsole().getSingleton(LsstMonitorPlugin.class);
        monPlugin.openManagedPage(page);
    }
    
    
// -- Standard monitoring page : -----------------------------------------------

    static class FocalPlaneMonitoringOptionalPage implements OptionalPage {

        @Override
        public boolean isAutoOpen() {
            return false;
        }

        @Override
        public String getPage(AgentInfo agent) {
            if (agent != null && agent.hasAgentProperty(HasFocalPlaneData.AGENT_PROPERTY)) {
                return "CCS Subsystems/" + agent.getName() +"/Focal plane/Table";
            }
            return null;
        }

        @Override
        public JComponent open(AgentInfo agent, JComponent existingComponent) {
            DataGroupMonitor dataSelection = new DataGroupMonitor(agent);
            return dataSelection;

        }
    }
    
    static private final class DataGroupMonitor extends JPanel {
        
        private final JComboBox<String> dataSelector;
        private final AgentInfo agent;
        
        SectionedTableView view = new SectionedTableView();
        
        DataGroupMonitor(AgentInfo agent) {
            this.agent = agent;
            setLayout(new BorderLayout());
            setName(agent.getName());
            String prop = agent.getAgentProperty(HasFocalPlaneData.AGENT_PROPERTY);
            String[] ss = HasFocalPlaneData.getFocalPlaneDataGroups(prop);
            dataSelector = new JComboBox(ss);
            add(dataSelector,BorderLayout.PAGE_START);
            makeFilteredTable();
            dataSelector.addActionListener(e -> {
                remove(view.getPanel());
                view = new SectionedTableView();
                makeFilteredTable();
            });
        }
        
        final void makeFilteredTable() {
            DataGroupFilter monitorFilter = new DataGroupFilter(agent.getName(), (String) dataSelector.getSelectedItem());
            view.uninstall();
            view.setFilter(monitorFilter);
            view.install();
            add(view.getPanel(),BorderLayout.CENTER);
            revalidate();
            repaint();
        }
        
    }
        
}
