package org.lsst.ccs.subsystem.common.ui;

import java.awt.Color;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.lang.reflect.InvocationTargetException;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JTextField;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.border.Border;
import javax.swing.border.LineBorder;
import javax.swing.border.TitledBorder;

/**
 * Some useful UI utility routines
 *
 * @author saxton
 */
public class UiUtilities {

    public interface ActionHandler {

        public default void handleButton(String name) {}

        public default void handleRadioButton(String name) {}

        public default void handleTextField(String name, String text) {}

        public default void handleTextFieldX(String name, Object value) {}

        public default void handleCheckBox(String name, boolean isSelected) {}

    }

    private final ActionHandler handler;

    /**
     * Constructor
     * 
     * @param handler The action handler
     */
    public UiUtilities(ActionHandler handler) {
        this.handler = handler;
    }

    /**
     * Creates a label with the default font.
     * 
     * @param text The label text
     * @param width The desired label width, 0 if text determines it, or -1 if no width set
     * @return The label
     */
    public static JLabel newLabel(String text, int width) {
        return newLabel(text, width, UiConstants.FONT);
    }

    /**
     * Creates a label
     * 
     * @param text The label text
     * @param width The desired label width, 0 if text determines it, or -1 if no width set
     * @param font The font to use
     * @return The label
     */
    public static JLabel newLabel(String text, int width, Font font) {
        JLabel label = new JLabel(text);
        label.setFont(font);
        Dimension d = label.getPreferredSize();
        if (width > 0) {
            d.width = width;
        }
        if (width >= 0) {
            label.setPreferredSize(d);
            label.setMinimumSize(d);
        }
        return label;
    }

    /**
     * Creates a radio button with the default font
     * 
     * @param title The button title
     * @param name The name to be passed to the handler
     * @return The radio button
     */
    public JRadioButton newRadioButton(String title, String name) {
        return newRadioButton(title, name, UiConstants.FONT);
    }

    /**
     * Creates a radio button.
     * 
     * @param title The button title
     * @param name The name to be passed to the action handler
     * @param font The font to use
     * @return The radio button
     */
    public JRadioButton newRadioButton(String title, String name, Font font) {
        return newRadioButton(title, name, font, false);
    }

    /**
     * Creates a radio button.
     * 
     * @param title The button title
     * @param name The name to be passed to the action handler
     * @param confirm True/false if there should be a Confirmation Dialog box.
     * @return The radio button
     */
    public JRadioButton newRadioButton(String title, String name, boolean confirm) {
        return newRadioButton(title, name, UiConstants.FONT, confirm);
    }

    /**
     * Creates a radio button.
     * 
     * @param title The button title
     * @param name The name to be passed to the action handler
     * @param font The font to use
     * @param confirm True/false if there should be a Confirmation Dialog box.
     * @return The radio button
     */
    public JRadioButton newRadioButton(String title, String name, Font font, boolean confirm) {
        JRadioButton rb = new JRadioButton(title);
        rb.setFont(font);
        Dimension d = rb.getPreferredSize();
        d.height = getLabelSize("X", font).height;
        rb.setPreferredSize(d);
        rb.setMinimumSize(d);
        rb.setFocusable(false);
        rb.setName(name);
        rb.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                int result = JOptionPane.YES_OPTION;
                if (confirm) {
                    result = JOptionPane.showConfirmDialog(rb, "Proceed with operation: "+title+"?", "Confirm Operation", JOptionPane.YES_NO_OPTION);
                }
                if ( result == JOptionPane.NO_OPTION ) {
                    //Since RadioButton can belong to ButtonGroup object, we disable
                    //the JRadioButton and the the handler decide what to do.
                    ((JRadioButton)evt.getSource()).getModel().setEnabled(false);                    
                }
                handler.handleRadioButton(((JRadioButton) evt.getSource()).getName());
            }
        });
        return rb;
    }
    
    
    /**
     * Creates a button with default font.
     * 
     * @param title The button title
     * @param name The name to be passed to the action handler
     * @return The button
     */
    public JButton newButton(String title, String name) {
        return newButton(title, name, false, null, UiConstants.FONT);
    }

    /**
     * Creates a button with default font.
     * 
     * @param title The button title
     * @param name The name to be passed to the action handler
     * @param small Create a less-tall button if true
     * @return The button
     */
    public JButton newButton(String title, String name, boolean small) {
        return newButton(title, name, small, null, UiConstants.FONT);
    }

    /**
     * Creates a button.
     * 
     * @param title The button title
     * @param name The name to be passed to the action handler
     * @param font The font to use
     * @return The button
     */
    public JButton newButton(String title, String name, Font font) {
        return newButton(title, name, false, null, font);
    }

    /**
     * Creates a button.
     * 
     * @param title The button title
     * @param name The name to be passed to the action handler
     * @param small Create a less-tall button if true
     * @param font The font to use
     * @return The button
     */
    public JButton newButton(String title, String name, boolean small, Font font) {
        return newButton(title, name, small, null, font);
    }

    /**
     * Creates a button.
     * 
     * @param title The button title
     * @param name The name to be passed to the action handler
     * @param small Create a less-tall button if true
     * @param lblFont Font to use to determine less-tall height
     * @param font The font to use
     * @return The button
     */
    public JButton newButton(String title, String name, boolean small, Font lblFont, Font font) {
        JButton btn = new JButton(title);
        btn.setFont(font);
        Dimension d = btn.getPreferredSize();
        if (small) {
            d.height = getLabelSize("X", lblFont == null ? font : lblFont).height;
        }
        btn.setPreferredSize(d);
        btn.setMinimumSize(d);
        btn.setFocusable(false);
        btn.setName(name);
        btn.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                handler.handleButton(((JButton)evt.getSource()).getName());
            }
        });
        return btn;
    }

    /**
     * Creates a text field.
     * 
     * @param text The initial text, used to set its size
     * @param name The name to be passed to the action handler
     * @return The text field
    */
    public JTextField newTextField(String text, String name) {
        return newTextField(text, name, false, null, null);
    }

    /**
     * Creates a text field.
     * 
     * @param text The initial text, used to set its size
     * @param name The name to be passed to the action handler
     * @param font The font to use
     * @return The text field
     */
    public JTextField newTextField(String text, String name, Font font) {
        return newTextField(text, name, false, null, font);
    }

    /**
     * Creates a text field.
     * 
     * @param text The initial text, used to set its size
     * @param name The name to be passed to the action handler
     * @param small Create a less-tall field if true
     * @return The text field
     */
    public JTextField newTextField(String text, String name, boolean small) {
        return newTextField(text, name, small, null, null);
    }

    /**
     * Creates a text field.
     * 
     * @param text The initial text, used to set its size
     * @param name The name to be passed to the action handler
     * @param small Create a less-tall field if true
     * @param lblFont Font to use to determine less-tall height
     * @return The text field
     */
    public JTextField newTextField(String text, String name, boolean small, Font lblFont) {
        return newTextField(text, name, small, lblFont, null);
    }

    /**
     * Creates a text field.
     * 
     * @param text The initial text, used to set its size
     * @param name The name to be passed to the action handler
     * @param small Create a less-tall field if true
     * @param lblFont Font to use to determine less-tall height
     * @param font The font to use
     * @return The text field
     */
    public JTextField newTextField(String text, String name, boolean small, Font lblFont, Font font) {
        JTextField tf = new JTextField(text);
        if (font != null) {
            tf.setFont(font);
        }
        Dimension d = tf.getPreferredSize();
        if (small) {
            d.height = Math.min(d.height, (int)(1.5 * getLabelSize("X", lblFont == null ? UiConstants.FONT : lblFont).height));
        }
        tf.setPreferredSize(d);
        tf.setMinimumSize(d);
        tf.setHorizontalAlignment(SwingConstants.CENTER);
        tf.setName(name);
        tf.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                JTextField tf = (JTextField)evt.getSource();
                handler.handleTextField(tf.getName(), tf.getText());
            }
        });
        return tf;
    }

    /**
     * Creates an extended text field.
     * 
     * @param text The initial text, used to set its size
     * @param name The name to be passed to the action handler
     * @param type The type of field (TYPE_TEXT, TYPE_INT or TYPE_DOUBLE)
     * @return The text field
     */
    public TextFieldX newTextFieldX(String text, String name, int type) {
        return newTextFieldX(text, name, type, false, null, null);
    }

    /**
     * Creates an extended text field.
     * 
     * @param text The initial text, used to set its size
     * @param name The name to be passed to the action handler
     * @param type The type of field (TYPE_TEXT, TYPE_INT or TYPE_DOUBLE)
     * @param font The font to use
     * @return The text field
     */
    public TextFieldX newTextFieldX(String text, String name, int type, Font font) {
        return newTextFieldX(text, name, type, false, null, font);
    }

    /**
     * Creates an extended text field.
     * 
     * @param text The initial text, used to set its size
     * @param name The name to be passed to the action handler
     * @param type The type of field (TYPE_TEXT, TYPE_INT or TYPE_DOUBLE)
     * @param small Create a less-tall field if true
     * @return The text field
     */
    public TextFieldX newTextFieldX(String text, String name, int type, boolean small) {
        return newTextFieldX(text, name, type, small, null, null);
    }

    /**
     * Creates an extended text field.
     * 
     * @param text The initial text, used to set its size
     * @param name The name to be passed to the action handler
     * @param type The type of field (TYPE_TEXT, TYPE_INT or TYPE_DOUBLE)
     * @param small Create a less-tall field if true
     * @param lblFont Font to use to determine less-tall height
     * @return The text field
     */
    public TextFieldX newTextFieldX(String text, String name, int type, boolean small, Font lblFont) {
        return newTextFieldX(text, name, type, small, lblFont, null);
    }

    /**
     * Creates an extended text field.
     * 
     * @param text The initial text, used to set its size
     * @param name The name to be passed to the action handler
     * @param type The type of field (TYPE_TEXT, TYPE_INT or TYPE_DOUBLE)
     * @param small Create a less-tall field if true
     * @param lblFont Font to use to determine less-tall height
     * @param font The font to use
     * @return The text field
     */
    public TextFieldX newTextFieldX(String text, String name, int type, boolean small, Font lblFont, Font font) {
        TextFieldX tf = new TextFieldX(text, type);
        if (font != null) {
            tf.setFont(font);
        }
        Dimension d = tf.getPreferredSize();
        if (small) {
            d.height = Math.min(d.height, (int)(2.0 * getLabelSize("X", lblFont == null ? UiConstants.FONT : lblFont).height));
        }
        tf.setPreferredSize(d);
        tf.setMinimumSize(d);
        tf.setHorizontalAlignment(SwingConstants.CENTER);
        tf.setName(name);
        tf.addFocusListener(new FocusAdapter() {
            @Override
            public void focusLost(FocusEvent evt) {
                ((TextFieldX)evt.getSource()).focusLost();
            }
        });
        tf.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                TextFieldX tf = (TextFieldX)evt.getSource();
                Object value = tf.action();
                if (value != null) {
                    handler.handleTextFieldX(tf.getName(), value);
                }
            }
        });
        tf.addKeyListener(new KeyAdapter() {
            @Override
            public void keyTyped(KeyEvent evt) {
                ((TextFieldX)evt.getSource()).keyTyped(evt.getKeyChar());
            }
        });
        return tf;
    }

    /**
     * Creates a check box.
     * 
     * @param title The check box title
     * @param name The name to be passed to the action handler
     * @return The check box
     */
    public JCheckBox newCheckBox(String title, String name) {
        return newCheckBox(title, name, UiConstants.FONT);
    }

    /**
     * Creates a check box.
     * 
     * @param title The check box title
     * @param name The name to be passed to the action handler
     * @param font The font to use
     * @return The check box
     */
    public JCheckBox newCheckBox(String title, String name, Font font) {
        JCheckBox cbx = new JCheckBox(title);
        cbx.setFont(font);
        cbx.setFocusable(false);
        cbx.setName(name);
        cbx.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                JCheckBox cbx = (JCheckBox)evt.getSource();
                handler.handleCheckBox(cbx.getName(), cbx.isSelected());
            }
        });
        return cbx;
    }

    /**
     * Creates a bordered panel.
     * 
     * @param title The panel title, or null if no title required
     * @return The panel
     */
    public static JPanel newBorderedPanel(String title) {
        return newBorderedPanel(title, UiConstants.FONT);
    }

    /**
     * Creates a bordered panel.
     * 
     * @param title The panel title, or null if no title required
     * @param font The font to use
     * @return The panel
     */
    public static JPanel newBorderedPanel(String title, Font font) {
        JPanel panel = new JPanel();
        Border border = new LineBorder(Color.BLACK);
        if (title != null) {
            TitledBorder tBorder = new TitledBorder(border, title);
            tBorder.setTitleJustification(TitledBorder.CENTER);
            tBorder.setTitleFont(font);
            tBorder.setTitleColor(UiConstants.BLUE);
            border = tBorder;
        }
        panel.setBorder(border);
        panel.setLayout(new GridBagLayout());
        return panel;
    }

    /**
     * Gets the maximum label width produced by a set of strings.
     * 
     * @param text An array of strings
     * @param term The terminating string that will be appended to string 
     * @return The maximum width 
     */
    public static int maxLabelWidth(String[] text, String term) {
        return maxLabelWidth(text, term, UiConstants.FONT);
    }

    /**
     * Gets the maximum label width produced by a set of strings.
     * 
     * @param text An array of strings
     * @param term The terminating string that will be appended to string 
     * @param font The font to use
     * @return The maximum width 
     */
    public static int maxLabelWidth(String[] text, String term, Font font) {
        JLabel label = new JLabel();
        label.setFont(font);
        int width = 0;
        for (String item : text) {
            label.setText(item + term);
            width = Math.max(width, label.getPreferredSize().width);
        }
        return width;
    }

    /**
     * Gets the maximum label width produced by the elements of an enumeration.
     * 
     * @param enumClass The enumeration class
     * @return The maximum width 
     */
    public static int maxEnumLabelWidth(Class enumClass) {
        return maxEnumLabelWidth(enumClass, UiConstants.FONT);
    }

    /**
     * Gets the maximum label width produced by the elements of an enumeration.
     * 
     * @param enumClass The enumeration class
     * @param font The font to use
     * @return The maximum width 
     */
    public static int maxEnumLabelWidth(Class enumClass, Font font) {
        JLabel label = new JLabel();
        label.setFont(font);
        int width = 0;
        for (Object en : enumClass.getEnumConstants()) {
            label.setText(en.toString());
            width = Math.max(width, label.getPreferredSize().width);
        }
        return width;
    }

    /**
     * Gets the size of a label with standard font.
     * 
     * @param text The label text
     * @return The dimension of the label
     */
    public static Dimension getLabelSize(String text) {
        return getLabelSize(text, UiConstants.FONT);
    }

    /**
     * Gets the size of a label.
     * 
     * @param text The label text
     * @param font The font to use
     * @return The dimension of the label
     */
    public static Dimension getLabelSize(String text, Font font) {
        JLabel label = new JLabel(text);
        label.setFont(font);
        return label.getPreferredSize();
    }

}
