package org.lsst.ccs.subsystem.common.ui.jas;

import java.awt.BorderLayout;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import javax.swing.JComponent;
import javax.swing.JPanel;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.services.optpage.OptionalPage;
import org.lsst.ccs.gconsole.plugins.monitor.SectionedTableView;
import org.lsst.ccs.gconsole.plugins.monitor.TabbedView;
import org.lsst.ccs.messaging.AgentMessagingLayer;

/**
 * @author tonyj
 */
public class MonOptionalPage implements OptionalPage {

    private final Map<String, ControlPanel> controls = new ConcurrentHashMap<>();
    private final String agentProperty;
    private final Class controlClass;
    private final boolean noMonitor;
    private final String section;
    private final boolean usePath;
    private final String[] exclSections;

    public MonOptionalPage(String agentProperty, Class controlClass, String section,
                           boolean usePath, String... exclSections) {
        this(agentProperty, controlClass, section, false, usePath, exclSections);
    }

    public MonOptionalPage(String agentProperty, Class controlClass, String section,
                           boolean noMonitor, boolean usePath, String... exclSections) {
        this.agentProperty = agentProperty;
        this.controlClass = controlClass;
        this.noMonitor = noMonitor;
        this.section = section;
        this.usePath = usePath;
        this.exclSections = exclSections;
    }

    @Override
    public boolean isAutoOpen() {
        return false;
    }

    @Override
    public String getPage(AgentInfo agent) {
        //System.out.println("getPage called: agent = " + agent.getName());
        return agent != null && agent.hasAgentProperty(agentProperty) ? getPageName(agent) : null;
    }

    @Override
    public boolean agentDisconnected(AgentInfo agent) {
        //System.out.println("agentDisconnected called: agent = " + agent.getName());
        ControlPanel control = controls.get(getPageName(agent));
        if (control == null) return true;
        control.disableControlPanel();
        return false;
    }

    @Override
    public JComponent open(AgentInfo agent, JComponent existingComponent) {
        //System.out.println("open called: agent = " + agent.getName());
        String pageName = getPageName(agent);
        String agentName = agent.getName();
        ControlPanel control = controls.get(pageName);
        if (control != null) {
            control.initControlPanel();
            return existingComponent;
        }

        // Create the control object
        try {
            control = controlClass == null ? new NoControl() : (ControlPanel)controlClass.newInstance();
            controls.put(pageName, control);
            //System.out.println("Control object created");
        }
        catch (Exception e) {
            //System.out.println("Control object creation failed: " + e);
            return null;
        }

        // Create the control panel
        JPanel cPanel = control.getControlPanel(agentName);

        // Create the monitoring panel, if requested, and fill top panel
        JPanel panel;
        if (!noMonitor) {
            TabbedView view = new TabbedView();
            view.setViewFactory(name -> new SectionedTableView()); 
            if (usePath) {
                view.setFilter(new PathMonitorFilter(Collections.singletonList(agentName), exclSections));
            }
            else {
                view.setFilter(new MonitorFilter(Collections.singletonList(agentName)));
            }
            view.install();
            panel = new JPanel();
            panel.setLayout(new java.awt.BorderLayout());
            panel.add(view.getPanel(), BorderLayout.CENTER);
            panel.add(cPanel, BorderLayout.SOUTH);
        }
        else {
            panel = cPanel;
        }
        panel.setName(agentName + (section == null ? "" : " " + section));

        // Set up the bus listener
        AgentMessagingLayer aml = Console.getConsole().getMessagingAccess();
        aml.addStatusMessageListener(control.getListener(),
                                     (msg) -> msg.getOriginAgentInfo().getName().equals(agentName)
                                                && msg instanceof StatusSubsystemData);

        // Initialize the control panel
        control.initControlPanel();
        return panel;
    }

    @Override
    public void closed(String pageName) {
        //System.out.println("closed called: page = " + pageName);
        ControlPanel control = controls.remove(pageName);
        if (control == null) {
            throw new RuntimeException("No page exists with the name " + pageName);
        }
        Console.getConsole().getMessagingAccess().removeStatusMessageListener(control.getListener());
    }

    private String getPageName(AgentInfo agent) {
        return "CCS Subsystems/" + agent.getName() + "/" + (section == null ? agent.getName() : section) ;
    }

}
